/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.expressions;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.Locale;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.iceberg.Schema;
import org.apache.iceberg.SingleValueParser;
import org.apache.iceberg.expressions.BoundPredicate;
import org.apache.iceberg.expressions.BoundTerm;
import org.apache.iceberg.expressions.BoundTransform;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.ExpressionVisitors;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.expressions.Literal;
import org.apache.iceberg.expressions.Reference;
import org.apache.iceberg.expressions.Term;
import org.apache.iceberg.expressions.UnboundPredicate;
import org.apache.iceberg.expressions.UnboundTerm;
import org.apache.iceberg.expressions.UnboundTransform;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.collect.Iterables;
import org.apache.iceberg.shaded.com.fasterxml.jackson.core.JsonGenerator;
import org.apache.iceberg.shaded.com.fasterxml.jackson.databind.JsonNode;
import org.apache.iceberg.shaded.com.fasterxml.jackson.databind.node.ArrayNode;
import org.apache.iceberg.transforms.Transforms;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.JsonUtil;

public class ExpressionParser {
    private static final String TYPE = "type";
    private static final String VALUE = "value";
    private static final String VALUES = "values";
    private static final String TRANSFORM = "transform";
    private static final String TERM = "term";
    private static final String LEFT = "left";
    private static final String RIGHT = "right";
    private static final String CHILD = "child";
    private static final String REFERENCE = "reference";
    private static final String LITERAL = "literal";

    private ExpressionParser() {
    }

    public static String toJson(Expression expression) {
        return ExpressionParser.toJson(expression, false);
    }

    public static String toJson(Expression expression, boolean pretty) {
        Preconditions.checkArgument(expression != null, "Invalid expression: null");
        return JsonUtil.generate(gen -> ExpressionParser.toJson(expression, gen), pretty);
    }

    public static void toJson(Expression expression, JsonGenerator gen) {
        ExpressionVisitors.visit(expression, new JsonGeneratorVisitor(gen));
    }

    public static Expression fromJson(String json) {
        return ExpressionParser.fromJson(json, null);
    }

    public static Expression fromJson(JsonNode json) {
        return ExpressionParser.fromJson(json, null);
    }

    public static Expression fromJson(String json, Schema schema) {
        return JsonUtil.parse(json, node -> ExpressionParser.fromJson(node, schema));
    }

    static Expression fromJson(JsonNode json, Schema schema) {
        Preconditions.checkArgument(null != json, "Cannot parse expression from null object");
        if (json.isBoolean()) {
            if (json.asBoolean()) {
                return Expressions.alwaysTrue();
            }
            return Expressions.alwaysFalse();
        }
        Preconditions.checkArgument(json.isObject(), "Cannot parse expression from non-object: %s", (Object)json);
        String type = JsonUtil.getString(TYPE, json);
        if (type.equalsIgnoreCase(LITERAL)) {
            if (JsonUtil.getBool(VALUE, json)) {
                return Expressions.alwaysTrue();
            }
            return Expressions.alwaysFalse();
        }
        Expression.Operation op = ExpressionParser.fromType(type);
        switch (op) {
            case NOT: {
                return Expressions.not(ExpressionParser.fromJson(JsonUtil.get(CHILD, json), schema));
            }
            case AND: {
                return Expressions.and(ExpressionParser.fromJson(JsonUtil.get(LEFT, json), schema), ExpressionParser.fromJson(JsonUtil.get(RIGHT, json), schema));
            }
            case OR: {
                return Expressions.or(ExpressionParser.fromJson(JsonUtil.get(LEFT, json), schema), ExpressionParser.fromJson(JsonUtil.get(RIGHT, json), schema));
            }
        }
        return ExpressionParser.predicateFromJson(op, json, schema);
    }

    private static Expression.Operation fromType(String type) {
        return Expression.Operation.fromString(type.replaceAll("-", "_"));
    }

    private static <T> UnboundPredicate<T> predicateFromJson(Expression.Operation op, JsonNode node, Schema schema) {
        Function<JsonNode, Object> convertValue;
        UnboundTerm<T> term = ExpressionParser.term(JsonUtil.get(TERM, node));
        if (schema != null) {
            BoundTerm bound = (BoundTerm)term.bind(schema.asStruct(), false);
            convertValue = valueNode -> SingleValueParser.fromJson(bound.type(), valueNode);
        } else {
            convertValue = valueNode -> ExpressionParser.asObject(valueNode);
        }
        switch (op) {
            case IS_NULL: 
            case NOT_NULL: 
            case IS_NAN: 
            case NOT_NAN: {
                Preconditions.checkArgument(!node.has(VALUE), "Cannot parse %s predicate: has invalid value field", (Object)op);
                Preconditions.checkArgument(!node.has(VALUES), "Cannot parse %s predicate: has invalid values field", (Object)op);
                return Expressions.predicate(op, term);
            }
            case LT: 
            case LT_EQ: 
            case GT: 
            case GT_EQ: 
            case EQ: 
            case NOT_EQ: 
            case STARTS_WITH: 
            case NOT_STARTS_WITH: {
                Preconditions.checkArgument(node.has(VALUE), "Cannot parse %s predicate: missing value", (Object)op);
                Preconditions.checkArgument(!node.has(VALUES), "Cannot parse %s predicate: has invalid values field", (Object)op);
                Object value = ExpressionParser.literal(JsonUtil.get(VALUE, node), convertValue);
                return Expressions.predicate(op, term, ImmutableList.of(value));
            }
            case IN: 
            case NOT_IN: {
                Preconditions.checkArgument(node.has(VALUES), "Cannot parse %s predicate: missing values", (Object)op);
                Preconditions.checkArgument(!node.has(VALUE), "Cannot parse %s predicate: has invalid value field", (Object)op);
                JsonNode valuesNode = JsonUtil.get(VALUES, node);
                Preconditions.checkArgument(valuesNode.isArray(), "Cannot parse literals from non-array: %s", (Object)valuesNode);
                return Expressions.predicate(op, term, Iterables.transform(((ArrayNode)valuesNode)::elements, valueNode -> ExpressionParser.literal(valueNode, convertValue)));
            }
        }
        throw new UnsupportedOperationException("Unsupported operation: " + op);
    }

    private static <T> T literal(JsonNode valueNode, Function<JsonNode, T> toValue) {
        if (valueNode.isObject() && valueNode.has(TYPE)) {
            String type = JsonUtil.getString(TYPE, valueNode);
            Preconditions.checkArgument(type.equalsIgnoreCase(LITERAL), "Cannot parse type as a literal: %s", (Object)type);
            return toValue.apply(JsonUtil.get(VALUE, valueNode));
        }
        return toValue.apply(valueNode);
    }

    private static Object asObject(JsonNode node) {
        if (node.isIntegralNumber() && node.canConvertToLong()) {
            return node.asLong();
        }
        if (node.isTextual()) {
            return node.asText();
        }
        if (node.isFloatingPointNumber()) {
            return node.asDouble();
        }
        if (node.isBoolean()) {
            return node.asBoolean();
        }
        throw new IllegalArgumentException("Cannot convert JSON to literal: " + node);
    }

    private static <T> UnboundTerm<T> term(JsonNode node) {
        if (node.isTextual()) {
            return Expressions.ref(node.asText());
        }
        if (node.isObject()) {
            String type;
            switch (type = JsonUtil.getString(TYPE, node)) {
                case "reference": {
                    return Expressions.ref(JsonUtil.getString(TERM, node));
                }
                case "transform": {
                    UnboundTerm<T> child = ExpressionParser.term(JsonUtil.get(TERM, node));
                    String transform = JsonUtil.getString(TRANSFORM, node);
                    return Expressions.transform(child.ref().name(), Transforms.fromString(transform));
                }
            }
            throw new IllegalArgumentException("Cannot parse type as a reference: " + type);
        }
        throw new IllegalArgumentException("Cannot parse reference (requires string or object): " + node);
    }

    private static class JsonGeneratorVisitor
    extends ExpressionVisitors.CustomOrderExpressionVisitor<Void> {
        private final JsonGenerator gen;

        private JsonGeneratorVisitor(JsonGenerator gen) {
            this.gen = gen;
        }

        private void toJson(Supplier<Void> child) {
            child.get();
        }

        private Void generate(Task task) {
            try {
                task.run();
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            return null;
        }

        @Override
        public Void alwaysTrue() {
            return this.generate(() -> this.gen.writeBoolean(true));
        }

        @Override
        public Void alwaysFalse() {
            return this.generate(() -> this.gen.writeBoolean(false));
        }

        @Override
        public Void not(Supplier<Void> child) {
            return this.generate(() -> {
                this.gen.writeStartObject();
                this.gen.writeStringField(ExpressionParser.TYPE, "not");
                this.gen.writeFieldName(ExpressionParser.CHILD);
                this.toJson(child);
                this.gen.writeEndObject();
            });
        }

        @Override
        public Void and(Supplier<Void> left, Supplier<Void> right) {
            return this.generate(() -> {
                this.gen.writeStartObject();
                this.gen.writeStringField(ExpressionParser.TYPE, "and");
                this.gen.writeFieldName(ExpressionParser.LEFT);
                this.toJson(left);
                this.gen.writeFieldName(ExpressionParser.RIGHT);
                this.toJson(right);
                this.gen.writeEndObject();
            });
        }

        @Override
        public Void or(Supplier<Void> left, Supplier<Void> right) {
            return this.generate(() -> {
                this.gen.writeStartObject();
                this.gen.writeStringField(ExpressionParser.TYPE, "or");
                this.gen.writeFieldName(ExpressionParser.LEFT);
                this.toJson(left);
                this.gen.writeFieldName(ExpressionParser.RIGHT);
                this.toJson(right);
                this.gen.writeEndObject();
            });
        }

        @Override
        public <T> Void predicate(BoundPredicate<T> pred) {
            return this.generate(() -> {
                this.gen.writeStartObject();
                this.gen.writeStringField(ExpressionParser.TYPE, this.operationType(pred.op()));
                this.gen.writeFieldName(ExpressionParser.TERM);
                this.term((Term)pred.term());
                if (pred.isLiteralPredicate()) {
                    this.gen.writeFieldName(ExpressionParser.VALUE);
                    SingleValueParser.toJson(((BoundTerm)pred.term()).type(), pred.asLiteralPredicate().literal().value(), this.gen);
                } else if (pred.isSetPredicate()) {
                    this.gen.writeArrayFieldStart(ExpressionParser.VALUES);
                    for (Object value : pred.asSetPredicate().literalSet()) {
                        SingleValueParser.toJson(((BoundTerm)pred.term()).type(), value, this.gen);
                    }
                    this.gen.writeEndArray();
                }
                this.gen.writeEndObject();
            });
        }

        @Override
        public <T> Void predicate(UnboundPredicate<T> pred) {
            return this.generate(() -> {
                this.gen.writeStartObject();
                this.gen.writeStringField(ExpressionParser.TYPE, this.operationType(pred.op()));
                this.gen.writeFieldName(ExpressionParser.TERM);
                this.term((Term)pred.term());
                if (pred.literals() != null) {
                    if (pred.op() == Expression.Operation.IN || pred.op() == Expression.Operation.NOT_IN) {
                        this.gen.writeArrayFieldStart(ExpressionParser.VALUES);
                        for (Literal lit : pred.literals()) {
                            this.unboundLiteral(lit.value());
                        }
                        this.gen.writeEndArray();
                    } else {
                        this.gen.writeFieldName(ExpressionParser.VALUE);
                        this.unboundLiteral(pred.literal().value());
                    }
                }
                this.gen.writeEndObject();
            });
        }

        private void unboundLiteral(Object object) throws IOException {
            if (object instanceof Integer) {
                SingleValueParser.toJson((Type)Types.IntegerType.get(), object, this.gen);
            } else if (object instanceof Long) {
                SingleValueParser.toJson((Type)Types.LongType.get(), object, this.gen);
            } else if (object instanceof String) {
                SingleValueParser.toJson((Type)Types.StringType.get(), object, this.gen);
            } else if (object instanceof Float) {
                SingleValueParser.toJson((Type)Types.FloatType.get(), object, this.gen);
            } else if (object instanceof Double) {
                SingleValueParser.toJson((Type)Types.DoubleType.get(), object, this.gen);
            } else if (object instanceof Boolean) {
                SingleValueParser.toJson((Type)Types.BooleanType.get(), object, this.gen);
            } else if (object instanceof ByteBuffer) {
                SingleValueParser.toJson((Type)Types.BinaryType.get(), object, this.gen);
            } else if (object instanceof byte[]) {
                SingleValueParser.toJson((Type)Types.BinaryType.get(), (Object)ByteBuffer.wrap((byte[])object), this.gen);
            } else if (object instanceof UUID) {
                SingleValueParser.toJson((Type)Types.UUIDType.get(), object, this.gen);
            } else if (object instanceof BigDecimal) {
                BigDecimal decimal = (BigDecimal)object;
                SingleValueParser.toJson((Type)Types.DecimalType.of(decimal.precision(), decimal.scale()), (Object)decimal, this.gen);
            }
        }

        private String operationType(Expression.Operation op) {
            return op.toString().replaceAll("_", "-").toLowerCase(Locale.ENGLISH);
        }

        private void term(Term term) throws IOException {
            if (term instanceof UnboundTransform) {
                UnboundTransform transform = (UnboundTransform)term;
                this.transform(transform.transform().toString(), transform.ref().name());
                return;
            }
            if (term instanceof BoundTransform) {
                BoundTransform transform = (BoundTransform)term;
                this.transform(transform.transform().toString(), transform.ref().name());
                return;
            }
            if (term instanceof Reference) {
                this.gen.writeString(((Reference)term).name());
                return;
            }
            throw new UnsupportedOperationException("Cannot write unsupported term: " + term);
        }

        private void transform(String transform, String name) throws IOException {
            this.gen.writeStartObject();
            this.gen.writeStringField(ExpressionParser.TYPE, ExpressionParser.TRANSFORM);
            this.gen.writeStringField(ExpressionParser.TRANSFORM, transform);
            this.gen.writeStringField(ExpressionParser.TERM, name);
            this.gen.writeEndObject();
        }

        @FunctionalInterface
        private static interface Task {
            public void run() throws IOException;
        }
    }
}

