/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.client.auth.oauth2;

import java.io.Closeable;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.projectnessie.client.auth.oauth2.AccessToken;
import org.projectnessie.client.auth.oauth2.Flow;
import org.projectnessie.client.auth.oauth2.GrantType;
import org.projectnessie.client.auth.oauth2.ImpersonationFlow;
import org.projectnessie.client.auth.oauth2.MustFetchNewTokensException;
import org.projectnessie.client.auth.oauth2.OAuth2Authenticator;
import org.projectnessie.client.auth.oauth2.OAuth2ClientConfig;
import org.projectnessie.client.auth.oauth2.OAuth2TokenRefreshExecutor;
import org.projectnessie.client.auth.oauth2.OAuth2Utils;
import org.projectnessie.client.auth.oauth2.Token;
import org.projectnessie.client.auth.oauth2.Tokens;
import org.projectnessie.client.http.HttpClient;
import org.projectnessie.client.http.HttpClientException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OAuth2Client
implements OAuth2Authenticator,
Closeable {
    static final Logger LOGGER = LoggerFactory.getLogger(OAuth2Client.class);
    private static final Duration MIN_WARN_INTERVAL = Duration.ofSeconds(10L);
    private final OAuth2ClientConfig config;
    private final HttpClient httpClient;
    private final ScheduledExecutorService executor;
    private final CompletableFuture<Void> started = new CompletableFuture();
    private final CompletableFuture<Void> used = new CompletableFuture();
    final AtomicBoolean sleeping = new AtomicBoolean();
    private final AtomicBoolean closing = new AtomicBoolean();
    private volatile CompletionStage<Tokens> currentTokensStage;
    private volatile ScheduledFuture<?> tokenRefreshFuture;
    private volatile Instant lastAccess;
    private volatile Instant lastWarn;

    OAuth2Client(OAuth2ClientConfig config) {
        this.config = config;
        this.httpClient = config.getHttpClient();
        this.executor = config.getExecutor().orElseGet(() -> new OAuth2TokenRefreshExecutor(config));
        CompletableFuture<Void> ready = config.getGrantType().requiresUserInteraction() ? CompletableFuture.allOf(this.started, this.used) : this.started;
        this.currentTokensStage = ((CompletableFuture)ready.thenApplyAsync(v -> this.fetchNewTokens(), (Executor)this.executor)).thenApply(this::maybeImpersonate);
        this.currentTokensStage.whenComplete((tokens, error) -> this.log((Throwable)error)).whenComplete((tokens, error) -> this.maybeScheduleTokensRenewal((Tokens)tokens));
    }

    @Override
    public AccessToken authenticate() {
        Instant now;
        if (this.closing.get()) {
            throw new IllegalStateException("Client is closing");
        }
        this.used.complete(null);
        this.lastAccess = now = this.config.getClock().get();
        if (this.sleeping.compareAndSet(true, false)) {
            this.wakeUp(now);
        }
        return this.getCurrentTokens().getAccessToken();
    }

    Tokens getCurrentTokens() {
        try {
            return this.currentTokensStage.toCompletableFuture().get();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof Error) {
                throw (Error)cause;
            }
            if (cause instanceof HttpClientException) {
                throw (HttpClientException)cause;
            }
            throw new RuntimeException("Cannot acquire a valid OAuth2 access token", cause);
        }
    }

    private Tokens getCurrentTokensIfAvailable() {
        try {
            return this.currentTokensStage.toCompletableFuture().getNow(null);
        }
        catch (CancellationException | CompletionException runtimeException) {
            return null;
        }
    }

    @Override
    public void start() {
        this.lastAccess = this.config.getClock().get();
        this.started.complete(null);
    }

    @Override
    public void close() {
        if (this.closing.compareAndSet(false, true)) {
            LOGGER.debug("[{}] Closing...", (Object)this.config.getClientName());
            try {
                this.currentTokensStage.toCompletableFuture().cancel(true);
                ScheduledFuture<?> tokenRefreshFuture = this.tokenRefreshFuture;
                if (tokenRefreshFuture != null) {
                    tokenRefreshFuture.cancel(true);
                }
                if (this.executor instanceof OAuth2TokenRefreshExecutor) {
                    ((OAuth2TokenRefreshExecutor)this.executor).close();
                }
                this.httpClient.close();
            }
            finally {
                this.used.cancel(true);
                this.tokenRefreshFuture = null;
            }
            LOGGER.debug("[{}] Closed", (Object)this.config.getClientName());
        }
    }

    @Override
    public OAuth2Authenticator copy() {
        return new OAuth2Client(OAuth2ClientConfig.builder().from(this.config).build());
    }

    private void wakeUp(Instant now) {
        if (this.closing.get()) {
            LOGGER.debug("[{}] Not waking up, client is closing", (Object)this.config.getClientName());
            return;
        }
        LOGGER.debug("[{}] Waking up...", (Object)this.config.getClientName());
        Tokens currentTokens = this.getCurrentTokensIfAvailable();
        Duration delay = this.nextTokenRefresh(currentTokens, now, Duration.ZERO);
        if (delay.compareTo(this.config.getMinRefreshSafetyWindow()) < 0) {
            LOGGER.debug("[{}] Refreshing tokens immediately", (Object)this.config.getClientName());
            this.renewTokens();
        } else {
            LOGGER.debug("[{}] Tokens are still valid", (Object)this.config.getClientName());
            this.scheduleTokensRenewal(delay);
        }
    }

    private void maybeScheduleTokensRenewal(Tokens currentTokens) {
        if (this.closing.get()) {
            LOGGER.debug("[{}] Not checking if token renewal is required, client is closing", (Object)this.config.getClientName());
            return;
        }
        Instant now = this.config.getClock().get();
        if (Duration.between(this.lastAccess, now).compareTo(this.config.getPreemptiveTokenRefreshIdleTimeout()) > 0) {
            this.sleeping.set(true);
            LOGGER.debug("[{}] Sleeping...", (Object)this.config.getClientName());
        } else {
            Duration delay = this.nextTokenRefresh(currentTokens, now, this.config.getMinRefreshSafetyWindow());
            this.scheduleTokensRenewal(delay);
        }
    }

    private void scheduleTokensRenewal(Duration delay) {
        if (this.closing.get()) {
            LOGGER.debug("[{}] Not scheduling token renewal, client is closing", (Object)this.config.getClientName());
            return;
        }
        LOGGER.debug("[{}] Scheduling token refresh in {}", (Object)this.config.getClientName(), (Object)delay);
        try {
            this.tokenRefreshFuture = this.executor.schedule(this::renewTokens, delay.toMillis(), TimeUnit.MILLISECONDS);
            if (this.closing.get()) {
                this.tokenRefreshFuture.cancel(true);
            }
        }
        catch (RejectedExecutionException e) {
            if (this.closing.get()) {
                return;
            }
            this.maybeWarn("Failed to schedule next token renewal, forcibly sleeping", null);
            this.sleeping.set(true);
        }
    }

    private void renewTokens() {
        CompletionStage<Tokens> oldTokensStage = this.currentTokensStage;
        this.currentTokensStage = oldTokensStage.thenApply(this::refreshTokens).exceptionally(error -> this.fetchNewTokens()).thenApply(this::maybeImpersonate);
        this.currentTokensStage.whenComplete((tokens, error) -> this.log((Throwable)error)).whenComplete((tokens, error) -> this.maybeScheduleTokensRenewal((Tokens)tokens));
    }

    private void log(Throwable error) {
        if (error != null) {
            if (this.closing.get()) {
                return;
            }
            if (error instanceof CompletionException) {
                error = error.getCause();
            }
            this.maybeWarn("Failed to fetch new tokens", error);
        } else {
            LOGGER.debug("[{}] Successfully fetched new tokens", (Object)this.config.getClientName());
        }
    }

    Tokens fetchNewTokens() {
        LOGGER.debug("[{}] Fetching new tokens using {}", (Object)this.config.getClientName(), (Object)this.config.getGrantType());
        try {
            Flow flow = this.config.getGrantType().newFlow(this.config);
            try {
                Tokens tokens = flow.fetchNewTokens(null);
                if (flow != null) {
                    flow.close();
                }
                return tokens;
            }
            catch (Throwable throwable) {
                if (flow != null) {
                    try {
                        flow.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
        }
        finally {
            if (this.config.getGrantType().requiresUserInteraction()) {
                this.lastAccess = this.config.getClock().get();
            }
        }
    }

    Tokens refreshTokens(Tokens currentTokens) {
        if (currentTokens.getRefreshToken() == null) {
            throw new MustFetchNewTokensException("No refresh token available");
        }
        if (this.isAboutToExpire(currentTokens.getRefreshToken(), this.config.getDefaultRefreshTokenLifespan())) {
            throw new MustFetchNewTokensException("Refresh token is about to expire");
        }
        LOGGER.debug("[{}] Refreshing tokens", (Object)this.config.getClientName());
        try (Flow flow = GrantType.REFRESH_TOKEN.newFlow(this.config);){
            Tokens tokens = flow.fetchNewTokens(currentTokens);
            return tokens;
        }
    }

    Tokens maybeImpersonate(Tokens currentTokens) {
        if (this.config.getImpersonationConfig().isEnabled()) {
            LOGGER.debug("[{}] Performing impersonation", (Object)this.config.getClientName());
            try (ImpersonationFlow flow = new ImpersonationFlow(this.config);){
                Tokens tokens = flow.fetchNewTokens(currentTokens);
                return tokens;
            }
        }
        return currentTokens;
    }

    private boolean isAboutToExpire(Token token, Duration defaultLifespan) {
        Instant now = this.config.getClock().get();
        Instant expirationTime = OAuth2Utils.tokenExpirationTime(now, token, defaultLifespan);
        return expirationTime.isBefore(now.plus(this.config.getRefreshSafetyWindow()));
    }

    private Duration nextTokenRefresh(Tokens currentTokens, Instant now, Duration minRefreshDelay) {
        if (currentTokens == null) {
            return minRefreshDelay;
        }
        Instant accessExpirationTime = OAuth2Utils.tokenExpirationTime(now, currentTokens.getAccessToken(), this.config.getDefaultAccessTokenLifespan());
        Instant refreshExpirationTime = currentTokens.getRefreshToken() == null ? null : OAuth2Utils.tokenExpirationTime(now, currentTokens.getRefreshToken(), this.config.getDefaultRefreshTokenLifespan());
        return OAuth2Utils.shortestDelay(now, accessExpirationTime, refreshExpirationTime, this.config.getRefreshSafetyWindow(), minRefreshDelay);
    }

    private void maybeWarn(String message, Throwable error) {
        boolean shouldWarn;
        Instant now = this.config.getClock().get();
        boolean bl = shouldWarn = this.lastWarn == null || Duration.between(this.lastWarn, now).compareTo(MIN_WARN_INTERVAL) > 0;
        if (shouldWarn) {
            if (error instanceof HttpClientException) {
                this.used.thenRun(() -> LOGGER.warn("{}: {}", (Object)message, (Object)error.toString()));
            } else {
                this.used.thenRun(() -> LOGGER.warn(message, error));
            }
            this.lastWarn = now;
        } else {
            LOGGER.debug(message, error);
        }
    }
}

