/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.client.auth.oauth2;

import java.io.PrintStream;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.annotation.Nullable;
import org.projectnessie.client.auth.oauth2.AbstractFlow;
import org.projectnessie.client.auth.oauth2.DeviceCodeResponse;
import org.projectnessie.client.auth.oauth2.DeviceCodeTokenRequest;
import org.projectnessie.client.auth.oauth2.DeviceCodeTokenResponse;
import org.projectnessie.client.auth.oauth2.OAuth2ClientConfig;
import org.projectnessie.client.auth.oauth2.OAuth2Exception;
import org.projectnessie.client.auth.oauth2.Tokens;
import org.projectnessie.client.http.HttpClientException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DeviceCodeFlow
extends AbstractFlow {
    static final String MSG_PREFIX = "[nessie-oauth2-client] ";
    private static final Logger LOGGER = LoggerFactory.getLogger(DeviceCodeFlow.class);
    private final PrintStream console;
    private final Duration flowTimeout;
    private final CompletableFuture<Tokens> tokensFuture = new CompletableFuture();
    private final CompletableFuture<Void> closeFuture = new CompletableFuture();
    private final ScheduledExecutorService executor;
    private volatile Duration pollInterval;
    private volatile Future<?> pollFuture;

    DeviceCodeFlow(OAuth2ClientConfig config) {
        super(config);
        this.console = config.getConsole();
        this.flowTimeout = config.getDeviceCodeFlowTimeout();
        this.pollInterval = config.getDeviceCodeFlowPollInterval();
        this.closeFuture.thenRun(this::doClose);
        LOGGER.debug("Device Code Flow: started");
        this.executor = Executors.newSingleThreadScheduledExecutor();
    }

    @Override
    public void close() {
        this.closeFuture.complete(null);
    }

    private void doClose() {
        LOGGER.debug("Device Code Flow: closing");
        this.executor.shutdownNow();
        this.pollFuture = null;
    }

    private void abort() {
        Future<?> pollFuture = this.pollFuture;
        if (pollFuture != null) {
            pollFuture.cancel(true);
        }
        this.tokensFuture.cancel(true);
    }

    @Override
    public Tokens fetchNewTokens(@Nullable Tokens currentTokens) {
        DeviceCodeResponse response = this.invokeDeviceAuthEndpoint();
        this.checkPollInterval(response.getIntervalSeconds());
        this.console.println();
        this.console.println("[nessie-oauth2-client] ======= Nessie authentication required =======");
        this.console.println("[nessie-oauth2-client] Browse to the following URL:");
        this.console.println(MSG_PREFIX + response.getVerificationUri());
        this.console.println("[nessie-oauth2-client] And enter the code:");
        this.console.println(MSG_PREFIX + response.getUserCode());
        this.printExpirationNotice(response.getExpiresInSeconds());
        this.console.println();
        this.console.flush();
        this.pollFuture = this.executor.submit(() -> this.pollForNewTokens(response.getDeviceCode()));
        try {
            return this.tokensFuture.get(this.flowTimeout.toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            LOGGER.error("Timed out waiting for user to authorize device.");
            this.abort();
            throw new RuntimeException("Timed out waiting for user to authorize device", e);
        }
        catch (InterruptedException e) {
            this.abort();
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            this.abort();
            Throwable cause = e.getCause();
            LOGGER.error("Authentication failed: {}", (Object)cause.getMessage());
            if (cause instanceof HttpClientException) {
                throw (HttpClientException)cause;
            }
            throw new RuntimeException(cause);
        }
    }

    private void checkPollInterval(Integer serverPollInterval) {
        if (!this.config.ignoreDeviceCodeFlowServerPollInterval() && serverPollInterval != null && (long)serverPollInterval.intValue() > this.pollInterval.getSeconds()) {
            LOGGER.debug("Device Code Flow: server requested minimum poll interval of {} seconds", (Object)serverPollInterval);
            this.pollInterval = Duration.ofSeconds(serverPollInterval.intValue());
        }
    }

    private void printExpirationNotice(int seconds) {
        String exp = seconds < 60 ? seconds + " seconds" : (seconds % 60 == 0 ? seconds / 60 + " minutes" : seconds / 60 + " minutes and " + seconds % 60 + " seconds");
        this.console.println("[nessie-oauth2-client] (The code will expire in " + exp + ")");
    }

    private void pollForNewTokens(String deviceCode) {
        try {
            LOGGER.debug("Device Code Flow: polling for new tokens");
            DeviceCodeTokenRequest.Builder request = DeviceCodeTokenRequest.builder().deviceCode(deviceCode);
            Tokens tokens = this.invokeTokenEndpoint(request, DeviceCodeTokenResponse.class);
            LOGGER.debug("Device Code Flow: new tokens received");
            this.tokensFuture.complete(tokens);
        }
        catch (OAuth2Exception e) {
            switch (e.getErrorCode()) {
                case "authorization_pending": {
                    LOGGER.debug("Device Code Flow: waiting for authorization to complete");
                    this.pollFuture = this.executor.schedule(() -> this.pollForNewTokens(deviceCode), this.pollInterval.toMillis(), TimeUnit.MILLISECONDS);
                    return;
                }
                case "slow_down": {
                    LOGGER.debug("Device Code Flow: server requested to slow down");
                    Duration pollInterval = this.pollInterval;
                    if (!this.config.ignoreDeviceCodeFlowServerPollInterval()) {
                        this.pollInterval = pollInterval = pollInterval.plus(pollInterval);
                    }
                    this.pollFuture = this.executor.schedule(() -> this.pollForNewTokens(deviceCode), pollInterval.toMillis(), TimeUnit.MILLISECONDS);
                    return;
                }
            }
            this.tokensFuture.completeExceptionally(e);
        }
        catch (Exception e) {
            this.tokensFuture.completeExceptionally(e);
        }
    }
}

