/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.jdbc;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.sql.SQLNonTransientConnectionException;
import java.sql.SQLTimeoutException;
import java.sql.SQLTransientConnectionException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.iceberg.BaseMetastoreCatalog;
import org.apache.iceberg.CatalogUtil;
import org.apache.iceberg.Schema;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.Transaction;
import org.apache.iceberg.catalog.Catalog;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.SupportsNamespaces;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.exceptions.AlreadyExistsException;
import org.apache.iceberg.exceptions.NamespaceNotEmptyException;
import org.apache.iceberg.exceptions.NoSuchNamespaceException;
import org.apache.iceberg.exceptions.NoSuchTableException;
import org.apache.iceberg.exceptions.NoSuchViewException;
import org.apache.iceberg.exceptions.NotFoundException;
import org.apache.iceberg.hadoop.Configurable;
import org.apache.iceberg.io.CloseableGroup;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.jdbc.JdbcClientPool;
import org.apache.iceberg.jdbc.JdbcTableOperations;
import org.apache.iceberg.jdbc.JdbcUtil;
import org.apache.iceberg.jdbc.JdbcViewOperations;
import org.apache.iceberg.jdbc.UncheckedInterruptedException;
import org.apache.iceberg.jdbc.UncheckedSQLException;
import org.apache.iceberg.relocated.com.google.common.base.Joiner;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.base.Strings;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMap;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.util.LocationUtil;
import org.apache.iceberg.util.PropertyUtil;
import org.apache.iceberg.view.BaseMetastoreViewCatalog;
import org.apache.iceberg.view.ViewOperations;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcCatalog
extends BaseMetastoreViewCatalog
implements Configurable<Object>,
SupportsNamespaces {
    public static final String PROPERTY_PREFIX = "jdbc.";
    private static final String NAMESPACE_EXISTS_PROPERTY = "exists";
    private static final Logger LOG = LoggerFactory.getLogger(JdbcCatalog.class);
    private static final Joiner SLASH = Joiner.on("/");
    static final String VIEW_WARNING_LOG_MESSAGE = "JDBC catalog is initialized without view support. To auto-migrate the database's schema and enable view support, set jdbc.schema-version=V1";
    private FileIO io;
    private String catalogName = "jdbc";
    private String warehouseLocation;
    private Object conf;
    private JdbcClientPool connections;
    private Map<String, String> catalogProperties;
    private final Function<Map<String, String>, FileIO> ioBuilder;
    private final Function<Map<String, String>, JdbcClientPool> clientPoolBuilder;
    private final boolean initializeCatalogTables;
    private CloseableGroup closeableGroup;
    private JdbcUtil.SchemaVersion schemaVersion = JdbcUtil.SchemaVersion.V0;

    public JdbcCatalog() {
        this(null, null, true);
    }

    public JdbcCatalog(Function<Map<String, String>, FileIO> ioBuilder, Function<Map<String, String>, JdbcClientPool> clientPoolBuilder, boolean initializeCatalogTables) {
        this.ioBuilder = ioBuilder;
        this.clientPoolBuilder = clientPoolBuilder;
        this.initializeCatalogTables = initializeCatalogTables;
    }

    @Override
    public void initialize(String name, Map<String, String> properties) {
        Preconditions.checkNotNull(properties, "Invalid catalog properties: null");
        String uri = properties.get("uri");
        Preconditions.checkNotNull(uri, "JDBC connection URI is required");
        String inputWarehouseLocation = properties.get("warehouse");
        Preconditions.checkArgument(!Strings.isNullOrEmpty(inputWarehouseLocation), "Cannot initialize JDBCCatalog because warehousePath must not be null or empty");
        this.warehouseLocation = LocationUtil.stripTrailingSlash(inputWarehouseLocation);
        this.catalogProperties = ImmutableMap.copyOf(properties);
        if (name != null) {
            this.catalogName = name;
        }
        if (null != this.ioBuilder) {
            this.io = this.ioBuilder.apply(properties);
        } else {
            String ioImpl = properties.getOrDefault("io-impl", "org.apache.iceberg.hadoop.HadoopFileIO");
            this.io = CatalogUtil.loadFileIO(ioImpl, properties, this.conf);
        }
        LOG.debug("Connecting to JDBC database {}", (Object)uri);
        this.connections = null != this.clientPoolBuilder ? this.clientPoolBuilder.apply(properties) : new JdbcClientPool(uri, properties);
        if (this.initializeCatalogTables) {
            this.initializeCatalogTables();
        }
        this.updateSchemaIfRequired();
        this.closeableGroup = new CloseableGroup();
        this.closeableGroup.addCloseable(this.metricsReporter());
        this.closeableGroup.addCloseable(this.connections);
        this.closeableGroup.setSuppressCloseFailure(true);
    }

    private void initializeCatalogTables() {
        LOG.trace("Creating database tables (if missing) to store iceberg catalog");
        try {
            this.connections.run(conn -> {
                DatabaseMetaData dbMeta = conn.getMetaData();
                ResultSet tableExists = dbMeta.getTables(null, null, "iceberg_tables", null);
                if (tableExists.next()) {
                    return true;
                }
                LOG.debug("Creating table {} to store iceberg catalog tables", (Object)"iceberg_tables");
                return conn.prepareStatement("CREATE TABLE iceberg_tables(catalog_name VARCHAR(255) NOT NULL,table_namespace VARCHAR(255) NOT NULL,table_name VARCHAR(255) NOT NULL,metadata_location VARCHAR(1000),previous_metadata_location VARCHAR(1000),PRIMARY KEY (catalog_name, table_namespace, table_name))").execute();
            });
            this.connections.run(conn -> {
                DatabaseMetaData dbMeta = conn.getMetaData();
                ResultSet tableExists = dbMeta.getTables(null, null, "iceberg_namespace_properties", null);
                if (tableExists.next()) {
                    return true;
                }
                LOG.debug("Creating table {} to store iceberg catalog namespace properties", (Object)"iceberg_namespace_properties");
                return conn.prepareStatement("CREATE TABLE iceberg_namespace_properties(catalog_name VARCHAR(255) NOT NULL,namespace VARCHAR(255) NOT NULL,property_key VARCHAR(255),property_value VARCHAR(1000),PRIMARY KEY (catalog_name, namespace, property_key))").execute();
            });
        }
        catch (SQLTimeoutException e) {
            throw new UncheckedSQLException(e, "Cannot initialize JDBC catalog: Query timed out", new Object[0]);
        }
        catch (SQLNonTransientConnectionException | SQLTransientConnectionException e) {
            throw new UncheckedSQLException(e, "Cannot initialize JDBC catalog: Connection failed", new Object[0]);
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Cannot initialize JDBC catalog", new Object[0]);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted in call to initialize", new Object[0]);
        }
    }

    private void updateSchemaIfRequired() {
        try {
            this.connections.run(conn -> {
                DatabaseMetaData dbMeta = conn.getMetaData();
                ResultSet typeColumn = dbMeta.getColumns(null, null, "iceberg_tables", "iceberg_type");
                if (typeColumn.next()) {
                    LOG.debug("{} already supports views", (Object)"iceberg_tables");
                    this.schemaVersion = JdbcUtil.SchemaVersion.V1;
                    return true;
                }
                if (PropertyUtil.propertyAsString(this.catalogProperties, "jdbc.schema-version", JdbcUtil.SchemaVersion.V0.name()).equalsIgnoreCase(JdbcUtil.SchemaVersion.V1.name())) {
                    LOG.debug("{} is being updated to support views", (Object)"iceberg_tables");
                    this.schemaVersion = JdbcUtil.SchemaVersion.V1;
                    return conn.prepareStatement("ALTER TABLE iceberg_tables ADD COLUMN iceberg_type VARCHAR(5)").execute();
                }
                LOG.warn(VIEW_WARNING_LOG_MESSAGE);
                return true;
            });
        }
        catch (SQLTimeoutException e) {
            throw new UncheckedSQLException(e, "Cannot update JDBC catalog: Query timed out", new Object[0]);
        }
        catch (SQLNonTransientConnectionException | SQLTransientConnectionException e) {
            throw new UncheckedSQLException(e, "Cannot update JDBC catalog: Connection failed", new Object[0]);
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Cannot check and eventually update SQL schema", new Object[0]);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted in call to initialize", new Object[0]);
        }
    }

    @Override
    protected TableOperations newTableOps(TableIdentifier tableIdentifier) {
        return new JdbcTableOperations(this.connections, this.io, this.catalogName, tableIdentifier, this.catalogProperties, this.schemaVersion);
    }

    @Override
    protected ViewOperations newViewOps(TableIdentifier viewIdentifier) {
        if (this.schemaVersion != JdbcUtil.SchemaVersion.V1) {
            throw new UnsupportedOperationException(VIEW_WARNING_LOG_MESSAGE);
        }
        return new JdbcViewOperations(this.connections, this.io, this.catalogName, viewIdentifier, this.catalogProperties);
    }

    @Override
    protected String defaultWarehouseLocation(TableIdentifier table) {
        return SLASH.join(this.defaultNamespaceLocation(table.namespace()), table.name(), new Object[0]);
    }

    @Override
    public boolean dropTable(TableIdentifier identifier, boolean purge) {
        int deletedRecords;
        TableOperations ops = this.newTableOps(identifier);
        TableMetadata lastMetadata = null;
        if (purge) {
            try {
                lastMetadata = ops.current();
            }
            catch (NotFoundException e) {
                LOG.warn("Failed to load table metadata for table: {}, continuing drop without purge", (Object)identifier, (Object)e);
            }
        }
        if ((deletedRecords = this.execute(this.schemaVersion == JdbcUtil.SchemaVersion.V1 ? "DELETE FROM iceberg_tables WHERE catalog_name = ? AND table_namespace  = ? AND table_name = ? AND (iceberg_type = 'TABLE' OR iceberg_type IS NULL)" : "DELETE FROM iceberg_tables WHERE catalog_name = ? AND table_namespace  = ? AND table_name = ?", this.catalogName, JdbcUtil.namespaceToString(identifier.namespace()), identifier.name())) == 0) {
            LOG.info("Skipping drop, table does not exist: {}", (Object)identifier);
            return false;
        }
        if (purge && lastMetadata != null) {
            CatalogUtil.dropTableData(ops.io(), lastMetadata);
        }
        LOG.info("Dropped table: {}", (Object)identifier);
        return true;
    }

    @Override
    public List<TableIdentifier> listTables(Namespace namespace) {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        return this.fetch(row -> JdbcUtil.stringToTableIdentifier(row.getString("table_namespace"), row.getString("table_name")), this.schemaVersion == JdbcUtil.SchemaVersion.V1 ? "SELECT * FROM iceberg_tables WHERE catalog_name = ? AND table_namespace = ? AND (iceberg_type = 'TABLE' OR iceberg_type IS NULL)" : "SELECT * FROM iceberg_tables WHERE catalog_name = ? AND table_namespace = ?", this.catalogName, JdbcUtil.namespaceToString(namespace));
    }

    @Override
    public void renameTable(TableIdentifier from, TableIdentifier to) {
        if (from.equals(to)) {
            return;
        }
        if (!this.tableExists(from)) {
            throw new NoSuchTableException("Table does not exist: %s", from);
        }
        if (!this.namespaceExists(to.namespace())) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", to.namespace());
        }
        if (this.schemaVersion == JdbcUtil.SchemaVersion.V1 && this.viewExists(to)) {
            throw new AlreadyExistsException("Cannot rename %s to %s. View already exists", from, to);
        }
        if (this.tableExists(to)) {
            throw new AlreadyExistsException("Table already exists: %s", to);
        }
        int updatedRecords = this.execute((SQLException err) -> {
            if (err instanceof SQLIntegrityConstraintViolationException || err.getMessage() != null && err.getMessage().contains("constraint failed")) {
                throw new AlreadyExistsException("Table already exists: %s", to);
            }
        }, this.schemaVersion == JdbcUtil.SchemaVersion.V1 ? "UPDATE iceberg_tables SET table_namespace = ?, table_name = ? WHERE catalog_name = ? AND table_namespace = ? AND table_name = ? AND (iceberg_type = 'TABLE' OR iceberg_type IS NULL)" : "UPDATE iceberg_tables SET table_namespace = ?, table_name = ? WHERE catalog_name = ? AND table_namespace = ? AND table_name = ?", JdbcUtil.namespaceToString(to.namespace()), to.name(), this.catalogName, JdbcUtil.namespaceToString(from.namespace()), from.name());
        if (updatedRecords == 1) {
            LOG.info("Renamed table from {}, to {}", (Object)from, (Object)to);
        } else {
            if (updatedRecords == 0) {
                throw new NoSuchTableException("Table does not exist: %s", from);
            }
            LOG.warn("Rename operation affected {} rows: the catalog table's primary key assumption has been violated", (Object)updatedRecords);
        }
    }

    @Override
    public String name() {
        return this.catalogName;
    }

    @Override
    public void setConf(Object conf) {
        this.conf = conf;
    }

    @Override
    public void createNamespace(Namespace namespace, Map<String, String> metadata) {
        if (this.namespaceExists(namespace)) {
            throw new AlreadyExistsException("Namespace already exists: %s", namespace);
        }
        ImmutableMap<String, String> createMetadata = metadata == null || metadata.isEmpty() ? ImmutableMap.of(NAMESPACE_EXISTS_PROPERTY, "true") : ImmutableMap.builder().putAll(metadata).put(NAMESPACE_EXISTS_PROPERTY, "true").buildOrThrow();
        this.insertProperties(namespace, createMetadata);
    }

    @Override
    public List<Namespace> listNamespaces() {
        ArrayList<Namespace> namespaces = Lists.newArrayList();
        namespaces.addAll(this.fetch(row -> JdbcUtil.stringToNamespace(row.getString("table_namespace")), "SELECT DISTINCT table_namespace FROM iceberg_tables WHERE catalog_name = ?", this.catalogName));
        namespaces.addAll(this.fetch(row -> JdbcUtil.stringToNamespace(row.getString("namespace")), "SELECT DISTINCT namespace FROM iceberg_namespace_properties WHERE catalog_name = ?", this.catalogName));
        namespaces = namespaces.stream().filter(n -> n.levels().length >= 1).map(n -> Namespace.of((String[])Arrays.stream(n.levels()).limit(1L).toArray(String[]::new))).distinct().collect(Collectors.toList());
        return namespaces;
    }

    @Override
    public List<Namespace> listNamespaces(Namespace namespace) throws NoSuchNamespaceException {
        if (namespace.isEmpty()) {
            return this.listNamespaces();
        }
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        ArrayList<Namespace> namespaces = Lists.newArrayList();
        namespaces.addAll(this.fetch(row -> JdbcUtil.stringToNamespace(row.getString("table_namespace")), "SELECT DISTINCT table_namespace FROM iceberg_tables WHERE catalog_name = ? AND table_namespace LIKE ?", this.catalogName, JdbcUtil.namespaceToString(namespace) + "%"));
        namespaces.addAll(this.fetch(row -> JdbcUtil.stringToNamespace(row.getString("namespace")), "SELECT DISTINCT namespace FROM iceberg_namespace_properties WHERE catalog_name = ? AND namespace LIKE ?", this.catalogName, JdbcUtil.namespaceToString(namespace) + "%"));
        int subNamespaceLevelLength = namespace.levels().length + 1;
        namespaces = namespaces.stream().filter(n -> !n.equals(namespace)).filter(n -> n.levels().length >= subNamespaceLevelLength).map(n -> Namespace.of((String[])Arrays.stream(n.levels()).limit(subNamespaceLevelLength).toArray(String[]::new))).distinct().collect(Collectors.toList());
        return namespaces;
    }

    @Override
    public Map<String, String> loadNamespaceMetadata(Namespace namespace) throws NoSuchNamespaceException {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        HashMap<String, String> properties = Maps.newHashMap();
        properties.putAll(this.fetchProperties(namespace));
        if (!properties.containsKey("location")) {
            properties.put("location", this.defaultNamespaceLocation(namespace));
        }
        properties.remove(NAMESPACE_EXISTS_PROPERTY);
        return ImmutableMap.copyOf(properties);
    }

    private String defaultNamespaceLocation(Namespace namespace) {
        if (namespace.isEmpty()) {
            return this.warehouseLocation;
        }
        return SLASH.join(this.warehouseLocation, SLASH.join(namespace.levels()), new Object[0]);
    }

    @Override
    public boolean dropNamespace(Namespace namespace) throws NamespaceNotEmptyException {
        if (!this.namespaceExists(namespace)) {
            return false;
        }
        List<TableIdentifier> tableIdentifiers = this.listTables(namespace);
        if (tableIdentifiers != null && !tableIdentifiers.isEmpty()) {
            throw new NamespaceNotEmptyException("Namespace %s is not empty. %s tables exist.", namespace, tableIdentifiers.size());
        }
        int deletedRows = this.execute("DELETE FROM iceberg_namespace_properties WHERE catalog_name = ? AND namespace = ?", this.catalogName, JdbcUtil.namespaceToString(namespace));
        return deletedRows > 0;
    }

    @Override
    public boolean setProperties(Namespace namespace, Map<String, String> properties) throws NoSuchNamespaceException {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        Preconditions.checkNotNull(properties, "Invalid properties to set: null");
        if (properties.isEmpty()) {
            return false;
        }
        Preconditions.checkArgument(!properties.containsKey(NAMESPACE_EXISTS_PROPERTY), "Cannot set reserved property: %s", (Object)NAMESPACE_EXISTS_PROPERTY);
        Map<String, String> startingProperties = this.fetchProperties(namespace);
        HashMap<String, String> inserts = Maps.newHashMap();
        HashMap<String, String> updates = Maps.newHashMap();
        for (String key : properties.keySet()) {
            String value = properties.get(key);
            if (startingProperties.containsKey(key)) {
                updates.put(key, value);
                continue;
            }
            inserts.put(key, value);
        }
        boolean hadInserts = false;
        if (!inserts.isEmpty()) {
            hadInserts = this.insertProperties(namespace, inserts);
        }
        boolean hadUpdates = false;
        if (!updates.isEmpty()) {
            hadUpdates = this.updateProperties(namespace, updates);
        }
        return hadInserts || hadUpdates;
    }

    @Override
    public boolean removeProperties(Namespace namespace, Set<String> properties) throws NoSuchNamespaceException {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        Preconditions.checkNotNull(properties, "Invalid properties to remove: null");
        if (properties.isEmpty()) {
            return false;
        }
        return this.deleteProperties(namespace, properties);
    }

    @Override
    public void close() {
        if (this.closeableGroup != null) {
            try {
                this.closeableGroup.close();
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }

    @Override
    public boolean namespaceExists(Namespace namespace) {
        return JdbcUtil.namespaceExists(this.catalogName, this.connections, namespace);
    }

    @Override
    public boolean dropView(TableIdentifier identifier) {
        if (this.schemaVersion != JdbcUtil.SchemaVersion.V1) {
            throw new UnsupportedOperationException(VIEW_WARNING_LOG_MESSAGE);
        }
        int deletedRecords = this.execute("DELETE FROM iceberg_tables WHERE catalog_name = ? AND table_namespace  = ? AND table_name = ? AND iceberg_type = 'VIEW'", this.catalogName, JdbcUtil.namespaceToString(identifier.namespace()), identifier.name());
        if (deletedRecords == 0) {
            LOG.info("Skipping drop, view does not exist: {}", (Object)identifier);
            return false;
        }
        LOG.info("Dropped view: {}", (Object)identifier);
        return true;
    }

    @Override
    public List<TableIdentifier> listViews(Namespace namespace) {
        if (this.schemaVersion != JdbcUtil.SchemaVersion.V1) {
            throw new UnsupportedOperationException(VIEW_WARNING_LOG_MESSAGE);
        }
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        return this.fetch(row -> JdbcUtil.stringToTableIdentifier(row.getString("table_namespace"), row.getString("table_name")), "SELECT * FROM iceberg_tables WHERE catalog_name = ? AND table_namespace = ? AND iceberg_type = 'VIEW'", this.catalogName, JdbcUtil.namespaceToString(namespace));
    }

    @Override
    public void renameView(TableIdentifier from, TableIdentifier to) {
        if (this.schemaVersion != JdbcUtil.SchemaVersion.V1) {
            throw new UnsupportedOperationException(VIEW_WARNING_LOG_MESSAGE);
        }
        if (from.equals(to)) {
            return;
        }
        if (!this.namespaceExists(to.namespace())) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", to.namespace());
        }
        if (!this.viewExists(from)) {
            throw new NoSuchViewException("View does not exist", new Object[0]);
        }
        if (this.tableExists(to)) {
            throw new AlreadyExistsException("Cannot rename %s to %s. Table already exists", from, to);
        }
        if (this.viewExists(to)) {
            throw new AlreadyExistsException("Cannot rename %s to %s. View already exists", from, to);
        }
        int updatedRecords = this.execute((SQLException err) -> {
            if (err instanceof SQLIntegrityConstraintViolationException || err.getMessage() != null && err.getMessage().contains("constraint failed")) {
                throw new AlreadyExistsException("Cannot rename %s to %s. View already exists", from, to);
            }
        }, "UPDATE iceberg_tables SET table_namespace = ?, table_name = ? WHERE catalog_name = ? AND table_namespace = ? AND table_name = ? AND iceberg_type = 'VIEW'", JdbcUtil.namespaceToString(to.namespace()), to.name(), this.catalogName, JdbcUtil.namespaceToString(from.namespace()), from.name());
        if (updatedRecords == 1) {
            LOG.info("Renamed view from {}, to {}", (Object)from, (Object)to);
        } else {
            if (updatedRecords == 0) {
                throw new NoSuchViewException("View does not exist: %s", from);
            }
            LOG.warn("Rename operation affected {} rows: the catalog view's primary key assumption has been violated", (Object)updatedRecords);
        }
    }

    private int execute(String sql, String ... args) {
        return this.execute((SQLException err) -> {}, sql, args);
    }

    private int execute(Consumer<SQLException> sqlErrorHandler, String sql, String ... args) {
        try {
            return this.connections.run(conn -> {
                PreparedStatement preparedStatement = conn.prepareStatement(sql);
                Throwable throwable = null;
                try {
                    for (int pos = 0; pos < args.length; ++pos) {
                        preparedStatement.setString(pos + 1, args[pos]);
                    }
                    Integer n = preparedStatement.executeUpdate();
                    return n;
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (preparedStatement != null) {
                        JdbcCatalog.$closeResource(throwable, preparedStatement);
                    }
                }
            });
        }
        catch (SQLException e) {
            sqlErrorHandler.accept(e);
            throw new UncheckedSQLException(e, "Failed to execute: %s", sql);
        }
        catch (InterruptedException e) {
            throw new UncheckedInterruptedException(e, "Interrupted in SQL command", new Object[0]);
        }
    }

    private <R> List<R> fetch(RowProducer<R> toRow, String sql, String ... args) {
        try {
            return this.connections.run(conn -> {
                ArrayList result = Lists.newArrayList();
                PreparedStatement preparedStatement = conn.prepareStatement(sql);
                Throwable throwable = null;
                try {
                    for (int pos = 0; pos < args.length; ++pos) {
                        preparedStatement.setString(pos + 1, args[pos]);
                    }
                    ResultSet rs = preparedStatement.executeQuery();
                    Throwable throwable2 = null;
                    try {
                        while (rs.next()) {
                            result.add(toRow.apply(rs));
                        }
                    }
                    catch (Throwable throwable3) {
                        throwable2 = throwable3;
                        throw throwable3;
                    }
                    finally {
                        if (rs != null) {
                            JdbcCatalog.$closeResource(throwable2, rs);
                        }
                    }
                }
                catch (Throwable throwable4) {
                    throwable = throwable4;
                    throw throwable4;
                }
                finally {
                    if (preparedStatement != null) {
                        JdbcCatalog.$closeResource(throwable, preparedStatement);
                    }
                }
                return result;
            });
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Failed to execute query: %s", sql);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted in SQL query", new Object[0]);
        }
    }

    private Map<String, String> fetchProperties(Namespace namespace) {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        String namespaceName = JdbcUtil.namespaceToString(namespace);
        List<Map.Entry> entries = this.fetch(row -> new AbstractMap.SimpleImmutableEntry<String, String>(row.getString("property_key"), row.getString("property_value")), "SELECT *  FROM iceberg_namespace_properties WHERE catalog_name = ? AND namespace = ? ", this.catalogName, namespaceName);
        return ImmutableMap.builder().putAll(entries).buildOrThrow();
    }

    private boolean insertProperties(Namespace namespace, Map<String, String> properties) {
        String namespaceName = JdbcUtil.namespaceToString(namespace);
        String[] args = (String[])properties.entrySet().stream().flatMap(entry -> Stream.of(this.catalogName, namespaceName, (String)entry.getKey(), (String)entry.getValue())).toArray(String[]::new);
        int insertedRecords = this.execute(JdbcUtil.insertPropertiesStatement(properties.size()), args);
        if (insertedRecords == properties.size()) {
            return true;
        }
        throw new IllegalStateException(String.format("Failed to insert: %d of %d succeeded", insertedRecords, properties.size()));
    }

    private boolean updateProperties(Namespace namespace, Map<String, String> properties) {
        String namespaceName = JdbcUtil.namespaceToString(namespace);
        Stream caseArgs = properties.entrySet().stream().flatMap(entry -> Stream.of((String)entry.getKey(), (String)entry.getValue()));
        Stream<String> whereArgs = Stream.concat(Stream.of(this.catalogName, namespaceName), properties.keySet().stream());
        String[] args = (String[])Stream.concat(caseArgs, whereArgs).toArray(String[]::new);
        int updatedRecords = this.execute(JdbcUtil.updatePropertiesStatement(properties.size()), args);
        if (updatedRecords == properties.size()) {
            return true;
        }
        throw new IllegalStateException(String.format("Failed to update: %d of %d succeeded", updatedRecords, properties.size()));
    }

    private boolean deleteProperties(Namespace namespace, Set<String> properties) {
        String namespaceName = JdbcUtil.namespaceToString(namespace);
        String[] args = (String[])Stream.concat(Stream.of(this.catalogName, namespaceName), properties.stream()).toArray(String[]::new);
        return this.execute(JdbcUtil.deletePropertiesStatement(properties), args) > 0;
    }

    @Override
    protected Map<String, String> properties() {
        return this.catalogProperties == null ? ImmutableMap.of() : this.catalogProperties;
    }

    @Override
    public Catalog.TableBuilder buildTable(TableIdentifier identifier, Schema schema) {
        return new ViewAwareTableBuilder(identifier, schema);
    }

    private static /* synthetic */ /* end resource */ void $closeResource(Throwable x0, AutoCloseable x1) {
        if (x0 != null) {
            try {
                x1.close();
            }
            catch (Throwable throwable) {
                x0.addSuppressed(throwable);
            }
        } else {
            x1.close();
        }
    }

    protected class ViewAwareTableBuilder
    extends BaseMetastoreCatalog.BaseMetastoreCatalogTableBuilder {
        private final TableIdentifier identifier;

        public ViewAwareTableBuilder(TableIdentifier identifier, Schema schema) {
            super(identifier, schema);
            this.identifier = identifier;
        }

        @Override
        public Transaction replaceTransaction() {
            if (JdbcCatalog.this.schemaVersion == JdbcUtil.SchemaVersion.V1 && JdbcCatalog.this.viewExists(this.identifier)) {
                throw new AlreadyExistsException("View with same name already exists: %s", this.identifier);
            }
            return super.replaceTransaction();
        }
    }

    @FunctionalInterface
    static interface RowProducer<R> {
        public R apply(ResultSet var1) throws SQLException;
    }
}

