/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.expressions;

import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Schema;
import org.apache.iceberg.Table;
import org.apache.iceberg.expressions.Binder;
import org.apache.iceberg.expressions.BoundPredicate;
import org.apache.iceberg.expressions.BoundReference;
import org.apache.iceberg.expressions.BoundTransform;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.ExpressionVisitors;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.expressions.Literal;
import org.apache.iceberg.expressions.Literals;
import org.apache.iceberg.expressions.NamedReference;
import org.apache.iceberg.expressions.Projections;
import org.apache.iceberg.expressions.Term;
import org.apache.iceberg.expressions.UnboundPredicate;
import org.apache.iceberg.expressions.UnboundTerm;
import org.apache.iceberg.expressions.UnboundTransform;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableSet;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.transforms.Transforms;
import org.apache.iceberg.types.Types;

public class ExpressionUtil {
    private static final Function<Object, Integer> HASH_FUNC = Transforms.bucket(Integer.MAX_VALUE).bind(Types.StringType.get());
    private static final OffsetDateTime EPOCH = Instant.ofEpochSecond(0L).atOffset(ZoneOffset.UTC);
    private static final long FIVE_MINUTES_IN_MICROS = TimeUnit.MINUTES.toMicros(5L);
    private static final long THREE_DAYS_IN_HOURS = TimeUnit.DAYS.toHours(3L);
    private static final long NINETY_DAYS_IN_HOURS = TimeUnit.DAYS.toHours(90L);
    private static final Pattern DATE = Pattern.compile("\\d{4}-\\d{2}-\\d{2}");
    private static final Pattern TIME = Pattern.compile("\\d{2}:\\d{2}(:\\d{2}(.\\d{1,6})?)?");
    private static final Pattern TIMESTAMP = Pattern.compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}(:\\d{2}(.\\d{1,6})?)?");
    private static final Pattern TIMESTAMPTZ = Pattern.compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}(:\\d{2}(.\\d{1,6})?)?([-+]\\d{2}:\\d{2}|Z)");
    static final int LONG_IN_PREDICATE_ABBREVIATION_THRESHOLD = 10;
    private static final int LONG_IN_PREDICATE_ABBREVIATION_MIN_GAIN = 5;

    private ExpressionUtil() {
    }

    public static Expression sanitize(Expression expr) {
        return ExpressionVisitors.visit(expr, new ExpressionSanitizer());
    }

    public static String toSanitizedString(Expression expr) {
        return ExpressionVisitors.visit(expr, new StringSanitizer());
    }

    public static Expression extractByIdInclusive(Expression expression, Schema schema, boolean caseSensitive, int ... ids) {
        PartitionSpec spec = ExpressionUtil.identitySpec(schema, ids);
        return Projections.inclusive(spec, caseSensitive).project(Expressions.rewriteNot(expression));
    }

    public static boolean equivalent(Expression left, Expression right, Types.StructType struct, boolean caseSensitive) {
        return Binder.bind(struct, Expressions.rewriteNot(left), caseSensitive).isEquivalentTo(Binder.bind(struct, Expressions.rewriteNot(right), caseSensitive));
    }

    public static boolean selectsPartitions(Expression expr, Table table, boolean caseSensitive) {
        return table.specs().values().stream().allMatch(spec -> ExpressionUtil.selectsPartitions(expr, spec, caseSensitive));
    }

    public static boolean selectsPartitions(Expression expr, PartitionSpec spec, boolean caseSensitive) {
        return ExpressionUtil.equivalent(Projections.inclusive(spec, caseSensitive).project(expr), Projections.strict(spec, caseSensitive).project(expr), spec.partitionType(), caseSensitive);
    }

    public static String describe(Term term) {
        if (term instanceof UnboundTransform) {
            return ((UnboundTransform)term).transform() + "(" + ExpressionUtil.describe(((UnboundTransform)term).ref()) + ")";
        }
        if (term instanceof BoundTransform) {
            return ((BoundTransform)term).transform() + "(" + ExpressionUtil.describe(((BoundTransform)term).ref()) + ")";
        }
        if (term instanceof NamedReference) {
            return ((NamedReference)term).name();
        }
        if (term instanceof BoundReference) {
            return ((BoundReference)term).name();
        }
        throw new UnsupportedOperationException("Unsupported term: " + term);
    }

    private static <T> List<String> abbreviateValues(List<String> sanitizedValues) {
        ImmutableSet<String> distinctValues;
        if (sanitizedValues.size() >= 10 && (distinctValues = ImmutableSet.copyOf(sanitizedValues)).size() <= sanitizedValues.size() - 5) {
            ArrayList<String> abbreviatedList = Lists.newArrayListWithCapacity(distinctValues.size() + 1);
            abbreviatedList.addAll(distinctValues);
            abbreviatedList.add(String.format("... (%d values hidden, %d in total)", sanitizedValues.size() - distinctValues.size(), sanitizedValues.size()));
            return abbreviatedList;
        }
        return sanitizedValues;
    }

    private static String sanitize(Literal<?> literal, long now, int today) {
        if (literal instanceof Literals.StringLiteral) {
            return ExpressionUtil.sanitizeString((CharSequence)((Literals.StringLiteral)literal).value(), now, today);
        }
        if (literal instanceof Literals.DateLiteral) {
            return ExpressionUtil.sanitizeDate((Integer)((Literals.DateLiteral)literal).value(), today);
        }
        if (literal instanceof Literals.TimestampLiteral) {
            return ExpressionUtil.sanitizeTimestamp((Long)((Literals.TimestampLiteral)literal).value(), now);
        }
        if (literal instanceof Literals.TimeLiteral) {
            return "(time)";
        }
        if (literal instanceof Literals.IntegerLiteral) {
            return ExpressionUtil.sanitizeNumber((Number)((Literals.IntegerLiteral)literal).value(), "int");
        }
        if (literal instanceof Literals.LongLiteral) {
            return ExpressionUtil.sanitizeNumber((Number)((Literals.LongLiteral)literal).value(), "int");
        }
        if (literal instanceof Literals.FloatLiteral) {
            return ExpressionUtil.sanitizeNumber((Number)((Literals.FloatLiteral)literal).value(), "float");
        }
        if (literal instanceof Literals.DoubleLiteral) {
            return ExpressionUtil.sanitizeNumber((Number)((Literals.DoubleLiteral)literal).value(), "float");
        }
        return ExpressionUtil.sanitizeSimpleString(literal.value().toString());
    }

    private static String sanitizeDate(int days, int today) {
        String isPast = today > days ? "ago" : "from-now";
        int diff = Math.abs(today - days);
        if (diff == 0) {
            return "(date-today)";
        }
        if (diff < 90) {
            return "(date-" + diff + "-days-" + isPast + ")";
        }
        return "(date)";
    }

    private static String sanitizeTimestamp(long micros, long now) {
        String isPast = now > micros ? "ago" : "from-now";
        long diff = Math.abs(now - micros);
        if (diff < FIVE_MINUTES_IN_MICROS) {
            return "(timestamp-about-now)";
        }
        long hours = TimeUnit.MICROSECONDS.toHours(diff);
        if (hours <= THREE_DAYS_IN_HOURS) {
            return "(timestamp-" + hours + "-hours-" + isPast + ")";
        }
        if (hours < NINETY_DAYS_IN_HOURS) {
            long days = hours / 24L;
            return "(timestamp-" + days + "-days-" + isPast + ")";
        }
        return "(timestamp)";
    }

    private static String sanitizeNumber(Number value, String type) {
        int numDigits = 0.0 == value.doubleValue() ? 1 : (int)Math.log10(Math.abs(value.doubleValue())) + 1;
        return "(" + numDigits + "-digit-" + type + ")";
    }

    private static String sanitizeString(CharSequence value, long now, int today) {
        try {
            if (DATE.matcher(value).matches()) {
                Literal date = Literal.of(value).to(Types.DateType.get());
                return ExpressionUtil.sanitizeDate((Integer)date.value(), today);
            }
            if (TIMESTAMP.matcher(value).matches()) {
                Literal ts = Literal.of(value).to(Types.TimestampType.withoutZone());
                return ExpressionUtil.sanitizeTimestamp((Long)ts.value(), now);
            }
            if (TIMESTAMPTZ.matcher(value).matches()) {
                Literal ts = Literal.of(value).to(Types.TimestampType.withZone());
                return ExpressionUtil.sanitizeTimestamp((Long)ts.value(), now);
            }
            if (TIME.matcher(value).matches()) {
                return "(time)";
            }
            return ExpressionUtil.sanitizeSimpleString(value);
        }
        catch (Exception ex) {
            return ExpressionUtil.sanitizeSimpleString(value);
        }
    }

    private static String sanitizeSimpleString(CharSequence value) {
        return String.format("(hash-%08x)", HASH_FUNC.apply(value));
    }

    private static PartitionSpec identitySpec(Schema schema, int ... ids) {
        PartitionSpec.Builder specBuilder = PartitionSpec.builderFor(schema);
        for (int id : ids) {
            specBuilder.identity(schema.findColumnName(id));
        }
        return specBuilder.build();
    }

    private static class StringSanitizer
    extends ExpressionVisitors.ExpressionVisitor<String> {
        private final long nowMicros;
        private final int today;

        private StringSanitizer() {
            long nowMillis = System.currentTimeMillis();
            OffsetDateTime nowDateTime = Instant.ofEpochMilli(nowMillis).atOffset(ZoneOffset.UTC);
            this.nowMicros = nowMillis * 1000L;
            this.today = (int)ChronoUnit.DAYS.between(EPOCH, nowDateTime);
        }

        @Override
        public String alwaysTrue() {
            return "true";
        }

        @Override
        public String alwaysFalse() {
            return "false";
        }

        @Override
        public String not(String result) {
            return "NOT (" + result + ")";
        }

        @Override
        public String and(String leftResult, String rightResult) {
            return "(" + leftResult + " AND " + rightResult + ")";
        }

        @Override
        public String or(String leftResult, String rightResult) {
            return "(" + leftResult + " OR " + rightResult + ")";
        }

        @Override
        public <T> String predicate(BoundPredicate<T> pred) {
            throw new UnsupportedOperationException("Cannot sanitize bound predicate: " + pred);
        }

        @Override
        public <T> String predicate(UnboundPredicate<T> pred) {
            String term = ExpressionUtil.describe(pred.term());
            switch (pred.op()) {
                case IS_NULL: {
                    return term + " IS NULL";
                }
                case NOT_NULL: {
                    return term + " IS NOT NULL";
                }
                case IS_NAN: {
                    return "is_nan(" + term + ")";
                }
                case NOT_NAN: {
                    return "not_nan(" + term + ")";
                }
                case LT: {
                    return term + " < " + ExpressionUtil.sanitize(pred.literal(), this.nowMicros, this.today);
                }
                case LT_EQ: {
                    return term + " <= " + ExpressionUtil.sanitize(pred.literal(), this.nowMicros, this.today);
                }
                case GT: {
                    return term + " > " + ExpressionUtil.sanitize(pred.literal(), this.nowMicros, this.today);
                }
                case GT_EQ: {
                    return term + " >= " + ExpressionUtil.sanitize(pred.literal(), this.nowMicros, this.today);
                }
                case EQ: {
                    return term + " = " + ExpressionUtil.sanitize(pred.literal(), this.nowMicros, this.today);
                }
                case NOT_EQ: {
                    return term + " != " + ExpressionUtil.sanitize(pred.literal(), this.nowMicros, this.today);
                }
                case IN: {
                    return term + " IN " + ExpressionUtil.abbreviateValues(pred.literals().stream().map(lit -> ExpressionUtil.sanitize(lit, this.nowMicros, this.today)).collect(Collectors.toList())).stream().collect(Collectors.joining(", ", "(", ")"));
                }
                case NOT_IN: {
                    return term + " NOT IN " + ExpressionUtil.abbreviateValues(pred.literals().stream().map(lit -> ExpressionUtil.sanitize(lit, this.nowMicros, this.today)).collect(Collectors.toList())).stream().collect(Collectors.joining(", ", "(", ")"));
                }
                case STARTS_WITH: {
                    return term + " STARTS WITH " + ExpressionUtil.sanitize(pred.literal(), this.nowMicros, this.today);
                }
                case NOT_STARTS_WITH: {
                    return term + " NOT STARTS WITH " + ExpressionUtil.sanitize(pred.literal(), this.nowMicros, this.today);
                }
            }
            throw new UnsupportedOperationException("Cannot sanitize unsupported predicate type: " + (Object)((Object)pred.op()));
        }
    }

    private static class ExpressionSanitizer
    extends ExpressionVisitors.ExpressionVisitor<Expression> {
        private final long now;
        private final int today;

        private ExpressionSanitizer() {
            long nowMillis = System.currentTimeMillis();
            OffsetDateTime nowDateTime = Instant.ofEpochMilli(nowMillis).atOffset(ZoneOffset.UTC);
            this.now = nowMillis * 1000L;
            this.today = (int)ChronoUnit.DAYS.between(EPOCH, nowDateTime);
        }

        @Override
        public Expression alwaysTrue() {
            return Expressions.alwaysTrue();
        }

        @Override
        public Expression alwaysFalse() {
            return Expressions.alwaysFalse();
        }

        @Override
        public Expression not(Expression result) {
            return Expressions.not(result);
        }

        @Override
        public Expression and(Expression leftResult, Expression rightResult) {
            return Expressions.and(leftResult, rightResult);
        }

        @Override
        public Expression or(Expression leftResult, Expression rightResult) {
            return Expressions.or(leftResult, rightResult);
        }

        @Override
        public <T> Expression predicate(BoundPredicate<T> pred) {
            throw new UnsupportedOperationException("Cannot sanitize bound predicate: " + pred);
        }

        @Override
        public <T> Expression predicate(UnboundPredicate<T> pred) {
            switch (pred.op()) {
                case IS_NULL: 
                case NOT_NULL: 
                case IS_NAN: 
                case NOT_NAN: {
                    return pred;
                }
                case LT: 
                case LT_EQ: 
                case GT: 
                case GT_EQ: 
                case EQ: 
                case NOT_EQ: 
                case STARTS_WITH: 
                case NOT_STARTS_WITH: {
                    return new UnboundPredicate<String>(pred.op(), (UnboundTerm)pred.term(), ExpressionUtil.sanitize(pred.literal(), this.now, this.today));
                }
                case IN: 
                case NOT_IN: {
                    Iterable iter = () -> pred.literals().stream().map(lit -> ExpressionUtil.sanitize(lit, this.now, this.today)).iterator();
                    return new UnboundPredicate(pred.op(), (UnboundTerm)pred.term(), iter);
                }
            }
            throw new UnsupportedOperationException("Cannot sanitize unsupported predicate type: " + (Object)((Object)pred.op()));
        }
    }
}

