/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.orc;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.iceberg.Metrics;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.Schema;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.hadoop.HadoopOutputFile;
import org.apache.iceberg.io.FileAppender;
import org.apache.iceberg.io.OutputFile;
import org.apache.iceberg.orc.EstimateOrcAvgWidthVisitor;
import org.apache.iceberg.orc.ORC;
import org.apache.iceberg.orc.ORCSchemaUtil;
import org.apache.iceberg.orc.OrcMetrics;
import org.apache.iceberg.orc.OrcRowWriter;
import org.apache.iceberg.orc.OrcSchemaVisitor;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.shaded.org.apache.orc.OrcFile;
import org.apache.iceberg.shaded.org.apache.orc.Reader;
import org.apache.iceberg.shaded.org.apache.orc.StripeInformation;
import org.apache.iceberg.shaded.org.apache.orc.TypeDescription;
import org.apache.iceberg.shaded.org.apache.orc.Writer;
import org.apache.iceberg.shaded.org.apache.orc.storage.ql.exec.vector.VectorizedRowBatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OrcFileAppender<D>
implements FileAppender<D> {
    private static final Logger LOG = LoggerFactory.getLogger(OrcFileAppender.class);
    private final int batchSize;
    private final OutputFile file;
    private final Writer writer;
    private final VectorizedRowBatch batch;
    private final int avgRowByteSize;
    private final OrcRowWriter<D> valueWriter;
    private boolean isClosed = false;
    private final Configuration conf;
    private final MetricsConfig metricsConfig;

    OrcFileAppender(Schema schema, OutputFile file, BiFunction<Schema, TypeDescription, OrcRowWriter<?>> createWriterFunc, Configuration conf, Map<String, byte[]> metadata, int batchSize, MetricsConfig metricsConfig) {
        this.conf = conf;
        this.file = file;
        this.batchSize = batchSize;
        this.metricsConfig = metricsConfig;
        TypeDescription orcSchema = ORCSchemaUtil.convert(schema);
        this.avgRowByteSize = OrcSchemaVisitor.visitSchema(orcSchema, new EstimateOrcAvgWidthVisitor()).stream().reduce(Integer::sum).orElse(0);
        if (this.avgRowByteSize == 0) {
            LOG.warn("The average length of the rows appears to be zero.");
        }
        this.batch = orcSchema.createRowBatch(this.batchSize);
        OrcFile.WriterOptions options = OrcFile.writerOptions(conf).useUTCTimestamp(true);
        if (file instanceof HadoopOutputFile) {
            options.fileSystem(((HadoopOutputFile)file).getFileSystem());
        }
        options.setSchema(orcSchema);
        this.writer = OrcFileAppender.newOrcWriter(file, options, metadata);
        this.valueWriter = OrcFileAppender.newOrcRowWriter(schema, orcSchema, createWriterFunc);
    }

    @Override
    public void add(D datum) {
        try {
            this.valueWriter.write(datum, this.batch);
            if (this.batch.size == this.batchSize) {
                this.writer.addRowBatch(this.batch);
                this.batch.reset();
            }
        }
        catch (IOException ioe) {
            throw new UncheckedIOException(String.format("Problem writing to ORC file %s", this.file.location()), ioe);
        }
    }

    @Override
    public Metrics metrics() {
        Preconditions.checkState(this.isClosed, "Cannot return metrics while appending to an open file.");
        return OrcMetrics.fromWriter(this.writer, this.valueWriter.metrics(), this.metricsConfig);
    }

    @Override
    public long length() {
        if (this.isClosed) {
            return this.file.toInputFile().getLength();
        }
        long estimateMemory = this.writer.estimateMemory();
        long dataLength = 0L;
        try {
            List<StripeInformation> stripes = this.writer.getStripes();
            if (!stripes.isEmpty()) {
                StripeInformation stripeInformation = stripes.get(stripes.size() - 1);
                dataLength = stripeInformation != null ? stripeInformation.getOffset() + stripeInformation.getLength() : 0L;
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format("Can't get Stripe's length from the file writer with path: %s.", this.file.location()), e);
        }
        return (long)Math.ceil((double)dataLength + (double)(estimateMemory + (long)this.batch.size * (long)this.avgRowByteSize) * 0.2);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public List<Long> splitOffsets() {
        Preconditions.checkState(this.isClosed, "File is not yet closed");
        try (Reader reader = ORC.newFileReader(this.file.toInputFile(), this.conf);){
            List<StripeInformation> stripes = reader.getStripes();
            List<Long> list = Collections.unmodifiableList(Lists.transform(stripes, StripeInformation::getOffset));
            return list;
        }
        catch (IOException e) {
            throw new RuntimeIOException(e, "Can't close ORC reader %s", this.file.location());
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.isClosed) {
            try {
                if (this.batch.size > 0) {
                    this.writer.addRowBatch(this.batch);
                    this.batch.reset();
                }
            }
            finally {
                this.writer.close();
                this.isClosed = true;
            }
        }
    }

    private static Writer newOrcWriter(OutputFile file, OrcFile.WriterOptions options, Map<String, byte[]> metadata) {
        Writer writer;
        Path locPath = new Path(file.location());
        try {
            writer = OrcFile.createWriter(locPath, options);
        }
        catch (IOException ioe) {
            throw new RuntimeIOException(ioe, "Can't create file %s", locPath);
        }
        metadata.forEach((key, value) -> writer.addUserMetadata((String)key, ByteBuffer.wrap(value)));
        return writer;
    }

    private static <D> OrcRowWriter<D> newOrcRowWriter(Schema schema, TypeDescription orcSchema, BiFunction<Schema, TypeDescription, OrcRowWriter<?>> createWriterFunc) {
        return createWriterFunc.apply(schema, orcSchema);
    }
}

