/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import java.util.List;
import java.util.Map;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.ManageSnapshots;
import org.apache.iceberg.MergingSnapshotProducer;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.exceptions.CherrypickAncestorCommitException;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.util.PartitionSet;
import org.apache.iceberg.util.PropertyUtil;
import org.apache.iceberg.util.SnapshotUtil;
import org.apache.iceberg.util.WapUtil;

public class SnapshotManager
extends MergingSnapshotProducer<ManageSnapshots>
implements ManageSnapshots {
    private final Map<Integer, PartitionSpec> specsById;
    private SnapshotManagerOperation managerOperation = null;
    private Long targetSnapshotId = null;
    private String snapshotOperation = null;
    private Long requiredCurrentSnapshotId = null;
    private Long overwriteParentId = null;
    private PartitionSet replacedPartitions = null;

    SnapshotManager(String tableName, TableOperations ops) {
        super(tableName, ops);
        this.specsById = ops.current().specsById();
    }

    @Override
    protected ManageSnapshots self() {
        return this;
    }

    @Override
    protected String operation() {
        Preconditions.checkNotNull(this.snapshotOperation, "[BUG] Detected uninitialized operation");
        return this.snapshotOperation;
    }

    @Override
    public ManageSnapshots cherrypick(long snapshotId) {
        TableMetadata current = this.current();
        ValidationException.check(current.snapshot(snapshotId) != null, "Cannot cherry pick unknown snapshot id: %s", snapshotId);
        Snapshot cherryPickSnapshot = current.snapshot(snapshotId);
        if (cherryPickSnapshot.operation().equals("append")) {
            this.managerOperation = SnapshotManagerOperation.CHERRYPICK;
            this.targetSnapshotId = snapshotId;
            this.snapshotOperation = cherryPickSnapshot.operation();
            for (DataFile addedFile : cherryPickSnapshot.addedFiles()) {
                this.add(addedFile);
            }
            String wapId = WapUtil.validateWapPublish(current, this.targetSnapshotId);
            if (wapId != null) {
                this.set("published-wap-id", wapId);
            }
            this.set("source-snapshot-id", String.valueOf(this.targetSnapshotId));
        } else if (cherryPickSnapshot.operation().equals("overwrite") && PropertyUtil.propertyAsBoolean(cherryPickSnapshot.summary(), "replace-partitions", false)) {
            this.overwriteParentId = cherryPickSnapshot.parentId();
            ValidationException.check(this.overwriteParentId == null || SnapshotManager.isCurrentAncestor(current, this.overwriteParentId), "Cannot cherry-pick overwrite not based on an ancestor of the current state: %s", snapshotId);
            this.managerOperation = SnapshotManagerOperation.CHERRYPICK;
            this.targetSnapshotId = snapshotId;
            this.snapshotOperation = cherryPickSnapshot.operation();
            this.replacedPartitions = PartitionSet.create(this.specsById);
            this.failMissingDeletePaths();
            for (DataFile addedFile : cherryPickSnapshot.addedFiles()) {
                this.add(addedFile);
                this.replacedPartitions.add(addedFile.specId(), addedFile.partition());
            }
            for (DataFile deletedFile : cherryPickSnapshot.deletedFiles()) {
                this.delete(deletedFile);
            }
            String overwriteWapId = WapUtil.validateWapPublish(current, this.targetSnapshotId);
            if (overwriteWapId != null) {
                this.set("published-wap-id", overwriteWapId);
            }
            this.set("source-snapshot-id", String.valueOf(this.targetSnapshotId));
        } else {
            this.managerOperation = SnapshotManagerOperation.ROLLBACK;
            this.requiredCurrentSnapshotId = cherryPickSnapshot.parentId();
            this.targetSnapshotId = snapshotId;
            SnapshotManager.validateCurrentSnapshot(current, this.requiredCurrentSnapshotId);
        }
        return this;
    }

    @Override
    public ManageSnapshots setCurrentSnapshot(long snapshotId) {
        ValidationException.check(this.current().snapshot(snapshotId) != null, "Cannot roll back to unknown snapshot id: %s", snapshotId);
        this.managerOperation = SnapshotManagerOperation.ROLLBACK;
        this.targetSnapshotId = snapshotId;
        return this;
    }

    @Override
    public ManageSnapshots rollbackToTime(long timestampMillis) {
        Snapshot snapshot = SnapshotManager.findLatestAncestorOlderThan(this.current(), timestampMillis);
        Preconditions.checkArgument(snapshot != null, "Cannot roll back, no valid snapshot older than: %s", timestampMillis);
        this.managerOperation = SnapshotManagerOperation.ROLLBACK;
        this.targetSnapshotId = snapshot.snapshotId();
        return this;
    }

    @Override
    public ManageSnapshots rollbackTo(long snapshotId) {
        TableMetadata current = this.current();
        ValidationException.check(current.snapshot(snapshotId) != null, "Cannot roll back to unknown snapshot id: %s", snapshotId);
        ValidationException.check(SnapshotManager.isCurrentAncestor(current, snapshotId), "Cannot roll back to snapshot, not an ancestor of the current state: %s", snapshotId);
        return this.setCurrentSnapshot(snapshotId);
    }

    @Override
    public Object updateEvent() {
        if (this.targetSnapshotId == null) {
            return null;
        }
        switch (this.managerOperation) {
            case ROLLBACK: {
                return null;
            }
            case CHERRYPICK: {
                TableMetadata tableMetadata = this.refresh();
                long snapshotId = tableMetadata.currentSnapshot().snapshotId();
                if (this.targetSnapshotId == snapshotId) {
                    return null;
                }
                return super.updateEvent();
            }
        }
        throw new UnsupportedOperationException((Object)((Object)this.managerOperation) + " is not supported");
    }

    @Override
    protected void validate(TableMetadata base) {
        SnapshotManager.validateCurrentSnapshot(base, this.requiredCurrentSnapshotId);
        SnapshotManager.validateNonAncestor(base, this.targetSnapshotId);
        SnapshotManager.validateReplacedPartitions(base, this.overwriteParentId, this.replacedPartitions);
        WapUtil.validateWapPublish(base, this.targetSnapshotId);
    }

    @Override
    public Snapshot apply() {
        TableMetadata base = this.refresh();
        if (this.targetSnapshotId == null) {
            return base.currentSnapshot();
        }
        switch (this.managerOperation) {
            case CHERRYPICK: {
                if (base.snapshot(this.targetSnapshotId).parentId() != null && base.currentSnapshot().snapshotId() == base.snapshot(this.targetSnapshotId).parentId().longValue()) {
                    this.validate(base);
                    return base.snapshot(this.targetSnapshotId);
                }
                return super.apply();
            }
            case ROLLBACK: {
                return base.snapshot(this.targetSnapshotId);
            }
        }
        throw new ValidationException("Invalid SnapshotManagerOperation: only cherrypick, rollback are supported", new Object[0]);
    }

    private static void validateCurrentSnapshot(TableMetadata meta, Long requiredSnapshotId) {
        if (requiredSnapshotId != null && meta.currentSnapshot() != null) {
            ValidationException.check(meta.currentSnapshot().snapshotId() == requiredSnapshotId.longValue(), "Cannot fast-forward to non-append snapshot; current has changed: current=%s != required=%s", meta.currentSnapshot().snapshotId(), requiredSnapshotId);
        }
    }

    private static void validateNonAncestor(TableMetadata meta, long snapshotId) {
        if (SnapshotManager.isCurrentAncestor(meta, snapshotId)) {
            throw new CherrypickAncestorCommitException(snapshotId);
        }
        Long ancestorId = SnapshotManager.lookupAncestorBySourceSnapshot(meta, snapshotId);
        if (ancestorId != null) {
            throw new CherrypickAncestorCommitException(snapshotId, ancestorId);
        }
    }

    private static void validateReplacedPartitions(TableMetadata meta, Long parentId, PartitionSet replacedPartitions) {
        if (replacedPartitions != null && meta.currentSnapshot() != null) {
            ValidationException.check(parentId == null || SnapshotManager.isCurrentAncestor(meta, parentId), "Cannot cherry-pick overwrite, based on non-ancestor of the current state: %s", parentId);
            List<DataFile> newFiles = SnapshotUtil.newFiles(parentId, meta.currentSnapshot().snapshotId(), meta::snapshot);
            for (DataFile newFile : newFiles) {
                ValidationException.check(!replacedPartitions.contains(newFile.specId(), newFile.partition()), "Cannot cherry-pick replace partitions with changed partition: %s", newFile.partition());
            }
        }
    }

    private static Long lookupAncestorBySourceSnapshot(TableMetadata meta, long snapshotId) {
        String snapshotIdStr = String.valueOf(snapshotId);
        for (long ancestorId : SnapshotManager.currentAncestors(meta)) {
            Map<String, String> summary = meta.snapshot(ancestorId).summary();
            if (summary == null || !snapshotIdStr.equals(summary.get("source-snapshot-id"))) continue;
            return ancestorId;
        }
        return null;
    }

    private static Snapshot findLatestAncestorOlderThan(TableMetadata meta, long timestampMillis) {
        long snapshotTimestamp = 0L;
        Snapshot result = null;
        for (Long snapshotId : SnapshotManager.currentAncestors(meta)) {
            Snapshot snapshot = meta.snapshot(snapshotId);
            if (snapshot.timestampMillis() >= timestampMillis || snapshot.timestampMillis() <= snapshotTimestamp) continue;
            result = snapshot;
            snapshotTimestamp = snapshot.timestampMillis();
        }
        return result;
    }

    private static List<Long> currentAncestors(TableMetadata meta) {
        return SnapshotUtil.ancestorIds(meta.currentSnapshot(), meta::snapshot);
    }

    private static boolean isCurrentAncestor(TableMetadata meta, long snapshotId) {
        return SnapshotManager.currentAncestors(meta).contains(snapshotId);
    }

    private static enum SnapshotManagerOperation {
        CHERRYPICK,
        ROLLBACK;

    }
}

