/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.shaded.org.apache.parquet.schema;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.TimeUnit;
import org.apache.iceberg.shaded.org.apache.parquet.io.api.Binary;

public abstract class PrimitiveStringifier {
    private final String name;
    private static final String BINARY_NULL = "null";
    private static final String BINARY_HEXA_PREFIX = "0x";
    private static final String BINARY_INVALID = "<INVALID>";
    static final PrimitiveStringifier DEFAULT_STRINGIFIER = new BinaryStringifierBase("DEFAULT_STRINGIFIER"){
        private final char[] digits = new char[]{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

        @Override
        public String stringify(boolean value) {
            return Boolean.toString(value);
        }

        @Override
        public String stringify(int value) {
            return Integer.toString(value);
        }

        @Override
        public String stringify(long value) {
            return Long.toString(value);
        }

        @Override
        public String stringify(float value) {
            return Float.toString(value);
        }

        @Override
        public String stringify(double value) {
            return Double.toString(value);
        }

        @Override
        String stringifyNotNull(Binary value) {
            ByteBuffer buffer = value.toByteBuffer();
            StringBuilder builder = new StringBuilder(2 + buffer.remaining() * 2);
            builder.append(PrimitiveStringifier.BINARY_HEXA_PREFIX);
            int n = buffer.limit();
            for (int i = buffer.position(); i < n; ++i) {
                byte b = buffer.get(i);
                builder.append(this.digits[b >>> 4 & 0xF]);
                builder.append(this.digits[b & 0xF]);
            }
            return builder.toString();
        }
    };
    static final PrimitiveStringifier UNSIGNED_STRINGIFIER = new PrimitiveStringifier("UNSIGNED_STRINGIFIER"){
        private static final long INT_MASK = 0xFFFFFFFFL;

        @Override
        public String stringify(int value) {
            return Long.toString((long)value & 0xFFFFFFFFL);
        }

        @Override
        public String stringify(long value) {
            if (value == 0L) {
                return "0";
            }
            if (value > 0L) {
                return Long.toString(value);
            }
            char[] buf = new char[64];
            int i = buf.length;
            long top = value >>> 32;
            long bot = (value & 0xFFFFFFFFL) + (top % 10L << 32);
            top /= 10L;
            while (bot > 0L || top > 0L) {
                buf[--i] = Character.forDigit((int)(bot % 10L), 10);
                bot = bot / 10L + (top % 10L << 32);
                top /= 10L;
            }
            return new String(buf, i, buf.length - i);
        }
    };
    static final PrimitiveStringifier UTF8_STRINGIFIER = new BinaryStringifierBase("UTF8_STRINGIFIER"){

        @Override
        String stringifyNotNull(Binary value) {
            return value.toStringUsingUTF8();
        }
    };
    static final PrimitiveStringifier INTERVAL_STRINGIFIER = new BinaryStringifierBase("INTERVAL_STRINGIFIER"){

        @Override
        String stringifyNotNull(Binary value) {
            if (value.length() != 12) {
                return PrimitiveStringifier.BINARY_INVALID;
            }
            ByteBuffer buffer = value.toByteBuffer().order(ByteOrder.LITTLE_ENDIAN);
            int pos = buffer.position();
            String months = UNSIGNED_STRINGIFIER.stringify(buffer.getInt(pos));
            String days = UNSIGNED_STRINGIFIER.stringify(buffer.getInt(pos + 4));
            String millis = UNSIGNED_STRINGIFIER.stringify(buffer.getInt(pos + 8));
            return "interval(" + months + " months, " + days + " days, " + millis + " millis)";
        }
    };
    static final PrimitiveStringifier DATE_STRINGIFIER = new DateStringifier("DATE_STRINGIFIER", "yyyy-MM-dd"){

        @Override
        Instant getInstant(int value) {
            return Instant.ofEpochMilli(TimeUnit.DAYS.toMillis(value));
        }
    };
    static final PrimitiveStringifier TIMESTAMP_MILLIS_STRINGIFIER = new DateStringifier("TIMESTAMP_MILLIS_STRINGIFIER", "yyyy-MM-dd'T'HH:mm:ss.SSS"){

        @Override
        Instant getInstant(long value) {
            return Instant.ofEpochMilli(value);
        }
    };
    static final PrimitiveStringifier TIMESTAMP_MICROS_STRINGIFIER = new DateStringifier("TIMESTAMP_MICROS_STRINGIFIER", "yyyy-MM-dd'T'HH:mm:ss.SSSSSS"){

        @Override
        Instant getInstant(long value) {
            return Instant.ofEpochSecond(TimeUnit.MICROSECONDS.toSeconds(value), TimeUnit.MICROSECONDS.toNanos(value % TimeUnit.SECONDS.toMicros(1L)));
        }
    };
    static final PrimitiveStringifier TIMESTAMP_NANOS_STRINGIFIER = new DateStringifier("TIMESTAMP_NANOS_STRINGIFIER", "yyyy-MM-dd'T'HH:mm:ss.SSSSSSSSS"){

        @Override
        Instant getInstant(long value) {
            return Instant.ofEpochSecond(TimeUnit.NANOSECONDS.toSeconds(value), TimeUnit.NANOSECONDS.toNanos(value % TimeUnit.SECONDS.toNanos(1L)));
        }
    };
    static final PrimitiveStringifier TIMESTAMP_MILLIS_UTC_STRINGIFIER = new DateStringifier("TIMESTAMP_MILLIS_UTC_STRINGIFIER", "yyyy-MM-dd'T'HH:mm:ss.SSSZ"){

        @Override
        Instant getInstant(long value) {
            return Instant.ofEpochMilli(value);
        }
    };
    static final PrimitiveStringifier TIMESTAMP_MICROS_UTC_STRINGIFIER = new DateStringifier("TIMESTAMP_MICROS_UTC_STRINGIFIER", "yyyy-MM-dd'T'HH:mm:ss.SSSSSSZ"){

        @Override
        Instant getInstant(long value) {
            return Instant.ofEpochSecond(TimeUnit.MICROSECONDS.toSeconds(value), TimeUnit.MICROSECONDS.toNanos(value % TimeUnit.SECONDS.toMicros(1L)));
        }
    };
    static final PrimitiveStringifier TIMESTAMP_NANOS_UTC_STRINGIFIER = new DateStringifier("TIMESTAMP_NANOS_UTC_STRINGIFIER", "yyyy-MM-dd'T'HH:mm:ss.SSSSSSSSSZ"){

        @Override
        Instant getInstant(long value) {
            return Instant.ofEpochSecond(TimeUnit.NANOSECONDS.toSeconds(value), TimeUnit.NANOSECONDS.toNanos(value % TimeUnit.SECONDS.toNanos(1L)));
        }
    };
    static final PrimitiveStringifier TIME_STRINGIFIER = new TimeStringifier("TIME_STRINGIFIER", false){

        @Override
        public String stringify(int millis) {
            return this.toTimeString(millis, TimeUnit.MILLISECONDS);
        }

        @Override
        public String stringify(long micros) {
            return this.toTimeString(micros, TimeUnit.MICROSECONDS);
        }
    };
    static final PrimitiveStringifier TIME_NANOS_STRINGIFIER = new TimeStringifier("TIME_NANOS_STRINGIFIER", false){

        @Override
        public String stringify(long nanos) {
            return this.toTimeString(nanos, TimeUnit.NANOSECONDS);
        }
    };
    static final PrimitiveStringifier TIME_UTC_STRINGIFIER = new TimeStringifier("TIME_UTC_STRINGIFIER", true){

        @Override
        public String stringify(int millis) {
            return this.toTimeString(millis, TimeUnit.MILLISECONDS);
        }

        @Override
        public String stringify(long micros) {
            return this.toTimeString(micros, TimeUnit.MICROSECONDS);
        }
    };
    static final PrimitiveStringifier TIME_NANOS_UTC_STRINGIFIER = new TimeStringifier("TIME_NANOS_UTC_STRINGIFIER", true){

        @Override
        public String stringify(long nanos) {
            return this.toTimeString(nanos, TimeUnit.NANOSECONDS);
        }
    };

    private PrimitiveStringifier(String name) {
        this.name = name;
    }

    public final String toString() {
        return this.name;
    }

    public String stringify(boolean value) {
        throw new UnsupportedOperationException("stringify(boolean) was called on a non-boolean stringifier: " + this.toString());
    }

    public String stringify(int value) {
        throw new UnsupportedOperationException("stringify(int) was called on a non-int stringifier: " + this.toString());
    }

    public String stringify(long value) {
        throw new UnsupportedOperationException("stringify(long) was called on a non-long stringifier: " + this.toString());
    }

    public String stringify(float value) {
        throw new UnsupportedOperationException("stringify(float) was called on a non-float stringifier: " + this.toString());
    }

    public String stringify(double value) {
        throw new UnsupportedOperationException("stringify(double) was called on a non-double stringifier: " + this.toString());
    }

    public String stringify(Binary value) {
        throw new UnsupportedOperationException("stringify(Binary) was called on a non-Binary stringifier: " + this.toString());
    }

    static PrimitiveStringifier createDecimalStringifier(final int scale) {
        return new BinaryStringifierBase("DECIMAL_STRINGIFIER(scale: " + scale + ")"){

            @Override
            public String stringify(int value) {
                return this.stringifyWithScale(BigInteger.valueOf(value));
            }

            @Override
            public String stringify(long value) {
                return this.stringifyWithScale(BigInteger.valueOf(value));
            }

            @Override
            String stringifyNotNull(Binary value) {
                try {
                    return this.stringifyWithScale(new BigInteger(value.getBytesUnsafe()));
                }
                catch (NumberFormatException e) {
                    return PrimitiveStringifier.BINARY_INVALID;
                }
            }

            private String stringifyWithScale(BigInteger i) {
                return new BigDecimal(i, scale).toString();
            }
        };
    }

    private static abstract class TimeStringifier
    extends PrimitiveStringifier {
        private final boolean withZone;

        TimeStringifier(String name, boolean withZone) {
            super(name);
            this.withZone = withZone;
        }

        protected String toTimeString(long duration, TimeUnit unit) {
            String additionalFormat = unit == TimeUnit.MILLISECONDS ? "3d" : (unit == TimeUnit.MICROSECONDS ? "6d" : "9d");
            String timeZone = this.withZone ? "+0000" : "";
            String format = "%02d:%02d:%02d.%0" + additionalFormat + timeZone;
            return String.format(format, unit.toHours(duration), this.convert(duration, unit, TimeUnit.MINUTES, TimeUnit.HOURS), this.convert(duration, unit, TimeUnit.SECONDS, TimeUnit.MINUTES), this.convert(duration, unit, unit, TimeUnit.SECONDS));
        }

        protected long convert(long duration, TimeUnit from, TimeUnit to, TimeUnit higher) {
            return Math.abs(to.convert(duration, from) % to.convert(1L, higher));
        }
    }

    private static class DateStringifier
    extends PrimitiveStringifier {
        private final DateTimeFormatter formatter;

        private DateStringifier(String name, String format) {
            super(name);
            this.formatter = DateTimeFormatter.ofPattern(format).withZone(ZoneOffset.UTC);
        }

        @Override
        public String stringify(int value) {
            return this.toFormattedString(this.getInstant(value));
        }

        @Override
        public String stringify(long value) {
            return this.toFormattedString(this.getInstant(value));
        }

        private String toFormattedString(Instant instant) {
            return this.formatter.format(instant);
        }

        Instant getInstant(int value) {
            super.stringify(value);
            return null;
        }

        Instant getInstant(long value) {
            super.stringify(value);
            return null;
        }
    }

    static abstract class BinaryStringifierBase
    extends PrimitiveStringifier {
        private BinaryStringifierBase(String name) {
            super(name);
        }

        @Override
        public final String stringify(Binary value) {
            return value == null ? PrimitiveStringifier.BINARY_NULL : this.stringifyNotNull(value);
        }

        abstract String stringifyNotNull(Binary var1);
    }
}

