/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.mr;

import java.util.HashMap;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import org.apache.hadoop.conf.Configuration;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.PartitionSpecParser;
import org.apache.iceberg.Schema;
import org.apache.iceberg.SchemaParser;
import org.apache.iceberg.Table;
import org.apache.iceberg.catalog.Catalog;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.common.DynConstructors;
import org.apache.iceberg.exceptions.NoSuchNamespaceException;
import org.apache.iceberg.hadoop.HadoopCatalog;
import org.apache.iceberg.hadoop.HadoopTables;
import org.apache.iceberg.hive.HiveCatalog;
import org.apache.iceberg.hive.HiveCatalogs;
import org.apache.iceberg.mr.CatalogLoader;
import org.apache.iceberg.relocated.com.google.common.annotations.VisibleForTesting;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Catalogs {
    private static final Logger LOG = LoggerFactory.getLogger(Catalogs.class);
    private static final String HADOOP = "hadoop";
    private static final String HIVE = "hive";
    public static final String NAME = "name";
    public static final String LOCATION = "location";
    private static final Set<String> PROPERTIES_TO_REMOVE = ImmutableSet.of("iceberg.mr.table.schema", "iceberg.mr.table.partition.spec", "location", "name");

    private Catalogs() {
    }

    public static Table loadTable(Configuration conf) {
        return Catalogs.loadTable(conf, conf.get("iceberg.mr.table.identifier"), conf.get("iceberg.mr.table.location"));
    }

    public static Table loadTable(Configuration conf, Properties props) {
        return Catalogs.loadTable(conf, props.getProperty(NAME), props.getProperty(LOCATION));
    }

    private static Table loadTable(Configuration conf, String tableIdentifier, String tableLocation) {
        Optional<Catalog> catalog = Catalogs.loadCatalog(conf);
        if (catalog.isPresent()) {
            Preconditions.checkArgument(tableIdentifier != null, "Table identifier not set");
            return catalog.get().loadTable(TableIdentifier.parse(tableIdentifier));
        }
        Preconditions.checkArgument(tableLocation != null, "Table location not set");
        return new HadoopTables(conf).load(tableLocation);
    }

    public static Table createTable(Configuration conf, Properties props) {
        String schemaString = props.getProperty("iceberg.mr.table.schema");
        Preconditions.checkNotNull(schemaString, "Table schema not set");
        Schema schema = SchemaParser.fromJson(props.getProperty("iceberg.mr.table.schema"));
        String specString = props.getProperty("iceberg.mr.table.partition.spec");
        PartitionSpec spec = PartitionSpec.unpartitioned();
        if (specString != null) {
            spec = PartitionSpecParser.fromJson(schema, specString);
        }
        String location = props.getProperty(LOCATION);
        HashMap<String, String> map = new HashMap<String, String>(props.size());
        for (Object key : props.keySet()) {
            if (PROPERTIES_TO_REMOVE.contains(key)) continue;
            map.put(key.toString(), props.get(key).toString());
        }
        Optional<Catalog> catalog = Catalogs.loadCatalog(conf);
        if (catalog.isPresent()) {
            String name = props.getProperty(NAME);
            Preconditions.checkNotNull(name, "Table identifier not set");
            return catalog.get().createTable(TableIdentifier.parse(name), schema, spec, location, map);
        }
        Preconditions.checkNotNull(location, "Table location not set");
        return new HadoopTables(conf).create(schema, spec, map, location);
    }

    public static boolean dropTable(Configuration conf, Properties props) {
        String location = props.getProperty(LOCATION);
        Optional<Catalog> catalog = Catalogs.loadCatalog(conf);
        if (catalog.isPresent()) {
            String name = props.getProperty(NAME);
            Preconditions.checkNotNull(name, "Table identifier not set");
            return catalog.get().dropTable(TableIdentifier.parse(name));
        }
        Preconditions.checkNotNull(location, "Table location not set");
        return new HadoopTables(conf).dropTable(location);
    }

    public static boolean hiveCatalog(Configuration conf) {
        return HIVE.equalsIgnoreCase(conf.get("iceberg.mr.catalog"));
    }

    @VisibleForTesting
    static Optional<Catalog> loadCatalog(Configuration conf) {
        String catalogLoaderClass = conf.get("iceberg.mr.catalog.loader.class");
        if (catalogLoaderClass != null) {
            CatalogLoader loader = (CatalogLoader)DynConstructors.builder(CatalogLoader.class).impl(catalogLoaderClass, new Class[0]).build().newInstance(new Object[0]);
            Catalog catalog = loader.load(conf);
            LOG.info("Loaded catalog {} using {}", (Object)catalog, (Object)catalogLoaderClass);
            return Optional.of(catalog);
        }
        String catalogName = conf.get("iceberg.mr.catalog");
        if (catalogName != null) {
            switch (catalogName.toLowerCase()) {
                case "hadoop": {
                    String warehouseLocation = conf.get("iceberg.mr.catalog.hadoop.warehouse.location");
                    HadoopCatalog catalog = warehouseLocation != null ? new HadoopCatalog(conf, warehouseLocation) : new HadoopCatalog(conf);
                    LOG.info("Loaded Hadoop catalog {}", (Object)catalog);
                    return Optional.of(catalog);
                }
                case "hive": {
                    HiveCatalog catalog = HiveCatalogs.loadCatalog(conf);
                    LOG.info("Loaded Hive Metastore catalog {}", (Object)catalog);
                    return Optional.of(catalog);
                }
            }
            throw new NoSuchNamespaceException("Catalog %s is not supported.", catalogName);
        }
        LOG.info("Catalog is not configured");
        return Optional.empty();
    }
}

