/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigquery.storage.v1;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Logger;
import org.apache.iceberg.gcp.shaded.com.google.common.annotations.VisibleForTesting;

public class RequestProfiler {
    private static final Logger log = Logger.getLogger(RequestProfiler.class.getName());
    private static final int MAX_CACHED_REQUEST = 100000;
    private static final RequestProfiler REQUEST_PROFILER_SINGLETON = new RequestProfiler();
    private static final int DEFAULT_TOP_K = 20;
    private static int TOP_K = 20;
    private static final Duration DEFAULT_FLUSH_PERIOD;
    private static Duration FLUSH_PERIOD;
    private final Map<String, IndividualRequestProfiler> idToIndividualOperation = new ConcurrentHashMap<String, IndividualRequestProfiler>();
    private Thread flushThread;
    private boolean enableProfiiler = false;
    AtomicLong droppedOperationCount = new AtomicLong(0L);

    void startOperation(OperationName operationName, String requestUniqueId) {
        try {
            if (!this.enableProfiiler) {
                return;
            }
            if (!this.idToIndividualOperation.containsKey(requestUniqueId)) {
                if (this.idToIndividualOperation.size() > 100000) {
                    log.warning(String.format("startOperation is triggered for request_id: %s that's hasn't seen before, this is possible when we are recording too much ongoing requests. So far we has dropped %s operations.", requestUniqueId, this.droppedOperationCount));
                    this.droppedOperationCount.incrementAndGet();
                    return;
                }
                this.idToIndividualOperation.put(requestUniqueId, new IndividualRequestProfiler(requestUniqueId));
            }
            this.idToIndividualOperation.get(requestUniqueId).startOperation(operationName);
        }
        catch (Exception ex) {
            log.warning("Exception thrown request profiler ignored, this is suggesting faulty implementation of RequestProfiler, exception context: " + ex.toString());
        }
    }

    void endOperation(OperationName operationName, String requestUniqueId) {
        try {
            if (!this.enableProfiiler) {
                return;
            }
            if (!this.idToIndividualOperation.containsKey(requestUniqueId)) {
                log.warning(String.format("endOperation is triggered for request_id: %s that's hasn't seen before, this is possible when we are recording too much ongoing requests. So far we has dropped %s operations.", requestUniqueId, this.droppedOperationCount));
                return;
            }
            this.idToIndividualOperation.get(requestUniqueId).endOperation(operationName);
        }
        catch (Exception ex) {
            log.warning("Exception thrown request profiler ignored, this is suggesting faulty implementation of RequestProfiler, exception context: " + ex.toString());
        }
    }

    void flushAndPrintReport() {
        if (!this.enableProfiiler) {
            return;
        }
        log.info(this.flushAndGenerateReportText());
    }

    void startPeriodicalReportFlushing() {
        this.enableProfiiler = true;
        if (this.flushThread == null || !this.flushThread.isAlive()) {
            this.flushThread = new Thread(new Runnable(){

                @Override
                public void run() {
                    try {
                        while (true) {
                            try {
                                TimeUnit.MILLISECONDS.sleep(FLUSH_PERIOD.toMillis());
                            }
                            catch (InterruptedException e) {
                                log.warning("Flush report thread is interrupted by " + e.toString());
                                throw new RuntimeException(e);
                            }
                            RequestProfiler.this.flushAndPrintReport();
                        }
                    }
                    catch (Exception ex) {
                        log.warning("Exception thrown request profiler ignored, this is suggesting faulty implementation of RequestProfiler, exception context: " + ex.toString());
                        return;
                    }
                }
            });
            this.flushThread.start();
        }
    }

    String flushAndGenerateReportText() {
        RequestProfilerComparator comparator = new RequestProfilerComparator();
        PriorityQueue<IndividualRequestProfiler> minHeap = new PriorityQueue<IndividualRequestProfiler>(comparator);
        Iterator<Map.Entry<String, IndividualRequestProfiler>> iterator = this.idToIndividualOperation.entrySet().iterator();
        int finishedRequestCount = 0;
        while (iterator.hasNext()) {
            Map.Entry<String, IndividualRequestProfiler> individualRequestProfiler = iterator.next();
            if (!individualRequestProfiler.getValue().finalized) continue;
            ++finishedRequestCount;
            if (minHeap.size() < TOP_K || individualRequestProfiler.getValue().totalTime > minHeap.peek().totalTime) {
                minHeap.add(individualRequestProfiler.getValue());
            }
            if (minHeap.size() > TOP_K) {
                minHeap.poll();
            }
            iterator.remove();
        }
        String reportText = String.format("During the last %s milliseconds at system time %s, in total %s requests finished. Total dropped request is %s. The top %s long latency requests details report:\n", FLUSH_PERIOD.toMillis(), System.currentTimeMillis(), finishedRequestCount, this.droppedOperationCount.getAndSet(0L), TOP_K);
        if (minHeap.isEmpty()) {
            reportText = reportText + "-----------------------------\n";
            reportText = reportText + "\t0 requests finished during the last period.";
        } else {
            ArrayList<String> reportList = new ArrayList<String>();
            while (minHeap.size() > 0) {
                reportList.add("-----------------------------\n" + minHeap.poll().generateReport());
            }
            for (int i = 0; i < reportList.size(); ++i) {
                reportText = reportText + (String)reportList.get(reportList.size() - i - 1);
            }
        }
        return reportText;
    }

    public static void setTopKRequestsToLog(int topK) {
        TOP_K = topK;
    }

    public static void setReportPeriod(Duration flushPeriod) {
        FLUSH_PERIOD = flushPeriod;
    }

    @VisibleForTesting
    void enableProfiler() {
        this.enableProfiiler = true;
    }

    void internalDisableAndClearProfiler() {
        this.enableProfiiler = false;
        if (this.flushThread != null) {
            this.flushThread.interrupt();
        }
        this.idToIndividualOperation.clear();
        this.droppedOperationCount.set(0L);
        TOP_K = 20;
        FLUSH_PERIOD = DEFAULT_FLUSH_PERIOD;
    }

    public static void disableAndResetProfiler() {
        REQUEST_PROFILER_SINGLETON.internalDisableAndClearProfiler();
    }

    static {
        FLUSH_PERIOD = DEFAULT_FLUSH_PERIOD = Duration.ofMinutes(1L);
    }

    private static final class IndividualRequestProfiler {
        private final Map<OperationName, Queue<Long>> timeRecorderMap = new ConcurrentHashMap<OperationName, Queue<Long>>();
        private final List<IndividualOperation> finishedOperations = Collections.synchronizedList(new ArrayList());
        private final String requestUniqueId;
        private long totalTime;
        private boolean finalized;

        IndividualRequestProfiler(String requestUniqueId) {
            this.requestUniqueId = requestUniqueId;
        }

        void startOperation(OperationName operationName) {
            this.timeRecorderMap.putIfAbsent(operationName, new ConcurrentLinkedDeque());
            this.timeRecorderMap.get((Object)operationName).add(System.currentTimeMillis());
        }

        void endOperation(OperationName operationName) {
            if (!this.timeRecorderMap.containsKey((Object)operationName)) {
                String warningMessage = String.format("Operation %s ignored for request %s due to startOperation() is not called before calling endOperation().", new Object[]{operationName, this.requestUniqueId});
                log.warning(warningMessage);
                return;
            }
            if (this.timeRecorderMap.get((Object)operationName).isEmpty()) {
                String warningMessage = String.format("Operation %s ignored for request %s due to no previous startOperation() triggered for this operation", new Object[]{operationName, this.requestUniqueId});
                log.warning(warningMessage);
                return;
            }
            long startTime = this.timeRecorderMap.get((Object)operationName).poll();
            long endTime = System.currentTimeMillis();
            long totalTime = endTime - startTime;
            this.finishedOperations.add(new IndividualOperation(operationName, startTime, endTime, totalTime));
            if (operationName == OperationName.TOTAL_LATENCY) {
                this.finalized = true;
                this.totalTime = totalTime;
            }
        }

        String generateReport() {
            String message = "\tRequest uuid: " + this.requestUniqueId + " with total time " + this.totalTime + " milliseconds\n";
            for (int i = 0; i < this.finishedOperations.size(); ++i) {
                if (this.finishedOperations.get((int)i).operationName == OperationName.TOTAL_LATENCY) continue;
                message = message + "\t\t";
                message = message + this.finishedOperations.get(i).format();
                message = message + "\n";
            }
            return message;
        }

        private static final class IndividualOperation {
            OperationName operationName;
            long totalTime;
            long startTimestamp;
            long endTimestamp;

            IndividualOperation(OperationName operationName, long startTimestamp, long endTimestamp, long totalTime) {
                this.operationName = operationName;
                this.startTimestamp = startTimestamp;
                this.endTimestamp = endTimestamp;
                this.totalTime = totalTime;
            }

            String format() {
                return String.format("Operation name %s starts at: %s, ends at: %s, total time: %s milliseconds", this.operationName.operationName, this.startTimestamp, this.endTimestamp, this.totalTime);
            }
        }
    }

    static enum OperationName {
        TOTAL_LATENCY("append_request_total_latency"),
        JSON_TO_PROTO_CONVERSION("json_to_proto_conversion"),
        WAIT_QUEUE("wait_queue"),
        RETRY_BACKOFF("retry_backoff"),
        RESPONSE_LATENCY("response_latency"),
        WAIT_INFLIGHT_QUOTA("wait_inflight_quota");

        private final String operationName;

        private OperationName(String operationName) {
            this.operationName = operationName;
        }
    }

    private class RequestProfilerComparator
    implements Comparator<IndividualRequestProfiler> {
        private RequestProfilerComparator() {
        }

        @Override
        public int compare(IndividualRequestProfiler x, IndividualRequestProfiler y) {
            if (x.totalTime > y.totalTime) {
                return 1;
            }
            if (x.totalTime < y.totalTime) {
                return -1;
            }
            return 0;
        }
    }

    static class RequestProfilerHook {
        private boolean enableRequestProfiler = false;

        RequestProfilerHook(boolean enableRequestProfiler) {
            this.enableRequestProfiler = enableRequestProfiler;
        }

        void startOperation(OperationName operationName, String requestUniqueId) {
            if (this.enableRequestProfiler) {
                REQUEST_PROFILER_SINGLETON.startOperation(operationName, requestUniqueId);
            }
        }

        void endOperation(OperationName operationName, String requestUniqueId) {
            if (this.enableRequestProfiler) {
                REQUEST_PROFILER_SINGLETON.endOperation(operationName, requestUniqueId);
            }
        }

        void startPeriodicalReportFlushing() {
            if (this.enableRequestProfiler) {
                REQUEST_PROFILER_SINGLETON.startPeriodicalReportFlushing();
            }
        }

        String flushAndGenerateReportText() {
            return REQUEST_PROFILER_SINGLETON.flushAndGenerateReportText();
        }

        void enableProfiler() {
            REQUEST_PROFILER_SINGLETON.enableProfiler();
        }
    }
}

