/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.data.orc;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import org.apache.iceberg.orc.OrcValueWriter;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.shaded.org.apache.orc.storage.common.type.HiveDecimal;
import org.apache.iceberg.shaded.org.apache.orc.storage.ql.exec.vector.BytesColumnVector;
import org.apache.iceberg.shaded.org.apache.orc.storage.ql.exec.vector.ColumnVector;
import org.apache.iceberg.shaded.org.apache.orc.storage.ql.exec.vector.DecimalColumnVector;
import org.apache.iceberg.shaded.org.apache.orc.storage.ql.exec.vector.DoubleColumnVector;
import org.apache.iceberg.shaded.org.apache.orc.storage.ql.exec.vector.ListColumnVector;
import org.apache.iceberg.shaded.org.apache.orc.storage.ql.exec.vector.LongColumnVector;
import org.apache.iceberg.shaded.org.apache.orc.storage.ql.exec.vector.MapColumnVector;
import org.apache.iceberg.shaded.org.apache.orc.storage.ql.exec.vector.TimestampColumnVector;
import org.apache.iceberg.util.ByteBuffers;

public class GenericOrcWriters {
    private static final OffsetDateTime EPOCH = Instant.ofEpochSecond(0L).atOffset(ZoneOffset.UTC);
    private static final LocalDate EPOCH_DAY = EPOCH.toLocalDate();

    private GenericOrcWriters() {
    }

    public static OrcValueWriter<Boolean> booleans() {
        return BooleanWriter.INSTANCE;
    }

    public static OrcValueWriter<Byte> bytes() {
        return ByteWriter.INSTANCE;
    }

    public static OrcValueWriter<Short> shorts() {
        return ShortWriter.INSTANCE;
    }

    public static OrcValueWriter<Integer> ints() {
        return IntWriter.INSTANCE;
    }

    public static OrcValueWriter<LocalTime> times() {
        return TimeWriter.INSTANCE;
    }

    public static OrcValueWriter<Long> longs() {
        return LongWriter.INSTANCE;
    }

    public static OrcValueWriter<Float> floats() {
        return FloatWriter.INSTANCE;
    }

    public static OrcValueWriter<Double> doubles() {
        return DoubleWriter.INSTANCE;
    }

    public static OrcValueWriter<String> strings() {
        return StringWriter.INSTANCE;
    }

    public static OrcValueWriter<ByteBuffer> byteBuffers() {
        return ByteBufferWriter.INSTANCE;
    }

    public static OrcValueWriter<UUID> uuids() {
        return UUIDWriter.INSTANCE;
    }

    public static OrcValueWriter<byte[]> byteArrays() {
        return ByteArrayWriter.INSTANCE;
    }

    public static OrcValueWriter<LocalDate> dates() {
        return DateWriter.INSTANCE;
    }

    public static OrcValueWriter<OffsetDateTime> timestampTz() {
        return TimestampTzWriter.INSTANCE;
    }

    public static OrcValueWriter<LocalDateTime> timestamp() {
        return TimestampWriter.INSTANCE;
    }

    public static OrcValueWriter<BigDecimal> decimal(int precision, int scale) {
        if (precision <= 18) {
            return new Decimal18Writer(precision, scale);
        }
        if (precision <= 38) {
            return new Decimal38Writer(precision, scale);
        }
        throw new IllegalArgumentException("Invalid precision: " + precision);
    }

    public static <T> OrcValueWriter<List<T>> list(OrcValueWriter<T> element) {
        return new ListWriter<T>(element);
    }

    public static <K, V> OrcValueWriter<Map<K, V>> map(OrcValueWriter<K> key, OrcValueWriter<V> value) {
        return new MapWriter<K, V>(key, value);
    }

    private static class MapWriter<K, V>
    implements OrcValueWriter<Map<K, V>> {
        private final OrcValueWriter<K> keyWriter;
        private final OrcValueWriter<V> valueWriter;

        MapWriter(OrcValueWriter<K> keyWriter, OrcValueWriter<V> valueWriter) {
            this.keyWriter = keyWriter;
            this.valueWriter = valueWriter;
        }

        @Override
        public Class<?> getJavaClass() {
            return Map.class;
        }

        @Override
        public void nonNullWrite(int rowId, Map<K, V> map, ColumnVector output) {
            ArrayList<K> keys = Lists.newArrayListWithExpectedSize(map.size());
            ArrayList<V> values = Lists.newArrayListWithExpectedSize(map.size());
            for (Map.Entry<K, V> entry : map.entrySet()) {
                keys.add(entry.getKey());
                values.add(entry.getValue());
            }
            MapColumnVector cv = (MapColumnVector)output;
            cv.lengths[rowId] = map.size();
            cv.offsets[rowId] = cv.childCount;
            cv.childCount = (int)((long)cv.childCount + cv.lengths[rowId]);
            cv.keys.ensureSize(cv.childCount, true);
            cv.values.ensureSize(cv.childCount, true);
            int e = 0;
            while ((long)e < cv.lengths[rowId]) {
                int pos = (int)((long)e + cv.offsets[rowId]);
                this.keyWriter.write(pos, keys.get(e), cv.keys);
                this.valueWriter.write(pos, values.get(e), cv.values);
                ++e;
            }
        }
    }

    private static class ListWriter<T>
    implements OrcValueWriter<List<T>> {
        private final OrcValueWriter<T> element;

        ListWriter(OrcValueWriter<T> element) {
            this.element = element;
        }

        @Override
        public Class<?> getJavaClass() {
            return List.class;
        }

        @Override
        public void nonNullWrite(int rowId, List<T> value, ColumnVector output) {
            ListColumnVector cv = (ListColumnVector)output;
            cv.lengths[rowId] = value.size();
            cv.offsets[rowId] = cv.childCount;
            cv.childCount = (int)((long)cv.childCount + cv.lengths[rowId]);
            cv.child.ensureSize(cv.childCount, true);
            int e = 0;
            while ((long)e < cv.lengths[rowId]) {
                this.element.write((int)((long)e + cv.offsets[rowId]), value.get(e), cv.child);
                ++e;
            }
        }
    }

    private static class Decimal38Writer
    implements OrcValueWriter<BigDecimal> {
        private final int precision;
        private final int scale;

        Decimal38Writer(int precision, int scale) {
            this.precision = precision;
            this.scale = scale;
        }

        @Override
        public Class<BigDecimal> getJavaClass() {
            return BigDecimal.class;
        }

        @Override
        public void nonNullWrite(int rowId, BigDecimal data, ColumnVector output) {
            Preconditions.checkArgument(data.scale() == this.scale, "Cannot write value as decimal(%s,%s), wrong scale: %s", (Object)this.precision, (Object)this.scale, (Object)data);
            Preconditions.checkArgument(data.precision() <= this.precision, "Cannot write value as decimal(%s,%s), invalid precision: %s", (Object)this.precision, (Object)this.scale, (Object)data);
            ((DecimalColumnVector)output).vector[rowId].set(HiveDecimal.create(data, false));
        }
    }

    private static class Decimal18Writer
    implements OrcValueWriter<BigDecimal> {
        private final int precision;
        private final int scale;

        Decimal18Writer(int precision, int scale) {
            this.precision = precision;
            this.scale = scale;
        }

        @Override
        public Class<BigDecimal> getJavaClass() {
            return BigDecimal.class;
        }

        @Override
        public void nonNullWrite(int rowId, BigDecimal data, ColumnVector output) {
            Preconditions.checkArgument(data.scale() == this.scale, "Cannot write value as decimal(%s,%s), wrong scale: %s", (Object)this.precision, (Object)this.scale, (Object)data);
            Preconditions.checkArgument(data.precision() <= this.precision, "Cannot write value as decimal(%s,%s), invalid precision: %s", (Object)this.precision, (Object)this.scale, (Object)data);
            ((DecimalColumnVector)output).vector[rowId].setFromLongAndScale(data.unscaledValue().longValueExact(), this.scale);
        }
    }

    private static class TimestampWriter
    implements OrcValueWriter<LocalDateTime> {
        private static final OrcValueWriter<LocalDateTime> INSTANCE = new TimestampWriter();

        private TimestampWriter() {
        }

        @Override
        public Class<LocalDateTime> getJavaClass() {
            return LocalDateTime.class;
        }

        @Override
        public void nonNullWrite(int rowId, LocalDateTime data, ColumnVector output) {
            TimestampColumnVector cv = (TimestampColumnVector)output;
            cv.setIsUTC(true);
            cv.time[rowId] = data.toInstant(ZoneOffset.UTC).toEpochMilli();
            cv.nanos[rowId] = data.getNano() / 1000 * 1000;
        }
    }

    private static class TimestampTzWriter
    implements OrcValueWriter<OffsetDateTime> {
        private static final OrcValueWriter<OffsetDateTime> INSTANCE = new TimestampTzWriter();

        private TimestampTzWriter() {
        }

        @Override
        public Class<OffsetDateTime> getJavaClass() {
            return OffsetDateTime.class;
        }

        @Override
        public void nonNullWrite(int rowId, OffsetDateTime data, ColumnVector output) {
            TimestampColumnVector cv = (TimestampColumnVector)output;
            cv.time[rowId] = data.toInstant().toEpochMilli();
            cv.nanos[rowId] = data.getNano() / 1000 * 1000;
        }
    }

    private static class DateWriter
    implements OrcValueWriter<LocalDate> {
        private static final OrcValueWriter<LocalDate> INSTANCE = new DateWriter();

        private DateWriter() {
        }

        @Override
        public Class<LocalDate> getJavaClass() {
            return LocalDate.class;
        }

        @Override
        public void nonNullWrite(int rowId, LocalDate data, ColumnVector output) {
            ((LongColumnVector)output).vector[rowId] = ChronoUnit.DAYS.between(EPOCH_DAY, data);
        }
    }

    private static class ByteArrayWriter
    implements OrcValueWriter<byte[]> {
        private static final OrcValueWriter<byte[]> INSTANCE = new ByteArrayWriter();

        private ByteArrayWriter() {
        }

        @Override
        public Class<byte[]> getJavaClass() {
            return byte[].class;
        }

        @Override
        public void nonNullWrite(int rowId, byte[] data, ColumnVector output) {
            ((BytesColumnVector)output).setRef(rowId, data, 0, data.length);
        }
    }

    private static class UUIDWriter
    implements OrcValueWriter<UUID> {
        private static final OrcValueWriter<UUID> INSTANCE = new UUIDWriter();

        private UUIDWriter() {
        }

        @Override
        public Class<UUID> getJavaClass() {
            return UUID.class;
        }

        @Override
        public void nonNullWrite(int rowId, UUID data, ColumnVector output) {
            ByteBuffer buffer = ByteBuffer.allocate(16);
            buffer.putLong(data.getMostSignificantBits());
            buffer.putLong(data.getLeastSignificantBits());
            ((BytesColumnVector)output).setRef(rowId, buffer.array(), 0, buffer.array().length);
        }
    }

    private static class ByteBufferWriter
    implements OrcValueWriter<ByteBuffer> {
        private static final OrcValueWriter<ByteBuffer> INSTANCE = new ByteBufferWriter();

        private ByteBufferWriter() {
        }

        @Override
        public Class<ByteBuffer> getJavaClass() {
            return ByteBuffer.class;
        }

        @Override
        public void nonNullWrite(int rowId, ByteBuffer data, ColumnVector output) {
            if (data.hasArray()) {
                ((BytesColumnVector)output).setRef(rowId, data.array(), data.arrayOffset() + data.position(), data.remaining());
            } else {
                byte[] rawData = ByteBuffers.toByteArray(data);
                ((BytesColumnVector)output).setRef(rowId, rawData, 0, rawData.length);
            }
        }
    }

    private static class StringWriter
    implements OrcValueWriter<String> {
        private static final OrcValueWriter<String> INSTANCE = new StringWriter();

        private StringWriter() {
        }

        @Override
        public Class<String> getJavaClass() {
            return String.class;
        }

        @Override
        public void nonNullWrite(int rowId, String data, ColumnVector output) {
            byte[] value = data.getBytes(StandardCharsets.UTF_8);
            ((BytesColumnVector)output).setRef(rowId, value, 0, value.length);
        }
    }

    private static class DoubleWriter
    implements OrcValueWriter<Double> {
        private static final OrcValueWriter<Double> INSTANCE = new DoubleWriter();

        private DoubleWriter() {
        }

        @Override
        public Class<Double> getJavaClass() {
            return Double.class;
        }

        @Override
        public void nonNullWrite(int rowId, Double data, ColumnVector output) {
            ((DoubleColumnVector)output).vector[rowId] = data;
        }
    }

    private static class FloatWriter
    implements OrcValueWriter<Float> {
        private static final OrcValueWriter<Float> INSTANCE = new FloatWriter();

        private FloatWriter() {
        }

        @Override
        public Class<Float> getJavaClass() {
            return Float.class;
        }

        @Override
        public void nonNullWrite(int rowId, Float data, ColumnVector output) {
            ((DoubleColumnVector)output).vector[rowId] = data.floatValue();
        }
    }

    private static class LongWriter
    implements OrcValueWriter<Long> {
        private static final OrcValueWriter<Long> INSTANCE = new LongWriter();

        private LongWriter() {
        }

        @Override
        public Class<Long> getJavaClass() {
            return Long.class;
        }

        @Override
        public void nonNullWrite(int rowId, Long data, ColumnVector output) {
            ((LongColumnVector)output).vector[rowId] = data;
        }
    }

    private static class TimeWriter
    implements OrcValueWriter<LocalTime> {
        private static final OrcValueWriter<LocalTime> INSTANCE = new TimeWriter();

        private TimeWriter() {
        }

        @Override
        public Class<LocalTime> getJavaClass() {
            return LocalTime.class;
        }

        @Override
        public void nonNullWrite(int rowId, LocalTime data, ColumnVector output) {
            ((LongColumnVector)output).vector[rowId] = data.toNanoOfDay() / 1000L;
        }
    }

    private static class IntWriter
    implements OrcValueWriter<Integer> {
        private static final OrcValueWriter<Integer> INSTANCE = new IntWriter();

        private IntWriter() {
        }

        @Override
        public Class<Integer> getJavaClass() {
            return Integer.class;
        }

        @Override
        public void nonNullWrite(int rowId, Integer data, ColumnVector output) {
            ((LongColumnVector)output).vector[rowId] = data.intValue();
        }
    }

    private static class ShortWriter
    implements OrcValueWriter<Short> {
        private static final OrcValueWriter<Short> INSTANCE = new ShortWriter();

        private ShortWriter() {
        }

        @Override
        public Class<Short> getJavaClass() {
            return Short.class;
        }

        @Override
        public void nonNullWrite(int rowId, Short data, ColumnVector output) {
            ((LongColumnVector)output).vector[rowId] = data.shortValue();
        }
    }

    private static class ByteWriter
    implements OrcValueWriter<Byte> {
        private static final OrcValueWriter<Byte> INSTANCE = new ByteWriter();

        private ByteWriter() {
        }

        @Override
        public Class<Byte> getJavaClass() {
            return Byte.class;
        }

        @Override
        public void nonNullWrite(int rowId, Byte data, ColumnVector output) {
            ((LongColumnVector)output).vector[rowId] = data.byteValue();
        }
    }

    private static class BooleanWriter
    implements OrcValueWriter<Boolean> {
        private static final OrcValueWriter<Boolean> INSTANCE = new BooleanWriter();

        private BooleanWriter() {
        }

        @Override
        public Class<Boolean> getJavaClass() {
            return Boolean.class;
        }

        @Override
        public void nonNullWrite(int rowId, Boolean data, ColumnVector output) {
            ((LongColumnVector)output).vector[rowId] = data != false ? 1L : 0L;
        }
    }
}

