/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.metastore.txn.retry;

import java.sql.SQLException;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.DatabaseProduct;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.conf.MetastoreConf;
import org.apache.hadoop.hive.metastore.txn.MetaWrapperException;
import org.apache.hadoop.hive.metastore.txn.retry.SqlRetryCallProperties;
import org.apache.hadoop.hive.metastore.txn.retry.SqlRetryFunction;
import org.apache.hadoop.hive.metastore.utils.StackThreadLocal;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.UncategorizedSQLException;

public class SqlRetryHandler {
    private static final Logger LOG = LoggerFactory.getLogger(SqlRetryHandler.class);
    private static final int ALLOWED_REPEATED_DEADLOCKS = 10;
    static final String MANUAL_RETRY = "ManualRetry";
    private final StackThreadLocal<Object> threadLocal = new StackThreadLocal();
    private final DatabaseProduct databaseProduct;
    private final long deadlockRetryInterval;
    private final long retryInterval;
    private final int retryLimit;
    private final Configuration conf;

    public static String getMessage(Exception ex) {
        StringBuilder sb = new StringBuilder();
        sb.append(ex.getMessage());
        if (ex instanceof SQLException) {
            SQLException sqlEx = (SQLException)ex;
            sb.append(" (SQLState=").append(sqlEx.getSQLState()).append(", ErrorCode=").append(sqlEx.getErrorCode()).append(")");
        }
        return sb.toString();
    }

    public SqlRetryHandler(Configuration conf, DatabaseProduct databaseProduct) {
        this.conf = conf;
        this.databaseProduct = databaseProduct;
        this.retryInterval = MetastoreConf.getTimeVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.HMS_HANDLER_INTERVAL, (TimeUnit)TimeUnit.MILLISECONDS);
        this.retryLimit = MetastoreConf.getIntVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.HMS_HANDLER_ATTEMPTS);
        this.deadlockRetryInterval = this.retryInterval / 10L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public <Result> Result executeWithRetry(SqlRetryCallProperties properties, SqlRetryFunction<Result> function) throws TException {
        Objects.requireNonNull(function, "RetryFunction<Result> cannot be null!");
        Objects.requireNonNull(properties, "RetryCallProperties cannot be null!");
        if (this.threadLocal.isSet() && properties.getRetryPropagation().canJoinContext()) {
            LOG.info("Already established retry-call detected, current call will join it. The passed SqlRetryCallProperties instance will be ignored, using the original one.");
            try {
                return function.execute();
            }
            catch (SQLException e) {
                throw new UncategorizedSQLException(null, null, e);
            }
        }
        if (!properties.getRetryPropagation().canCreateContext()) {
            throw new MetaException("The current RetryPropagation mode (" + properties.getRetryPropagation().name() + ") allows only to join an existing retry-call, but there is no retry-call upper in the callstack!");
        }
        if (StringUtils.isEmpty((CharSequence)properties.getCaller())) {
            properties.withCallerId(function.toString());
        }
        properties.setRetryCount(this.retryLimit).setDeadlockCount(10);
        try {
            if (properties.isLockInternally()) {
                this.databaseProduct.lockInternal();
            }
            this.threadLocal.set(new Object());
            Result Result2 = this.executeWithRetryInternal(properties, function);
            return Result2;
        }
        finally {
            this.threadLocal.unset();
            if (properties.isLockInternally()) {
                this.databaseProduct.unlockInternal();
            }
        }
    }

    private <Result> Result executeWithRetryInternal(SqlRetryCallProperties properties, SqlRetryFunction<Result> function) throws TException {
        LOG.debug("Running retry function:" + String.valueOf(properties));
        try {
            return function.execute();
        }
        catch (SQLException | DataAccessException e) {
            SQLException sqlEx = null;
            if (e.getCause() instanceof SQLException) {
                sqlEx = (SQLException)e.getCause();
            } else if (e instanceof SQLException) {
                sqlEx = (SQLException)e;
            }
            if (sqlEx != null) {
                if (this.checkDeadlock(sqlEx, properties)) {
                    properties.setDeadlockCount(properties.getDeadlockCount() - 1);
                    return this.executeWithRetryInternal(properties, function);
                }
                if (this.checkRetryable(sqlEx, properties)) {
                    properties.setRetryCount(properties.getRetryCount() - 1);
                    return this.executeWithRetryInternal(properties, function);
                }
            }
            LOG.error("Execution failed for caller {}", (Object)properties, (Object)e);
            throw new MetaException("Failed to execute function: " + properties.getCaller() + ", details:" + e.getMessage());
        }
        catch (MetaWrapperException e) {
            LOG.error("Execution failed for caller {}", (Object)properties, (Object)e.getCause());
            throw (MetaException)e.getCause();
        }
    }

    private boolean checkDeadlock(SQLException e, SqlRetryCallProperties properties) {
        if (this.databaseProduct.isDeadlock(e)) {
            if (properties.getDeadlockCount() > 0) {
                long waitInterval = this.deadlockRetryInterval * (long)(10 - properties.getDeadlockCount());
                LOG.warn("Deadlock detected in {}. Will wait {} ms try again up to {} times.", new Object[]{properties.getCaller(), waitInterval, properties.getDeadlockCount()});
                try {
                    Thread.sleep(waitInterval);
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                }
                return true;
            }
            LOG.error("Too many repeated deadlocks in {}, giving up.", (Object)properties.getCaller());
        }
        return false;
    }

    private boolean checkRetryable(SQLException e, SqlRetryCallProperties properties) {
        LOG.error("Execution error, checking if retry is possible", (Throwable)e);
        boolean retry = false;
        if (SqlRetryHandler.isRetryable(this.conf, e)) {
            retry = this.waitForRetry(properties.getCaller(), e.getMessage(), properties.getRetryCount());
        } else if (properties.isRetryOnDuplicateKey() && this.databaseProduct.isDuplicateKeyError(e)) {
            retry = this.waitForRetry(properties.getCaller(), e.getMessage(), properties.getRetryCount());
        } else {
            LOG.info("Non-retryable error in {} : {}", (Object)properties.getCaller(), (Object)SqlRetryHandler.getMessage(e));
        }
        return retry;
    }

    private boolean waitForRetry(String caller, String errMsg, int retryCount) {
        if (retryCount > 0) {
            LOG.warn("Retryable error detected in {}. Will wait {} ms and retry up to {} times. Error: {}", new Object[]{caller, this.retryInterval, retryCount, errMsg});
            try {
                Thread.sleep(this.retryInterval);
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
            }
            return true;
        }
        LOG.error("Fatal error in {}. Retry limit ({}) reached. Last error: {}", new Object[]{caller, this.retryLimit, errMsg});
        return false;
    }

    static boolean isRetryable(Configuration conf, Exception ex) {
        if (ex instanceof SQLException) {
            SQLException sqlException = (SQLException)ex;
            if (MANUAL_RETRY.equalsIgnoreCase(sqlException.getSQLState())) {
                return true;
            }
            if ("08S01".equalsIgnoreCase(sqlException.getSQLState())) {
                return true;
            }
            if ("ORA-08176".equalsIgnoreCase(sqlException.getSQLState()) || sqlException.getMessage().contains("consistent read failure; rollback data not available")) {
                return true;
            }
            String regex = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.TXN_RETRYABLE_SQLEX_REGEX);
            if (regex != null && !regex.isEmpty()) {
                String[] patterns = regex.split(",(?=\\S)");
                String message = SqlRetryHandler.getMessage(ex);
                for (String p : patterns) {
                    if (!Pattern.matches(p, message)) continue;
                    return true;
                }
            }
        }
        return false;
    }
}

