/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.client;

import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.MetaTableAccessor;
import org.apache.hadoop.hbase.RegionLocations;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.client.AdvancedScanResultConsumer;
import org.apache.hadoop.hbase.client.AsyncConnectionImpl;
import org.apache.hadoop.hbase.client.AsyncRegionLocator;
import org.apache.hadoop.hbase.client.ConnectionUtils;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.RegionLocateType;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.CollectionUtils;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
class AsyncNonMetaRegionLocator {
    private static final Logger LOG = LoggerFactory.getLogger(AsyncNonMetaRegionLocator.class);
    static final String MAX_CONCURRENT_LOCATE_REQUEST_PER_TABLE = "hbase.client.meta.max.concurrent.locate.per.table";
    private static final int DEFAULT_MAX_CONCURRENT_LOCATE_REQUEST_PER_TABLE = 8;
    private final AsyncConnectionImpl conn;
    private final int maxConcurrentLocateRequestPerTable;
    private final ConcurrentMap<TableName, TableCache> cache = new ConcurrentHashMap<TableName, TableCache>();

    AsyncNonMetaRegionLocator(AsyncConnectionImpl conn) {
        this.conn = conn;
        this.maxConcurrentLocateRequestPerTable = conn.getConfiguration().getInt(MAX_CONCURRENT_LOCATE_REQUEST_PER_TABLE, 8);
    }

    private TableCache getTableCache(TableName tableName) {
        return (TableCache)CollectionUtils.computeIfAbsent(this.cache, (Object)tableName, () -> new TableCache());
    }

    private void removeFromCache(HRegionLocation loc) {
        TableCache tableCache = (TableCache)this.cache.get(loc.getRegion().getTable());
        if (tableCache == null) {
            return;
        }
        tableCache.cache.computeIfPresent(loc.getRegion().getStartKey(), (k, oldLoc) -> {
            if (oldLoc.getSeqNum() > loc.getSeqNum() || !oldLoc.getServerName().equals((Object)loc.getServerName())) {
                return oldLoc;
            }
            return null;
        });
    }

    private boolean addToCache(TableCache tableCache, HRegionLocation loc) {
        byte[] startKey;
        HRegionLocation oldLoc;
        if (LOG.isTraceEnabled()) {
            LOG.trace("Try adding " + loc + " to cache");
        }
        if ((oldLoc = tableCache.cache.putIfAbsent(startKey = loc.getRegion().getStartKey(), loc)) == null) {
            return true;
        }
        if (oldLoc.getSeqNum() > loc.getSeqNum() || oldLoc.getServerName().equals((Object)loc.getServerName())) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Will not add " + loc + " to cache because the old value " + oldLoc + " is newer than us or has the same server name");
            }
            return false;
        }
        return loc == tableCache.cache.compute(startKey, (k, oldValue) -> {
            if (oldValue == null || oldValue.getSeqNum() <= loc.getSeqNum()) {
                return loc;
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace("Will not add " + loc + " to cache because the old value " + oldValue + " is newer than us or has the same server name." + " Maybe it is updated before we replace it");
            }
            return oldValue;
        });
    }

    @SuppressWarnings(value={"UPM_UNCALLED_PRIVATE_METHOD"}, justification="Called by lambda expression")
    private void addToCache(HRegionLocation loc) {
        this.addToCache(this.getTableCache(loc.getRegion().getTable()), loc);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Try adding " + loc + " to cache");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void complete(TableName tableName, LocateRequest req, HRegionLocation loc, Throwable error) {
        if (error != null) {
            LOG.warn("Failed to locate region in '" + tableName + "', row='" + Bytes.toStringBinary((byte[])req.row) + "', locateType=" + (Object)((Object)req.locateType), error);
        }
        Optional<Object> toSend = Optional.empty();
        TableCache tableCache = this.getTableCache(tableName);
        if (loc != null && !this.addToCache(tableCache, loc)) {
            TableCache tableCache2 = tableCache;
            synchronized (tableCache2) {
                tableCache.pendingRequests.remove(req);
                tableCache.clearCompletedRequests(Optional.empty());
                toSend = tableCache.getCandidate();
                toSend.ifPresent(r -> tableCache.send((LocateRequest)r));
            }
            toSend.ifPresent(r -> this.locateInMeta(tableName, (LocateRequest)r));
            return;
        }
        TableCache tableCache3 = tableCache;
        synchronized (tableCache3) {
            CompletableFuture<HRegionLocation> future;
            tableCache.pendingRequests.remove(req);
            if (error instanceof DoNotRetryIOException && (future = tableCache.allRequests.remove(req)) != null) {
                future.completeExceptionally(error);
            }
            tableCache.clearCompletedRequests(Optional.ofNullable(loc));
            toSend = tableCache.getCandidate();
            toSend.ifPresent(r -> tableCache.send((LocateRequest)r));
        }
        toSend.ifPresent(r -> this.locateInMeta(tableName, (LocateRequest)r));
    }

    private boolean onScanNext(TableName tableName, LocateRequest req, Result result) {
        RegionLocations locs = MetaTableAccessor.getRegionLocations(result);
        LOG.debug("The fetched location of '{}', row='{}', locateType={} is {}", new Object[]{tableName, Bytes.toStringBinary((byte[])req.row), req.locateType, locs});
        if (locs == null || locs.getDefaultRegionLocation() == null) {
            this.complete(tableName, req, null, new IOException(String.format("No location found for '%s', row='%s', locateType=%s", new Object[]{tableName, Bytes.toStringBinary((byte[])req.row), req.locateType})));
            return true;
        }
        HRegionLocation loc = locs.getDefaultRegionLocation();
        RegionInfo info = loc.getRegion();
        if (info == null) {
            this.complete(tableName, req, null, new IOException(String.format("HRegionInfo is null for '%s', row='%s', locateType=%s", new Object[]{tableName, Bytes.toStringBinary((byte[])req.row), req.locateType})));
            return true;
        }
        if (info.isSplitParent()) {
            return false;
        }
        if (loc.getServerName() == null) {
            this.complete(tableName, req, null, new IOException(String.format("No server address listed for region '%s', row='%s', locateType=%s", new Object[]{info.getRegionNameAsString(), Bytes.toStringBinary((byte[])req.row), req.locateType})));
            return true;
        }
        this.complete(tableName, req, loc, null);
        return true;
    }

    private HRegionLocation locateRowInCache(TableCache tableCache, TableName tableName, byte[] row) {
        Map.Entry entry = tableCache.cache.floorEntry(row);
        if (entry == null) {
            return null;
        }
        HRegionLocation loc = (HRegionLocation)entry.getValue();
        byte[] endKey = loc.getRegion().getEndKey();
        if (ConnectionUtils.isEmptyStopRow(endKey) || Bytes.compareTo((byte[])row, (byte[])endKey) < 0) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Found " + loc + " in cache for '" + tableName + "', row='" + Bytes.toStringBinary((byte[])row) + "', locateType=" + (Object)((Object)RegionLocateType.CURRENT));
            }
            return loc;
        }
        return null;
    }

    private HRegionLocation locateRowBeforeInCache(TableCache tableCache, TableName tableName, byte[] row) {
        Map.Entry entry;
        boolean isEmptyStopRow = ConnectionUtils.isEmptyStopRow(row);
        Map.Entry entry2 = entry = isEmptyStopRow ? tableCache.cache.lastEntry() : tableCache.cache.lowerEntry(row);
        if (entry == null) {
            return null;
        }
        HRegionLocation loc = (HRegionLocation)entry.getValue();
        if (ConnectionUtils.isEmptyStopRow(loc.getRegion().getEndKey()) || !isEmptyStopRow && Bytes.compareTo((byte[])loc.getRegion().getEndKey(), (byte[])row) >= 0) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Found " + loc + " in cache for '" + tableName + "', row='" + Bytes.toStringBinary((byte[])row) + "', locateType=" + (Object)((Object)RegionLocateType.BEFORE));
            }
            return loc;
        }
        return null;
    }

    private void locateInMeta(final TableName tableName, final LocateRequest req) {
        byte[] metaStartKey;
        if (LOG.isTraceEnabled()) {
            LOG.trace("Try locate '" + tableName + "', row='" + Bytes.toStringBinary((byte[])req.row) + "', locateType=" + (Object)((Object)req.locateType) + " in meta");
        }
        if (req.locateType.equals((Object)RegionLocateType.BEFORE)) {
            if (ConnectionUtils.isEmptyStopRow(req.row)) {
                byte[] binaryTableName = tableName.getName();
                metaStartKey = Arrays.copyOf(binaryTableName, binaryTableName.length + 1);
            } else {
                metaStartKey = RegionInfo.createRegionName(tableName, req.row, "00000000000000", false);
            }
        } else {
            metaStartKey = RegionInfo.createRegionName(tableName, req.row, "99999999999999", false);
        }
        byte[] metaStopKey = RegionInfo.createRegionName(tableName, HConstants.EMPTY_START_ROW, "", false);
        this.conn.getTable(TableName.META_TABLE_NAME).scan(new Scan().withStartRow(metaStartKey).withStopRow(metaStopKey, true).addFamily(HConstants.CATALOG_FAMILY).setReversed(true).setCaching(5).setReadType(Scan.ReadType.PREAD), new AdvancedScanResultConsumer(){
            private boolean completeNormally = false;
            private boolean tableNotFound = true;

            @Override
            public void onError(Throwable error) {
                AsyncNonMetaRegionLocator.this.complete(tableName, req, null, error);
            }

            @Override
            public void onComplete() {
                if (this.tableNotFound) {
                    AsyncNonMetaRegionLocator.this.complete(tableName, req, null, (Throwable)((Object)new TableNotFoundException(tableName)));
                } else if (!this.completeNormally) {
                    AsyncNonMetaRegionLocator.this.complete(tableName, req, null, new IOException("Unable to find region for " + Bytes.toStringBinary((byte[])req.row) + " in " + tableName));
                }
            }

            @Override
            public void onNext(Result[] results, AdvancedScanResultConsumer.ScanController controller) {
                for (Result result : results) {
                    this.tableNotFound = false;
                    if (!AsyncNonMetaRegionLocator.this.onScanNext(tableName, req, result)) continue;
                    this.completeNormally = true;
                    controller.terminate();
                    return;
                }
            }
        });
    }

    private HRegionLocation locateInCache(TableCache tableCache, TableName tableName, byte[] row, RegionLocateType locateType) {
        return locateType.equals((Object)RegionLocateType.BEFORE) ? this.locateRowBeforeInCache(tableCache, tableName, row) : this.locateRowInCache(tableCache, tableName, row);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableFuture<HRegionLocation> getRegionLocationInternal(TableName tableName, byte[] row, RegionLocateType locateType, boolean reload) {
        CompletableFuture<HRegionLocation> future;
        LocateRequest req;
        HRegionLocation loc;
        assert (!locateType.equals((Object)RegionLocateType.AFTER));
        TableCache tableCache = this.getTableCache(tableName);
        if (!reload && (loc = this.locateInCache(tableCache, tableName, row, locateType)) != null) {
            return CompletableFuture.completedFuture(loc);
        }
        boolean sendRequest = false;
        TableCache tableCache2 = tableCache;
        synchronized (tableCache2) {
            HRegionLocation loc2;
            if (!reload && (loc2 = this.locateInCache(tableCache, tableName, row, locateType)) != null) {
                return CompletableFuture.completedFuture(loc2);
            }
            req = new LocateRequest(row, locateType);
            future = tableCache.allRequests.get(req);
            if (future == null) {
                future = new CompletableFuture();
                tableCache.allRequests.put(req, future);
                if (tableCache.hasQuota(this.maxConcurrentLocateRequestPerTable) && !tableCache.isPending(req)) {
                    tableCache.send(req);
                    sendRequest = true;
                }
            }
        }
        if (sendRequest) {
            this.locateInMeta(tableName, req);
        }
        return future;
    }

    CompletableFuture<HRegionLocation> getRegionLocation(TableName tableName, byte[] row, RegionLocateType locateType, boolean reload) {
        if (locateType.equals((Object)RegionLocateType.BEFORE)) {
            return this.getRegionLocationInternal(tableName, row, locateType, reload);
        }
        if (locateType.equals((Object)RegionLocateType.AFTER)) {
            row = ConnectionUtils.createClosestRowAfter(row);
        }
        return this.getRegionLocationInternal(tableName, row, RegionLocateType.CURRENT, reload);
    }

    void updateCachedLocation(HRegionLocation loc, Throwable exception) {
        AsyncRegionLocator.updateCachedLocation(loc, exception, l -> {
            TableCache tableCache = (TableCache)this.cache.get(l.getRegion().getTable());
            if (tableCache == null) {
                return null;
            }
            return (HRegionLocation)tableCache.cache.get(l.getRegion().getStartKey());
        }, this::addToCache, this::removeFromCache);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void clearCache(TableName tableName) {
        TableCache tableCache = (TableCache)this.cache.remove(tableName);
        if (tableCache == null) {
            return;
        }
        TableCache tableCache2 = tableCache;
        synchronized (tableCache2) {
            if (!tableCache.allRequests.isEmpty()) {
                IOException error = new IOException("Cache cleared");
                tableCache.allRequests.values().forEach(f -> f.completeExceptionally(error));
            }
        }
    }

    private static final class TableCache {
        public final ConcurrentNavigableMap<byte[], HRegionLocation> cache = new ConcurrentSkipListMap<byte[], HRegionLocation>(Bytes.BYTES_COMPARATOR);
        public final Set<LocateRequest> pendingRequests = new HashSet<LocateRequest>();
        public final Map<LocateRequest, CompletableFuture<HRegionLocation>> allRequests = new LinkedHashMap<LocateRequest, CompletableFuture<HRegionLocation>>();

        private TableCache() {
        }

        public boolean hasQuota(int max) {
            return this.pendingRequests.size() < max;
        }

        public boolean isPending(LocateRequest req) {
            return this.pendingRequests.contains(req);
        }

        public void send(LocateRequest req) {
            this.pendingRequests.add(req);
        }

        public Optional<LocateRequest> getCandidate() {
            return this.allRequests.keySet().stream().filter(r -> !this.isPending((LocateRequest)r)).findFirst();
        }

        public void clearCompletedRequests(Optional<HRegionLocation> location) {
            Iterator<Map.Entry<LocateRequest, CompletableFuture<HRegionLocation>>> iter = this.allRequests.entrySet().iterator();
            while (iter.hasNext()) {
                Map.Entry<LocateRequest, CompletableFuture<HRegionLocation>> entry = iter.next();
                if (!this.tryComplete(entry.getKey(), entry.getValue(), location)) continue;
                iter.remove();
            }
        }

        private boolean tryComplete(LocateRequest req, CompletableFuture<HRegionLocation> future, Optional<HRegionLocation> location) {
            int c;
            if (future.isDone()) {
                return true;
            }
            if (!location.isPresent()) {
                return false;
            }
            HRegionLocation loc = location.get();
            boolean completed = req.locateType.equals((Object)RegionLocateType.BEFORE) ? (c = Bytes.compareTo((byte[])loc.getRegion().getEndKey(), (byte[])req.row)) == 0 || c > 0 && Bytes.compareTo((byte[])loc.getRegion().getStartKey(), (byte[])req.row) < 0 : loc.getRegion().containsRow(req.row);
            if (completed) {
                future.complete(loc);
                return true;
            }
            return false;
        }
    }

    private static final class LocateRequest {
        public final byte[] row;
        public final RegionLocateType locateType;

        public LocateRequest(byte[] row, RegionLocateType locateType) {
            this.row = row;
            this.locateType = locateType;
        }

        public int hashCode() {
            return Bytes.hashCode((byte[])this.row) ^ this.locateType.hashCode();
        }

        public boolean equals(Object obj) {
            if (obj == null || obj.getClass() != LocateRequest.class) {
                return false;
            }
            LocateRequest that = (LocateRequest)obj;
            return this.locateType.equals((Object)that.locateType) && Bytes.equals((byte[])this.row, (byte[])that.row);
        }
    }
}

