/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils.db.cache;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.hadoop.hdds.annotation.InterfaceAudience;
import org.apache.hadoop.hdds.annotation.InterfaceStability;
import org.apache.hadoop.hdds.utils.db.cache.CacheKey;
import org.apache.hadoop.hdds.utils.db.cache.CacheResult;
import org.apache.hadoop.hdds.utils.db.cache.CacheValue;
import org.apache.hadoop.hdds.utils.db.cache.EpochEntry;
import org.apache.hadoop.hdds.utils.db.cache.TableCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class TableCacheImpl<CACHEKEY extends CacheKey, CACHEVALUE extends CacheValue>
implements TableCache<CACHEKEY, CACHEVALUE> {
    public static final Logger LOG = LoggerFactory.getLogger(TableCacheImpl.class);
    private final Map<CACHEKEY, CACHEVALUE> cache;
    private final NavigableSet<EpochEntry<CACHEKEY>> epochEntries;
    private ExecutorService executorService;
    private CacheCleanupPolicy cleanupPolicy;

    public TableCacheImpl(CacheCleanupPolicy cleanupPolicy) {
        this.cache = cleanupPolicy == CacheCleanupPolicy.NEVER ? new ConcurrentSkipListMap<CACHEKEY, CACHEVALUE>() : new ConcurrentHashMap<CACHEKEY, CACHEVALUE>();
        this.epochEntries = new ConcurrentSkipListSet<EpochEntry<CACHEKEY>>();
        ThreadFactory build = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("PartialTableCache Cleanup Thread - %d").build();
        this.executorService = Executors.newSingleThreadExecutor(build);
        this.cleanupPolicy = cleanupPolicy;
    }

    @Override
    public CACHEVALUE get(CACHEKEY cachekey) {
        return (CACHEVALUE)((CacheValue)this.cache.get(cachekey));
    }

    @Override
    public void loadInitial(CACHEKEY cacheKey, CACHEVALUE cacheValue) {
        this.cache.put(cacheKey, cacheValue);
    }

    @Override
    public void put(CACHEKEY cacheKey, CACHEVALUE value) {
        this.cache.put(cacheKey, value);
        this.epochEntries.add(new EpochEntry<CACHEKEY>(((CacheValue)value).getEpoch(), cacheKey));
    }

    @Override
    public void cleanup(List<Long> epochs) {
        this.executorService.execute(() -> this.evictCache(epochs));
    }

    @Override
    public int size() {
        return this.cache.size();
    }

    @Override
    public Iterator<Map.Entry<CACHEKEY, CACHEVALUE>> iterator() {
        return this.cache.entrySet().iterator();
    }

    @VisibleForTesting
    protected void evictCache(List<Long> epochs) {
        AtomicBoolean removed = new AtomicBoolean();
        long lastEpoch = epochs.get(epochs.size() - 1);
        Iterator<EpochEntry<CACHEKEY>> iterator = this.epochEntries.iterator();
        while (iterator.hasNext()) {
            EpochEntry<CACHEKEY> currentEntry = iterator.next();
            CacheKey cachekey = (CacheKey)currentEntry.getCachekey();
            long currentEpoch = currentEntry.getEpoch();
            CacheValue cacheValue = this.cache.computeIfPresent(cachekey, (k, v) -> {
                if (this.cleanupPolicy == CacheCleanupPolicy.MANUAL) {
                    if (v.getEpoch() == currentEpoch && epochs.contains(v.getEpoch())) {
                        LOG.debug("CacheKey {} with epoch {} is removed from cache", k.getCacheKey(), (Object)currentEpoch);
                        iterator.remove();
                        removed.set(true);
                        return null;
                    }
                } else if (this.cleanupPolicy == CacheCleanupPolicy.NEVER && v.getEpoch() == currentEpoch && epochs.contains(v.getEpoch()) && v.getCacheValue() == null) {
                    LOG.debug("CacheKey {} with epoch {} is removed from cache", k.getCacheKey(), (Object)currentEpoch);
                    removed.set(true);
                    iterator.remove();
                    return null;
                }
                return v;
            });
            if (!removed.get() && cacheValue == null) {
                LOG.debug("CacheKey {} with epoch {} is removed from epochEntry for a key not existing in cache", cachekey.getCacheKey(), (Object)currentEpoch);
                iterator.remove();
            } else if (currentEpoch >= lastEpoch) break;
            removed.set(false);
        }
    }

    @Override
    public CacheResult<CACHEVALUE> lookup(CACHEKEY cachekey) {
        CacheValue cachevalue = (CacheValue)this.cache.get(cachekey);
        if (cachevalue == null) {
            if (this.cleanupPolicy == CacheCleanupPolicy.NEVER) {
                return new CacheResult<Object>(CacheResult.CacheStatus.NOT_EXIST, null);
            }
            return new CacheResult<Object>(CacheResult.CacheStatus.MAY_EXIST, null);
        }
        if (cachevalue.getCacheValue() != null) {
            return new CacheResult<CacheValue>(CacheResult.CacheStatus.EXISTS, cachevalue);
        }
        return new CacheResult<Object>(CacheResult.CacheStatus.NOT_EXIST, null);
    }

    @Override
    @VisibleForTesting
    public Set<EpochEntry<CACHEKEY>> getEpochEntrySet() {
        return this.epochEntries;
    }

    public static enum CacheCleanupPolicy {
        NEVER,
        MANUAL;

    }
}

