/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.data.management.dataset;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.time.Duration;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.regex.Pattern;
import org.apache.gobblin.data.management.dataset.DefaultFileSystemGlobFinder;
import org.apache.gobblin.data.management.dataset.SimpleFileSystemDataset;
import org.apache.gobblin.dataset.DatasetsFinder;
import org.apache.gobblin.dataset.FileSystemDataset;
import org.apache.gobblin.time.TimeIterator;
import org.apache.gobblin.util.PathUtils;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimePartitionGlobFinder
implements DatasetsFinder<FileSystemDataset> {
    private static final Logger log = LoggerFactory.getLogger(TimePartitionGlobFinder.class);
    private static final String CONF_PREFIX = "timePartitionGlobFinder.";
    public static final String PARTITION_PREFIX = "timePartitionGlobFinder.partitionPrefix";
    public static final String TIME_FORMAT = "timePartitionGlobFinder.timeFormat";
    public static final String ENABLE_VIRTUAL_PARTITION = "timePartitionGlobFinder.enableVirtualPartition";
    public static final String GRANULARITY = "timePartitionGlobFinder.granularity";
    public static final String TIME_ZONE = "timePartitionGlobFinder.timeZone";
    public static final String LOOKBACK_SPEC = "timePartitionGlobFinder.lookbackSpec";
    private static final String DEFAULT_TIME_ZONE = "America/Los_Angeles";
    private static final Pattern SUPPORTED_TIME_FORMAT = Pattern.compile("(yyyy/MM(/.*)*)|(yyyy-MM(-.*)*)");
    private final String datasetPattern;
    private final String datasetPartitionPattern;
    private final String partitionPrefix;
    private final DateTimeFormatter timeFormatter;
    private final boolean enableVirtualPartition;
    private final ZonedDateTime startTime;
    private final ZonedDateTime endTime;
    private final TimeIterator.Granularity granularity;
    private final Properties props;
    private final FileSystem fs;

    public TimePartitionGlobFinder(FileSystem fs, Properties properties) {
        this(fs, properties, ZonedDateTime.now(ZoneId.of(properties.getProperty(TIME_ZONE, DEFAULT_TIME_ZONE))));
    }

    @VisibleForTesting
    TimePartitionGlobFinder(FileSystem fs, Properties properties, ZonedDateTime curTime) {
        this.datasetPattern = properties.getProperty("gobblin.dataset.pattern");
        Path datasetPath = new Path(this.datasetPattern);
        this.partitionPrefix = properties.getProperty(PARTITION_PREFIX, "");
        String timeFormat = properties.getProperty(TIME_FORMAT).trim();
        Preconditions.checkState((boolean)TimePartitionGlobFinder.isTimeFormatSupported(timeFormat), (Object)String.format("Unsupported time format %s, expecting %s", timeFormat, SUPPORTED_TIME_FORMAT));
        this.timeFormatter = DateTimeFormatter.ofPattern(timeFormat);
        this.endTime = curTime;
        Duration lookback = Duration.parse(properties.getProperty(LOOKBACK_SPEC));
        this.startTime = this.endTime.minus(lookback);
        this.granularity = TimeIterator.Granularity.valueOf(properties.getProperty(GRANULARITY).toUpperCase());
        this.datasetPartitionPattern = new Path(datasetPath, this.partitionPrefix + TimePartitionGlobFinder.derivePartitionPattern(this.startTime, this.endTime, timeFormat)).toString();
        log.info("Dataset partition pattern is {}", (Object)this.datasetPartitionPattern);
        this.enableVirtualPartition = Boolean.valueOf(properties.getProperty(ENABLE_VIRTUAL_PARTITION, "false"));
        this.props = properties;
        this.fs = fs;
    }

    @VisibleForTesting
    static boolean isTimeFormatSupported(String timeFormat) {
        return SUPPORTED_TIME_FORMAT.matcher(timeFormat).matches();
    }

    @VisibleForTesting
    static String derivePartitionPattern(ZonedDateTime start, ZonedDateTime end, String timeFormat) {
        int startYear = start.getYear();
        int endYear = end.getYear();
        StringBuilder yearOptions = new StringBuilder("{" + startYear);
        TimePartitionGlobFinder.appendOptions(yearOptions, startYear + 1, endYear);
        yearOptions.append("}");
        StringBuilder monthOptions = TimePartitionGlobFinder.buildMonthOptions(start, end);
        StringBuilder pattern = new StringBuilder(yearOptions);
        if (timeFormat.contains("-")) {
            pattern.append("-");
            pattern.append((CharSequence)monthOptions);
            if (!monthOptions.toString().equals("*")) {
                pattern.append("*");
            }
        } else {
            pattern.append("/");
            pattern.append((CharSequence)monthOptions);
            String[] parts = timeFormat.split("/");
            for (int i = 2; i < parts.length; ++i) {
                pattern.append("/*");
            }
        }
        return pattern.toString();
    }

    private static StringBuilder buildMonthOptions(ZonedDateTime start, ZonedDateTime end) {
        int startMonth = start.getMonthValue();
        int endMonth = end.getMonthValue();
        int yearDiff = end.getYear() - start.getYear();
        if (yearDiff > 1 || yearDiff == 1 && endMonth >= startMonth) {
            return new StringBuilder("*");
        }
        StringBuilder monthOptions = new StringBuilder("{");
        if (startMonth < 10) {
            monthOptions.append("0");
        }
        monthOptions.append(startMonth);
        if (endMonth >= startMonth) {
            TimePartitionGlobFinder.appendOptions(monthOptions, startMonth + 1, endMonth);
        } else {
            TimePartitionGlobFinder.appendOptions(monthOptions, startMonth + 1, 12);
            TimePartitionGlobFinder.appendOptions(monthOptions, 1, endMonth);
        }
        monthOptions.append("}");
        return monthOptions;
    }

    private static void appendOptions(StringBuilder stringBuilder, int start, int end) {
        for (int i = start; i <= end; ++i) {
            stringBuilder.append(",");
            if (i < 10) {
                stringBuilder.append("0");
            }
            stringBuilder.append(i);
        }
    }

    public List<FileSystemDataset> findDatasets() throws IOException {
        try {
            List<FileSystemDataset> list = this.doFindDatasets();
            return list;
        }
        finally {
            this.props.setProperty("gobblin.dataset.pattern", this.datasetPattern);
        }
    }

    private List<FileSystemDataset> doFindDatasets() throws IOException {
        List<FileSystemDataset> datasets = this.findDatasets(this.datasetPattern);
        HashSet computedPartitions = new HashSet();
        datasets.forEach(dataset -> computedPartitions.addAll(this.computePartitions((FileSystemDataset)dataset)));
        ArrayList<FileSystemDataset> resultPartitions = new ArrayList<FileSystemDataset>(computedPartitions.size());
        List<FileSystemDataset> actualPartitions = this.findDatasets(this.datasetPartitionPattern);
        for (FileSystemDataset physicalPartition : actualPartitions) {
            String pathStr = physicalPartition.datasetRoot().toString();
            if (!computedPartitions.contains(pathStr)) continue;
            resultPartitions.add(physicalPartition);
            computedPartitions.remove(pathStr);
        }
        if (this.enableVirtualPartition) {
            computedPartitions.forEach(partition -> {
                log.info("Creating virtual partition {}", partition);
                resultPartitions.add(new SimpleFileSystemDataset(new Path(partition), true));
            });
        } else {
            log.info("Will not create virtual partitions");
        }
        return resultPartitions;
    }

    private Collection<String> computePartitions(FileSystemDataset dataset) {
        ArrayList<String> partitions = new ArrayList<String>();
        TimeIterator iterator = new TimeIterator(this.startTime, this.endTime, this.granularity);
        while (iterator.hasNext()) {
            partitions.add(new Path(dataset.datasetRoot(), this.partitionPrefix + this.timeFormatter.format(iterator.next())).toString());
        }
        return partitions;
    }

    private List<FileSystemDataset> findDatasets(String pattern) throws IOException {
        this.props.setProperty("gobblin.dataset.pattern", pattern);
        DefaultFileSystemGlobFinder datasetFinder = new DefaultFileSystemGlobFinder(this.fs, this.props);
        return datasetFinder.findDatasets();
    }

    public Path commonDatasetRoot() {
        return PathUtils.deepestNonGlobPath((Path)new Path(this.datasetPattern));
    }
}

