/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.data.management.copy.recovery;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.concurrent.TimeUnit;
import org.apache.gobblin.configuration.State;
import org.apache.gobblin.data.management.copy.CopyEntity;
import org.apache.gobblin.data.management.copy.CopySource;
import org.apache.gobblin.data.management.copy.CopyableFile;
import org.apache.gobblin.util.guid.Guid;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RecoveryHelper {
    private static final Logger log = LoggerFactory.getLogger(RecoveryHelper.class);
    public static final String PERSIST_DIR_KEY = "distcp.persist.dir";
    public static final String PERSIST_RETENTION_KEY = "distcp.persist.retention.hours";
    public static final int DEFAULT_PERSIST_RETENTION = 24;
    private final FileSystem fs;
    private final Optional<Path> persistDir;
    private final int retentionHours;

    public RecoveryHelper(FileSystem fs, State state) throws IOException {
        this.fs = fs;
        this.persistDir = RecoveryHelper.getPersistDir(state);
        this.retentionHours = state.getPropAsInt(PERSIST_RETENTION_KEY, 24);
    }

    public static Optional<Path> getPersistDir(State state) throws IOException {
        if (state.contains(PERSIST_DIR_KEY)) {
            return Optional.of((Object)new Path(state.getProp(PERSIST_DIR_KEY), UserGroupInformation.getCurrentUser().getShortUserName()));
        }
        return Optional.absent();
    }

    public boolean persistFile(State state, CopyableFile file, Path path) throws IOException {
        if (!this.persistDir.isPresent()) {
            return false;
        }
        String guid = RecoveryHelper.computeGuid(state, file);
        Path guidPath = new Path((Path)this.persistDir.get(), guid);
        if (!this.fs.exists(guidPath)) {
            this.fs.mkdirs(guidPath, new FsPermission(FsAction.ALL, FsAction.READ, FsAction.NONE));
        }
        Path targetPath = new Path(guidPath, RecoveryHelper.shortenPathName(file.getOrigin().getPath(), 250 - guid.length()));
        log.info(String.format("Persisting file %s with guid %s to location %s.", path, guid, targetPath));
        if (this.fs.rename(path, targetPath)) {
            this.fs.setTimes(targetPath, System.currentTimeMillis(), -1L);
            return true;
        }
        return false;
    }

    public Optional<FileStatus> findPersistedFile(State state, CopyEntity file, Predicate<FileStatus> filter) throws IOException {
        FileStatus[] statuses;
        if (!this.persistDir.isPresent() || !this.fs.exists((Path)this.persistDir.get())) {
            return Optional.absent();
        }
        Path guidPath = new Path((Path)this.persistDir.get(), RecoveryHelper.computeGuid(state, file));
        try {
            statuses = this.fs.listStatus(guidPath);
        }
        catch (FileNotFoundException e) {
            return Optional.absent();
        }
        for (FileStatus fileStatus : statuses) {
            if (!filter.apply((Object)fileStatus)) continue;
            return Optional.of((Object)fileStatus);
        }
        return Optional.absent();
    }

    public void purgeOldPersistedFile() throws IOException {
        if (!this.persistDir.isPresent() || !this.fs.exists((Path)this.persistDir.get())) {
            log.info("No persist directory to clean.");
            return;
        }
        long retentionMillis = TimeUnit.HOURS.toMillis(this.retentionHours);
        long now = System.currentTimeMillis();
        for (FileStatus fileStatus : this.fs.listStatus((Path)this.persistDir.get())) {
            if (now - fileStatus.getModificationTime() <= retentionMillis || this.fs.delete(fileStatus.getPath(), true)) continue;
            log.warn("Failed to delete path " + fileStatus.getPath());
        }
    }

    static String shortenPathName(Path path, int bytes) {
        String pathString = path.toUri().getPath();
        String replaced = pathString.replace("/", "_");
        if (replaced.length() <= bytes) {
            return replaced;
        }
        int bytesPerHalf = (bytes - 3) / 2;
        return replaced.substring(0, bytesPerHalf) + "..." + replaced.substring(replaced.length() - bytesPerHalf);
    }

    private static String computeGuid(State state, CopyEntity file) throws IOException {
        Optional<Guid> stateGuid = CopySource.getWorkUnitGuid(state);
        if (stateGuid.isPresent()) {
            return Guid.combine((Guid[])new Guid[]{file.guid(), (Guid)stateGuid.get()}).toString();
        }
        throw new IOException("State does not contain a guid.");
    }
}

