/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.compaction.verify;

import com.google.common.base.Splitter;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.gobblin.compaction.audit.AuditCountClient;
import org.apache.gobblin.compaction.audit.AuditCountClientFactory;
import org.apache.gobblin.compaction.parser.CompactionPathParser;
import org.apache.gobblin.compaction.verify.CompactionVerifier;
import org.apache.gobblin.configuration.State;
import org.apache.gobblin.dataset.FileSystemDataset;
import org.apache.gobblin.util.ClassAliasResolver;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionAuditCountVerifier
implements CompactionVerifier<FileSystemDataset> {
    private static final Logger log = LoggerFactory.getLogger(CompactionAuditCountVerifier.class);
    public static final String COMPACTION_COMPLETENESS_THRESHOLD = "compaction.completeness.threshold";
    public static final double DEFAULT_COMPACTION_COMPLETENESS_THRESHOLD = 0.99;
    public static final String PRODUCER_TIER = "producer.tier";
    public static final String ORIGIN_TIER = "origin.tier";
    public static final String GOBBLIN_TIER = "gobblin.tier";
    private Collection<String> referenceTiers;
    private Collection<String> originTiers;
    private String producerTier;
    private String gobblinTier;
    private double threshold;
    private final State state;
    private final AuditCountClient auditCountClient;

    public CompactionAuditCountVerifier(State state) {
        this(state, CompactionAuditCountVerifier.getClientFactory(state).createAuditCountClient(state));
    }

    public CompactionAuditCountVerifier(State state, AuditCountClient client) {
        this.auditCountClient = client;
        this.state = state;
        if (client != null) {
            this.threshold = state.getPropAsDouble(COMPACTION_COMPLETENESS_THRESHOLD, 0.99);
            this.producerTier = state.getProp(PRODUCER_TIER);
            this.gobblinTier = state.getProp(GOBBLIN_TIER);
            this.originTiers = Splitter.on((String)",").omitEmptyStrings().trimResults().splitToList((CharSequence)state.getProp(ORIGIN_TIER));
            this.referenceTiers = new HashSet<String>(this.originTiers);
            this.referenceTiers.add(this.producerTier);
        }
    }

    private static AuditCountClientFactory getClientFactory(State state) {
        if (!state.contains("audit.count.client.factory")) {
            return new EmptyAuditCountClientFactory();
        }
        try {
            String factoryName = state.getProp("audit.count.client.factory");
            ClassAliasResolver conditionClassAliasResolver = new ClassAliasResolver(AuditCountClientFactory.class);
            AuditCountClientFactory factory = (AuditCountClientFactory)conditionClassAliasResolver.resolveClass(factoryName).newInstance();
            return factory;
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public CompactionVerifier.Result verify(FileSystemDataset dataset) {
        if (this.auditCountClient == null) {
            log.debug("No audit count client specified, skipped");
            return new CompactionVerifier.Result(true, "");
        }
        CompactionPathParser.CompactionParserResult result = new CompactionPathParser(this.state).parse(dataset);
        DateTime startTime = result.getTime();
        DateTime endTime = startTime.plusHours(1);
        String datasetName = result.getDatasetName();
        try {
            Map<String, Long> countsByTier = this.auditCountClient.fetch(datasetName, startTime.getMillis(), endTime.getMillis());
            for (String tier : this.referenceTiers) {
                CompactionVerifier.Result rst = this.passed(datasetName, countsByTier, tier);
                if (!rst.isSuccessful()) continue;
                return new CompactionVerifier.Result(true, "");
            }
        }
        catch (IOException e) {
            return new CompactionVerifier.Result(false, ExceptionUtils.getFullStackTrace((Throwable)e));
        }
        return new CompactionVerifier.Result(false, String.format("%s data is not complete between %s and %s", datasetName, startTime, endTime));
    }

    private CompactionVerifier.Result passed(String datasetName, Map<String, Long> countsByTier, String referenceTier) {
        if (!countsByTier.containsKey(this.gobblinTier)) {
            log.info("Missing entry for dataset: " + datasetName + " in gobblin tier: " + this.gobblinTier + "; setting count to 0.");
        }
        if (!countsByTier.containsKey(referenceTier)) {
            log.info("Missing entry for dataset: " + datasetName + " in reference tier: " + referenceTier + "; setting count to 0.");
        }
        long refCount = countsByTier.getOrDefault(referenceTier, 0L);
        long gobblinCount = countsByTier.getOrDefault(this.gobblinTier, 0L);
        if (refCount == 0L) {
            return new CompactionVerifier.Result(true, "");
        }
        if ((double)gobblinCount / (double)refCount < this.threshold) {
            return new CompactionVerifier.Result(false, String.format("%s failed for %s : gobblin count = %d, %s count = %d (%f < threshold %f)", this.getName(), datasetName, gobblinCount, referenceTier, refCount, (double)gobblinCount / (double)refCount, this.threshold));
        }
        return new CompactionVerifier.Result(true, "");
    }

    @Override
    public String getName() {
        return this.getClass().getName();
    }

    @Override
    public boolean isRetriable() {
        return true;
    }

    private static class EmptyAuditCountClientFactory
    implements AuditCountClientFactory {
        private EmptyAuditCountClientFactory() {
        }

        @Override
        public AuditCountClient createAuditCountClient(State state) {
            return null;
        }
    }
}

