/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.aws;

import com.amazonaws.regions.Region;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.autoscaling.model.AutoScalingGroup;
import com.amazonaws.services.autoscaling.model.BlockDeviceMapping;
import com.amazonaws.services.autoscaling.model.InstanceMonitoring;
import com.amazonaws.services.autoscaling.model.Tag;
import com.amazonaws.services.autoscaling.model.TagDescription;
import com.amazonaws.services.ec2.model.AvailabilityZone;
import com.amazonaws.services.ec2.model.Instance;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Optional;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.eventbus.EventBus;
import com.google.common.io.Closer;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.mail.EmailException;
import org.apache.gobblin.annotation.Alpha;
import org.apache.gobblin.aws.AWSClusterSecurityManager;
import org.apache.gobblin.aws.AWSSdkClient;
import org.apache.gobblin.aws.AWSShutdownHandler;
import org.apache.gobblin.aws.CloudInitScriptBuilder;
import org.apache.gobblin.aws.GobblinAWSConfigurationKeys;
import org.apache.gobblin.aws.GobblinAWSUtils;
import org.apache.gobblin.cluster.GobblinClusterUtils;
import org.apache.gobblin.cluster.HelixMessageSubTypes;
import org.apache.gobblin.cluster.HelixUtils;
import org.apache.gobblin.configuration.State;
import org.apache.gobblin.util.ConfigUtils;
import org.apache.gobblin.util.EmailUtils;
import org.apache.helix.Criteria;
import org.apache.helix.HelixManager;
import org.apache.helix.HelixManagerFactory;
import org.apache.helix.InstanceType;
import org.apache.helix.messaging.AsyncCallback;
import org.apache.helix.model.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Alpha
public class GobblinAWSClusterLauncher {
    private static final Logger LOGGER = LoggerFactory.getLogger(GobblinAWSClusterLauncher.class);
    public static final String CLUSTER_NAME_ASG_TAG = "ClusterName";
    public static final String CLUSTER_ID_ASG_TAG = "ClusterId";
    public static final String ASG_TYPE_ASG_TAG = "AsgType";
    public static final String ASG_TYPE_MASTER = "master";
    public static final String ASG_TYPE_WORKERS = "workers";
    public static final String MASTER_ASG_NAME_PREFIX = "GobblinMasterASG_";
    public static final String MASTER_LAUNCH_CONFIG_NAME_PREFIX = "GobblinMasterLaunchConfig_";
    public static final String WORKERS_ASG_NAME_PREFIX = "GobblinWorkerASG_";
    public static final String WORKERS_LAUNCH_CONFIG_PREFIX = "GobblinWorkerLaunchConfig_";
    private final Config config;
    private final String zkConnectionString;
    private final String helixClusterName;
    private final HelixManager helixManager;
    private final EventBus eventBus = new EventBus(GobblinAWSClusterLauncher.class.getSimpleName());
    private final CountDownLatch countDownLatch = new CountDownLatch(1);
    private AWSClusterSecurityManager awsClusterSecurityManager;
    private AWSSdkClient awsSdkClient;
    private final Closer closer = Closer.create();
    private final String clusterName;
    private volatile Optional<String> clusterId = Optional.absent();
    private volatile boolean stopped = false;
    private final boolean emailNotificationOnShutdown;
    private final String awsRegion;
    private final String awsConfDir;
    private final String masterAmiId;
    private final String masterInstanceType;
    private final String masterJvmMemory;
    private final String workerAmiId;
    private final String workerInstanceType;
    private final String workerJvmMemory;
    private final Integer minWorkers;
    private final Integer maxWorkers;
    private final Integer desiredWorkers;
    private final Optional<String> masterJvmArgs;
    private final Optional<String> workerJvmArgs;
    private String masterPublicIp;
    private final String nfsParentDir;
    private final String masterJarsDir;
    private final String masterS3ConfUri;
    private final String masterS3ConfFiles;
    private final String masterS3JarsUri;
    private final String masterS3JarsFiles;
    private final String workerJarsDir;
    private final String workerS3ConfUri;
    private final String workerS3ConfFiles;
    private final String workerS3JarsUri;
    private final String workerS3JarsFiles;
    private final String sinkLogRootDir;
    private final String appWorkDir;
    private String masterLaunchConfigName;
    private String masterAutoScalingGroupName;
    private String workerLaunchConfigName;
    private String workerAutoScalingGroupName;
    private final Optional<String> gobblinVersion;

    public GobblinAWSClusterLauncher(Config config) throws IOException {
        this.config = config;
        this.zkConnectionString = config.getString("gobblin.cluster.zk.connection.string");
        LOGGER.info("Using ZooKeeper connection string: " + this.zkConnectionString);
        this.clusterName = ConfigUtils.getString((Config)config, (String)"gobblin.aws.cluster.name", (String)"gobblinApplication");
        this.helixClusterName = ConfigUtils.getString((Config)config, (String)"gobblin.cluster.helix.cluster.name", (String)this.clusterName);
        this.nfsParentDir = GobblinAWSUtils.appendSlash(ConfigUtils.getString((Config)config, (String)"gobblin.aws.nfs.parent.dir", (String)"/home/ec2-user/gobblinApplication/"));
        this.awsRegion = ConfigUtils.getString((Config)config, (String)"gobblin.aws.region", (String)"us-west-2");
        this.awsConfDir = GobblinAWSUtils.appendSlash(ConfigUtils.getString((Config)config, (String)"gobblin.aws.conf.dir", (String)(this.nfsParentDir + "cluster-conf")));
        this.masterAmiId = ConfigUtils.getString((Config)config, (String)"gobblin.aws.master.ami.id", (String)"ami-f303fb93");
        this.masterInstanceType = ConfigUtils.getString((Config)config, (String)"gobblin.aws.master.instance.type", (String)"m3-medium");
        this.masterJvmMemory = ConfigUtils.getString((Config)config, (String)"gobblin.aws.master.jvm.memory", (String)"3G");
        this.workerAmiId = ConfigUtils.getString((Config)config, (String)"gobblin.aws.worker.ami.id", (String)"ami-f303fb93");
        this.workerInstanceType = ConfigUtils.getString((Config)config, (String)"gobblin.aws.worker.instance.type", (String)"m3-medium");
        this.workerJvmMemory = ConfigUtils.getString((Config)config, (String)"gobblin.aws.worker.jvm.memory", (String)"3G");
        this.minWorkers = ConfigUtils.getInt((Config)config, (String)"gobblin.aws.min.workers", (Integer)2);
        this.maxWorkers = ConfigUtils.getInt((Config)config, (String)"gobblin.aws.max.workers", (Integer)4);
        this.desiredWorkers = ConfigUtils.getInt((Config)config, (String)"gobblin.aws.desired.workers", (Integer)2);
        this.masterJvmArgs = config.hasPath("gobblin.aws.master.jvm.args") ? Optional.of((Object)config.getString("gobblin.aws.master.jvm.args")) : Optional.absent();
        this.workerJvmArgs = config.hasPath("gobblin.aws.worker.jvm.args") ? Optional.of((Object)config.getString("gobblin.aws.worker.jvm.args")) : Optional.absent();
        this.masterJarsDir = GobblinAWSUtils.appendSlash(ConfigUtils.getString((Config)config, (String)"gobblin.aws.master.jars.dir", (String)(this.nfsParentDir + "gobblin-lib")));
        this.masterS3ConfUri = GobblinAWSUtils.appendSlash(ConfigUtils.getString((Config)config, (String)"gobblin.aws.master.s3.conf.uri", (String)"https://s3-region.amazonaws.com/s3bucket/gobblin-confs/cluster-conf/"));
        this.masterS3ConfFiles = ConfigUtils.getString((Config)config, (String)"gobblin.aws.master.s3.conf.files", (String)"application.conf,log4j-aws.properties,quartz.properties");
        this.masterS3JarsUri = ConfigUtils.getString((Config)config, (String)"gobblin.aws.master.s3.jars.uri", (String)"https://s3-us-west-2.amazonaws.com/gobblin-libs/latest-jars/");
        this.masterS3JarsFiles = ConfigUtils.getString((Config)config, (String)"gobblin.aws.master.s3.jars.files", (String)GobblinAWSConfigurationKeys.DEFAULT_MASTER_S3_JARS_FILES);
        this.workerJarsDir = GobblinAWSUtils.appendSlash(ConfigUtils.getString((Config)config, (String)"gobblin.aws.worker.jars.dir", (String)(this.nfsParentDir + "gobblin-lib")));
        this.workerS3ConfUri = GobblinAWSUtils.appendSlash(ConfigUtils.getString((Config)config, (String)"gobblin.aws.worker.s3.conf.uri", (String)"https://s3-region.amazonaws.com/s3bucket/gobblin-confs/cluster-conf/"));
        this.workerS3ConfFiles = ConfigUtils.getString((Config)config, (String)"gobblin.aws.worker.s3.conf.files", (String)"application.conf,log4j-aws.properties,quartz.properties");
        this.workerS3JarsUri = ConfigUtils.getString((Config)config, (String)"gobblin.aws.worker.s3.jars.uri", (String)"https://s3-us-west-2.amazonaws.com/gobblin-libs/latest-jars/");
        this.workerS3JarsFiles = ConfigUtils.getString((Config)config, (String)"gobblin.aws.worker.s3.jars.files", (String)GobblinAWSConfigurationKeys.DEFAULT_WORKER_S3_JARS_FILES);
        this.sinkLogRootDir = GobblinAWSUtils.appendSlash(ConfigUtils.getString((Config)config, (String)"gobblin.aws.logs.sink.root.dir", (String)(this.nfsParentDir + "logs")));
        this.appWorkDir = GobblinAWSUtils.appendSlash(ConfigUtils.getString((Config)config, (String)"gobblin.aws.work.dir", (String)(this.nfsParentDir + "work.dir")));
        this.emailNotificationOnShutdown = ConfigUtils.getBoolean((Config)config, (String)"gobblin.aws.email.notification.on.shutdown", (boolean)false);
        this.awsClusterSecurityManager = new AWSClusterSecurityManager(this.config);
        this.awsSdkClient = this.createAWSSdkClient();
        this.gobblinVersion = config.hasPath("gobblin.aws.version") ? Optional.of((Object)config.getString("gobblin.aws.version")) : Optional.absent();
        this.helixManager = HelixManagerFactory.getZKHelixManager((String)this.helixClusterName, (String)GobblinClusterUtils.getHostname(), (InstanceType)InstanceType.SPECTATOR, (String)this.zkConnectionString);
    }

    public void launch() throws IOException, InterruptedException {
        this.eventBus.register((Object)this);
        HelixUtils.createGobblinHelixCluster((String)this.zkConnectionString, (String)this.helixClusterName, (boolean)false);
        LOGGER.info("Created Helix cluster " + this.helixClusterName);
        this.connectHelixManager();
        this.clusterId = this.getClusterId();
        this.countDownLatch.await();
    }

    public synchronized void stop() throws IOException, TimeoutException {
        if (this.stopped) {
            return;
        }
        LOGGER.info("Stopping the " + GobblinAWSClusterLauncher.class.getSimpleName());
        try {
            if (this.clusterId.isPresent()) {
                this.sendShutdownRequest();
            }
            this.disconnectHelixManager();
        }
        finally {
            try {
                if (this.clusterId.isPresent()) {
                    this.cleanUpClusterWorkDirectory((String)this.clusterId.get());
                }
            }
            finally {
                this.closer.close();
            }
        }
        this.countDownLatch.countDown();
        this.stopped = true;
    }

    @VisibleForTesting
    void connectHelixManager() {
        try {
            this.helixManager.connect();
        }
        catch (Exception e) {
            LOGGER.error("HelixManager failed to connect", (Throwable)e);
            throw Throwables.propagate((Throwable)e);
        }
    }

    @VisibleForTesting
    void disconnectHelixManager() {
        if (this.helixManager.isConnected()) {
            this.helixManager.disconnect();
        }
    }

    @VisibleForTesting
    AWSSdkClient createAWSSdkClient() {
        return new AWSSdkClient(this.awsClusterSecurityManager, Region.getRegion((Regions)Regions.fromName((String)this.awsRegion)));
    }

    private Optional<String> getClusterId() throws IOException {
        Optional<String> reconnectableClusterId = this.getReconnectableClusterId();
        if (reconnectableClusterId.isPresent()) {
            LOGGER.info("Found reconnectable cluster with cluster ID: " + (String)reconnectableClusterId.get());
            return reconnectableClusterId;
        }
        LOGGER.info("No reconnectable cluster found so creating a cluster");
        return Optional.of((Object)this.setupGobblinCluster());
    }

    @VisibleForTesting
    Optional<String> getReconnectableClusterId() throws IOException {
        Tag clusterNameTag = new Tag().withKey(CLUSTER_NAME_ASG_TAG).withValue(this.clusterName);
        List<AutoScalingGroup> autoScalingGroups = this.awsSdkClient.getAutoScalingGroupsWithTag(clusterNameTag);
        if (autoScalingGroups.size() == 0) {
            return Optional.absent();
        }
        if (autoScalingGroups.size() != 2) {
            throw new IOException("Expected 2 auto scaling groups (1 each for master and workers) but found: " + autoScalingGroups.size());
        }
        Optional clusterId = Optional.absent();
        Optional masterAsg = Optional.absent();
        Optional workersAsg = Optional.absent();
        for (TagDescription tagDescription : autoScalingGroups.get(0).getTags()) {
            LOGGER.info("Found tag: " + tagDescription);
            if (tagDescription.getKey().equalsIgnoreCase(CLUSTER_ID_ASG_TAG)) {
                clusterId = Optional.of((Object)tagDescription.getValue());
            }
            if (!tagDescription.getKey().equalsIgnoreCase(ASG_TYPE_ASG_TAG)) continue;
            if (tagDescription.getValue().equalsIgnoreCase(ASG_TYPE_MASTER)) {
                masterAsg = Optional.of((Object)autoScalingGroups.get(0));
                workersAsg = Optional.of((Object)autoScalingGroups.get(1));
                continue;
            }
            masterAsg = Optional.of((Object)autoScalingGroups.get(1));
            workersAsg = Optional.of((Object)autoScalingGroups.get(0));
        }
        if (!clusterId.isPresent()) {
            throw new IOException("Found 2 auto scaling group names for: " + this.clusterName + " but tags seem to be corrupted, hence could not determine cluster id");
        }
        if (!masterAsg.isPresent() || !workersAsg.isPresent()) {
            throw new IOException("Found 2 auto scaling group names for: " + this.clusterName + " but tags seem to be corrupted, hence could not determine master and workers ASG");
        }
        this.masterAutoScalingGroupName = ((AutoScalingGroup)masterAsg.get()).getAutoScalingGroupName();
        this.masterLaunchConfigName = ((AutoScalingGroup)masterAsg.get()).getLaunchConfigurationName();
        this.workerAutoScalingGroupName = ((AutoScalingGroup)workersAsg.get()).getAutoScalingGroupName();
        this.workerLaunchConfigName = ((AutoScalingGroup)workersAsg.get()).getLaunchConfigurationName();
        LOGGER.info("Trying to find cluster master public ip");
        this.masterPublicIp = this.getMasterPublicIp();
        LOGGER.info("Master public ip: " + this.masterPublicIp);
        return clusterId;
    }

    @VisibleForTesting
    String setupGobblinCluster() throws IOException {
        String uuid = UUID.randomUUID().toString();
        String securityGroupName = "GobblinSecurityGroup_" + uuid;
        this.awsSdkClient.createSecurityGroup(securityGroupName, "Gobblin cluster security group");
        this.awsSdkClient.addPermissionsToSecurityGroup(securityGroupName, "0.0.0.0/0", "tcp", 0, 65535);
        String keyName = "GobblinKey_" + uuid;
        String material = this.awsSdkClient.createKeyValuePair(keyName);
        LOGGER.debug("Material is: " + material);
        FileUtils.writeStringToFile((File)new File(keyName + ".pem"), (String)material);
        List<AvailabilityZone> availabilityZones = this.awsSdkClient.getAvailabilityZones();
        String clusterId = this.launchClusterMaster(uuid, keyName, securityGroupName, availabilityZones.get(0));
        this.launchWorkUnitRunners(uuid, keyName, securityGroupName, availabilityZones.get(0));
        return clusterId;
    }

    private String launchClusterMaster(String uuid, String keyName, String securityGroups, AvailabilityZone availabilityZone) {
        String userData = CloudInitScriptBuilder.buildClusterMasterCommand(this.clusterName, this.nfsParentDir, this.sinkLogRootDir, this.awsConfDir, this.appWorkDir, this.masterS3ConfUri, this.masterS3ConfFiles, this.masterS3JarsUri, this.masterS3JarsFiles, this.masterJarsDir, this.masterJvmMemory, this.masterJvmArgs, this.gobblinVersion);
        this.masterLaunchConfigName = MASTER_LAUNCH_CONFIG_NAME_PREFIX + uuid;
        this.awsSdkClient.createLaunchConfig(this.masterLaunchConfigName, this.masterAmiId, this.masterInstanceType, keyName, securityGroups, (Optional<String>)Optional.absent(), (Optional<String>)Optional.absent(), (Optional<BlockDeviceMapping>)Optional.absent(), (Optional<String>)Optional.absent(), (Optional<InstanceMonitoring>)Optional.absent(), userData);
        this.masterAutoScalingGroupName = MASTER_ASG_NAME_PREFIX + uuid;
        boolean minNumMasters = true;
        boolean maxNumMasters = true;
        boolean desiredNumMasters = true;
        Tag clusterNameTag = new Tag().withKey(CLUSTER_NAME_ASG_TAG).withValue(this.clusterName);
        Tag clusterUuidTag = new Tag().withKey(CLUSTER_ID_ASG_TAG).withValue(uuid);
        Tag asgTypeTag = new Tag().withKey(ASG_TYPE_ASG_TAG).withValue(ASG_TYPE_MASTER);
        this.awsSdkClient.createAutoScalingGroup(this.masterAutoScalingGroupName, this.masterLaunchConfigName, 1, 1, 1, (Optional<String>)Optional.of((Object)availabilityZone.getZoneName()), (Optional<Integer>)Optional.absent(), (Optional<Integer>)Optional.absent(), (Optional<String>)Optional.absent(), (Optional<String>)Optional.absent(), (Optional<String>)Optional.absent(), Lists.newArrayList((Object[])new Tag[]{clusterNameTag, clusterUuidTag, asgTypeTag}));
        LOGGER.info("Waiting for cluster master to launch");
        this.masterPublicIp = this.getMasterPublicIp();
        LOGGER.info("Master public ip: " + this.masterPublicIp);
        return uuid;
    }

    private String getMasterPublicIp() {
        long startTime = System.currentTimeMillis();
        long launchTimeout = TimeUnit.MINUTES.toMillis(10L);
        boolean isMasterLaunched = false;
        List<Object> instanceIds = Collections.emptyList();
        while (!isMasterLaunched && System.currentTimeMillis() - startTime < launchTimeout) {
            try {
                Thread.sleep(5000L);
            }
            catch (InterruptedException e) {
                throw new RuntimeException("Interrupted while waiting for cluster master to boot up", e);
            }
            instanceIds = this.awsSdkClient.getInstancesForGroup(this.masterAutoScalingGroupName, "running");
            isMasterLaunched = instanceIds.size() > 0;
        }
        if (!isMasterLaunched) {
            throw new RuntimeException("Timed out while waiting for cluster master. Check for issue manually for ASG: " + this.masterAutoScalingGroupName);
        }
        return ((Instance)instanceIds.get(0)).getPublicIpAddress();
    }

    private void launchWorkUnitRunners(String uuid, String keyName, String securityGroups, AvailabilityZone availabilityZone) {
        String userData = CloudInitScriptBuilder.buildClusterWorkerCommand(this.clusterName, this.nfsParentDir, this.sinkLogRootDir, this.awsConfDir, this.appWorkDir, this.masterPublicIp, this.workerS3ConfUri, this.workerS3ConfFiles, this.workerS3JarsUri, this.workerS3JarsFiles, this.workerJarsDir, this.workerJvmMemory, this.workerJvmArgs, this.gobblinVersion);
        this.workerLaunchConfigName = WORKERS_LAUNCH_CONFIG_PREFIX + uuid;
        this.awsSdkClient.createLaunchConfig(this.workerLaunchConfigName, this.workerAmiId, this.workerInstanceType, keyName, securityGroups, (Optional<String>)Optional.absent(), (Optional<String>)Optional.absent(), (Optional<BlockDeviceMapping>)Optional.absent(), (Optional<String>)Optional.absent(), (Optional<InstanceMonitoring>)Optional.absent(), userData);
        this.workerAutoScalingGroupName = WORKERS_ASG_NAME_PREFIX + uuid;
        Tag clusterNameTag = new Tag().withKey(CLUSTER_NAME_ASG_TAG).withValue(this.clusterName);
        Tag clusterUuidTag = new Tag().withKey(CLUSTER_ID_ASG_TAG).withValue(uuid);
        Tag asgTypeTag = new Tag().withKey(ASG_TYPE_ASG_TAG).withValue(ASG_TYPE_WORKERS);
        this.awsSdkClient.createAutoScalingGroup(this.workerAutoScalingGroupName, this.workerLaunchConfigName, this.minWorkers, this.maxWorkers, this.desiredWorkers, (Optional<String>)Optional.of((Object)availabilityZone.getZoneName()), (Optional<Integer>)Optional.absent(), (Optional<Integer>)Optional.absent(), (Optional<String>)Optional.absent(), (Optional<String>)Optional.absent(), (Optional<String>)Optional.absent(), Lists.newArrayList((Object[])new Tag[]{clusterNameTag, clusterUuidTag, asgTypeTag}));
    }

    @VisibleForTesting
    void sendShutdownRequest() {
        Criteria criteria = new Criteria();
        criteria.setInstanceName("%");
        criteria.setResource("%");
        criteria.setPartition("%");
        criteria.setPartitionState("%");
        criteria.setRecipientInstanceType(InstanceType.CONTROLLER);
        criteria.setSessionSpecific(true);
        Message shutdownRequest = new Message("SHUTDOWN", HelixMessageSubTypes.APPLICATION_MASTER_SHUTDOWN.toString().toLowerCase() + UUID.randomUUID().toString());
        shutdownRequest.setMsgSubType(HelixMessageSubTypes.APPLICATION_MASTER_SHUTDOWN.toString());
        shutdownRequest.setMsgState(Message.MessageState.NEW);
        shutdownRequest.setTgtSessionId("*");
        int timeout = 300000;
        int messagesSent = this.helixManager.getMessagingService().send(criteria, shutdownRequest, this.shutdownASG(), 300000);
        if (messagesSent == 0) {
            LOGGER.error(String.format("Failed to send the %s message to the controller", shutdownRequest.getMsgSubType()));
        }
    }

    private AsyncCallback shutdownASG() {
        Optional optionalLaunchConfigurationNames = Optional.of(Arrays.asList(this.masterLaunchConfigName, this.workerLaunchConfigName));
        Optional optionalAutoScalingGroupNames = Optional.of(Arrays.asList(this.masterAutoScalingGroupName, this.workerAutoScalingGroupName));
        return new AWSShutdownHandler(this.awsSdkClient, (Optional<List<String>>)optionalLaunchConfigurationNames, (Optional<List<String>>)optionalAutoScalingGroupNames);
    }

    private void cleanUpClusterWorkDirectory(String clusterId) throws IOException {
        File appWorkDir = new File(GobblinClusterUtils.getAppWorkDirPath((String)this.clusterName, (String)clusterId));
        if (appWorkDir.exists() && appWorkDir.isDirectory()) {
            LOGGER.info("Deleting application working directory " + appWorkDir);
            FileUtils.deleteDirectory((File)appWorkDir);
        }
    }

    private void sendEmailOnShutdown(Optional<String> report) {
        String subject = String.format("Gobblin AWS cluster %s completed", this.clusterName);
        StringBuilder messageBuilder = new StringBuilder("Gobblin AWS cluster was shutdown at: " + new Date());
        if (report.isPresent()) {
            messageBuilder.append(' ').append((String)report.get());
        }
        try {
            EmailUtils.sendEmail((State)ConfigUtils.configToState((Config)this.config), (String)subject, (String)messageBuilder.toString());
        }
        catch (EmailException ee) {
            LOGGER.error("Failed to send email notification on shutdown", (Throwable)ee);
        }
    }

    public static void main(String[] args) throws Exception {
        final GobblinAWSClusterLauncher gobblinAWSClusterLauncher = new GobblinAWSClusterLauncher(ConfigFactory.load());
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                try {
                    gobblinAWSClusterLauncher.stop();
                }
                catch (IOException ioe) {
                    LOGGER.error("Failed to shutdown the " + GobblinAWSClusterLauncher.class.getSimpleName(), (Throwable)ioe);
                }
                catch (TimeoutException te) {
                    LOGGER.error("Timeout in stopping the service manager", (Throwable)te);
                }
                finally {
                    if (gobblinAWSClusterLauncher.emailNotificationOnShutdown) {
                        gobblinAWSClusterLauncher.sendEmailOnShutdown((Optional<String>)Optional.absent());
                    }
                }
            }
        });
        gobblinAWSClusterLauncher.launch();
    }
}

