/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.state.forst;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;
import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.flink.runtime.state.CompositeKeySerializationUtils;
import org.apache.flink.runtime.state.KeyGroupRange;
import org.apache.flink.runtime.state.KeyedStateHandle;
import org.apache.flink.shaded.guava32.com.google.common.primitives.UnsignedBytes;
import org.apache.flink.util.Preconditions;
import org.forstdb.ColumnFamilyHandle;
import org.forstdb.LiveFileMetaData;
import org.forstdb.RocksDB;
import org.forstdb.RocksDBException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ForStIncrementalCheckpointUtils {
    private static final Logger logger = LoggerFactory.getLogger(ForStIncrementalCheckpointUtils.class);

    private static Score stateHandleEvaluator(KeyedStateHandle stateHandle, KeyGroupRange targetKeyGroupRange, double overlapFractionThreshold) {
        KeyGroupRange handleKeyGroupRange = stateHandle.getKeyGroupRange();
        KeyGroupRange intersectGroup = handleKeyGroupRange.getIntersection(targetKeyGroupRange);
        double overlapFraction = (double)intersectGroup.getNumberOfKeyGroups() / (double)handleKeyGroupRange.getNumberOfKeyGroups();
        if (overlapFraction < overlapFractionThreshold) {
            return Score.MIN;
        }
        return new Score(intersectGroup.getNumberOfKeyGroups(), overlapFraction);
    }

    public static void clipDBWithKeyGroupRange(@Nonnull RocksDB db, @Nonnull List<ColumnFamilyHandle> columnFamilyHandles, @Nonnull KeyGroupRange targetKeyGroupRange, @Nonnull KeyGroupRange currentKeyGroupRange, @Nonnegative int keyGroupPrefixBytes, boolean useDeleteFilesInRange) throws RocksDBException {
        ArrayList<byte[]> deleteFilesRanges = new ArrayList<byte[]>(4);
        if (currentKeyGroupRange.getStartKeyGroup() < targetKeyGroupRange.getStartKeyGroup()) {
            ForStIncrementalCheckpointUtils.prepareRangeDeletes(keyGroupPrefixBytes, currentKeyGroupRange.getStartKeyGroup(), targetKeyGroupRange.getStartKeyGroup(), deleteFilesRanges);
        }
        if (currentKeyGroupRange.getEndKeyGroup() > targetKeyGroupRange.getEndKeyGroup()) {
            ForStIncrementalCheckpointUtils.prepareRangeDeletes(keyGroupPrefixBytes, targetKeyGroupRange.getEndKeyGroup() + 1, currentKeyGroupRange.getEndKeyGroup() + 1, deleteFilesRanges);
        }
        logger.info("Performing range delete for backend with target key-groups range {} with boundaries set {} - deleteFilesInRanges = {}.", new Object[]{targetKeyGroupRange.prettyPrintInterval(), deleteFilesRanges.stream().map(Arrays::toString).collect(Collectors.toList()), useDeleteFilesInRange});
        ForStIncrementalCheckpointUtils.deleteRangeData(db, columnFamilyHandles, deleteFilesRanges, useDeleteFilesInRange);
    }

    private static void prepareRangeDeletes(int keyGroupPrefixBytes, int beginKeyGroup, int endKeyGroup, List<byte[]> deleteFilesRangesOut) {
        byte[] beginKeyGroupBytes = new byte[keyGroupPrefixBytes];
        byte[] endKeyGroupBytes = new byte[keyGroupPrefixBytes];
        CompositeKeySerializationUtils.serializeKeyGroup((int)beginKeyGroup, (byte[])beginKeyGroupBytes);
        CompositeKeySerializationUtils.serializeKeyGroup((int)endKeyGroup, (byte[])endKeyGroupBytes);
        deleteFilesRangesOut.add(beginKeyGroupBytes);
        deleteFilesRangesOut.add(endKeyGroupBytes);
    }

    private static void deleteRangeData(RocksDB db, List<ColumnFamilyHandle> columnFamilyHandles, List<byte[]> deleteRanges, boolean useDeleteFilesInRange) throws RocksDBException {
        if (deleteRanges.isEmpty()) {
            return;
        }
        Preconditions.checkArgument((deleteRanges.size() % 2 == 0 ? 1 : 0) != 0);
        for (ColumnFamilyHandle columnFamilyHandle : columnFamilyHandles) {
            if (useDeleteFilesInRange) {
                db.deleteFilesInRanges(columnFamilyHandle, deleteRanges, false);
            }
            for (int i = 0; i < deleteRanges.size() / 2; ++i) {
                db.deleteRange(columnFamilyHandle, deleteRanges.get(i * 2), deleteRanges.get(i * 2 + 1));
            }
        }
    }

    public static RangeCheckResult checkSstDataAgainstKeyGroupRange(RocksDB db, int keyGroupPrefixBytes, KeyGroupRange dbExpectedKeyGroupRange) {
        byte[] beginKeyGroupBytes = new byte[keyGroupPrefixBytes];
        byte[] endKeyGroupBytes = new byte[keyGroupPrefixBytes];
        CompositeKeySerializationUtils.serializeKeyGroup((int)dbExpectedKeyGroupRange.getStartKeyGroup(), (byte[])beginKeyGroupBytes);
        CompositeKeySerializationUtils.serializeKeyGroup((int)(dbExpectedKeyGroupRange.getEndKeyGroup() + 1), (byte[])endKeyGroupBytes);
        KeyRange dbKeyRange = ForStIncrementalCheckpointUtils.getDBKeyRange(db);
        return RangeCheckResult.of(beginKeyGroupBytes, endKeyGroupBytes, dbKeyRange.minKey, dbKeyRange.maxKey, keyGroupPrefixBytes);
    }

    private static KeyRange getDBKeyRange(RocksDB db) {
        Comparator comparator = UnsignedBytes.lexicographicalComparator();
        List liveFilesMetaData = db.getLiveFilesMetaData();
        if (liveFilesMetaData.isEmpty()) {
            return KeyRange.EMPTY;
        }
        Iterator liveFileMetaDataIterator = liveFilesMetaData.iterator();
        LiveFileMetaData fileMetaData = (LiveFileMetaData)liveFileMetaDataIterator.next();
        byte[] smallestKey = fileMetaData.smallestKey();
        byte[] largestKey = fileMetaData.largestKey();
        while (liveFileMetaDataIterator.hasNext()) {
            fileMetaData = (LiveFileMetaData)liveFileMetaDataIterator.next();
            byte[] sstSmallestKey = fileMetaData.smallestKey();
            byte[] sstLargestKey = fileMetaData.largestKey();
            if (comparator.compare(sstSmallestKey, smallestKey) < 0) {
                smallestKey = sstSmallestKey;
            }
            if (comparator.compare(sstLargestKey, largestKey) <= 0) continue;
            largestKey = sstLargestKey;
        }
        return KeyRange.of(smallestKey, largestKey);
    }

    public static boolean beforeThePrefixBytes(@Nonnull byte[] bytes, @Nonnull byte[] prefixBytes) {
        int prefixLength = prefixBytes.length;
        for (int i = 0; i < prefixLength; ++i) {
            int r = (char)prefixBytes[i] - (char)bytes[i];
            if (r == 0) continue;
            return r > 0;
        }
        return false;
    }

    public static <T extends KeyedStateHandle> int findTheBestStateHandleForInitial(@Nonnull List<T> restoreStateHandles, @Nonnull KeyGroupRange targetKeyGroupRange, double overlapFractionThreshold) {
        if (restoreStateHandles.isEmpty()) {
            return -1;
        }
        if (restoreStateHandles.size() == 1) {
            return 0;
        }
        int currentPos = 0;
        int bestHandlePos = 0;
        Score bestScore = Score.MIN;
        for (KeyedStateHandle rawStateHandle : restoreStateHandles) {
            Score handleScore = ForStIncrementalCheckpointUtils.stateHandleEvaluator(rawStateHandle, targetKeyGroupRange, overlapFractionThreshold);
            if (handleScore.compareTo(bestScore) > 0) {
                bestHandlePos = currentPos;
                bestScore = handleScore;
            }
            ++currentPos;
        }
        return bestHandlePos;
    }

    public static final class RangeCheckResult {
        private final byte[] proclaimedMinKey;
        private final byte[] proclaimedMaxKey;
        private final byte[] actualMinKey;
        private final byte[] actualMaxKey;
        final boolean leftInRange;
        final boolean rightInRange;
        final int keyGroupPrefixBytes;

        private RangeCheckResult(byte[] proclaimedMinKey, byte[] proclaimedMaxKey, byte[] actualMinKey, byte[] actualMaxKey, int keyGroupPrefixBytes) {
            Comparator comparator = UnsignedBytes.lexicographicalComparator();
            this.proclaimedMinKey = proclaimedMinKey;
            this.proclaimedMaxKey = proclaimedMaxKey;
            this.actualMinKey = actualMinKey;
            this.actualMaxKey = actualMaxKey;
            this.leftInRange = comparator.compare(actualMinKey, proclaimedMinKey) >= 0;
            this.rightInRange = comparator.compare(actualMaxKey, proclaimedMaxKey) < 0;
            this.keyGroupPrefixBytes = keyGroupPrefixBytes;
        }

        public boolean allInRange() {
            return this.leftInRange && this.rightInRange;
        }

        public byte[] getProclaimedMinKey() {
            return this.proclaimedMinKey;
        }

        public byte[] getProclaimedMaxKey() {
            return this.proclaimedMaxKey;
        }

        public byte[] getActualMinKey() {
            return this.actualMinKey;
        }

        public byte[] getActualMaxKey() {
            return this.actualMaxKey;
        }

        public int getKeyGroupPrefixBytes() {
            return this.keyGroupPrefixBytes;
        }

        public boolean isLeftInRange() {
            return this.leftInRange;
        }

        public boolean isRightInRange() {
            return this.rightInRange;
        }

        static RangeCheckResult of(byte[] proclaimedMinKey, byte[] proclaimedMaxKey, byte[] actualMinKey, byte[] actualMaxKey, int keyGroupPrefixBytes) {
            return new RangeCheckResult(proclaimedMinKey, proclaimedMaxKey, actualMinKey, actualMaxKey, keyGroupPrefixBytes);
        }

        public String toString() {
            return "RangeCheckResult{leftInRange=" + this.leftInRange + ", rightInRange=" + this.rightInRange + ", actualMinKeyGroup=" + CompositeKeySerializationUtils.extractKeyGroup((int)this.keyGroupPrefixBytes, (byte[])this.getActualMinKey()) + ", actualMaxKeyGroup=" + CompositeKeySerializationUtils.extractKeyGroup((int)this.keyGroupPrefixBytes, (byte[])this.getActualMaxKey()) + "}";
        }
    }

    private static final class KeyRange {
        static final KeyRange EMPTY = KeyRange.of(new byte[0], new byte[0]);
        final byte[] minKey;
        final byte[] maxKey;

        private KeyRange(byte[] minKey, byte[] maxKey) {
            this.minKey = minKey;
            this.maxKey = maxKey;
        }

        static KeyRange of(byte[] minKey, byte[] maxKey) {
            return new KeyRange(minKey, maxKey);
        }
    }

    private static class Score
    implements Comparable<Score> {
        public static final Score MIN = new Score(Integer.MIN_VALUE, -1.0);
        private final int intersectGroupRange;
        private final double overlapFraction;

        public Score(int intersectGroupRange, double overlapFraction) {
            this.intersectGroupRange = intersectGroupRange;
            this.overlapFraction = overlapFraction;
        }

        public int getIntersectGroupRange() {
            return this.intersectGroupRange;
        }

        public double getOverlapFraction() {
            return this.overlapFraction;
        }

        @Override
        public int compareTo(@Nullable Score other) {
            return Comparator.nullsFirst(Comparator.comparing(Score::getIntersectGroupRange).thenComparing(Score::getIntersectGroupRange).thenComparing(Score::getOverlapFraction)).compare(this, other);
        }
    }
}

