/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.types.variant;

import java.io.CharArrayWriter;
import java.io.IOException;
import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Base64;
import java.util.Objects;
import org.apache.flink.annotation.Internal;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.core.JsonFactory;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.core.JsonGenerator;
import org.apache.flink.types.variant.BinaryVariantUtil;
import org.apache.flink.types.variant.Variant;
import org.apache.flink.types.variant.VariantTypeException;

@Internal
public final class BinaryVariant
implements Variant {
    private final byte[] value;
    private final byte[] metadata;
    private final int pos;

    public BinaryVariant(byte[] value, byte[] metadata) {
        this(value, metadata, 0);
    }

    private BinaryVariant(byte[] value, byte[] metadata, int pos) {
        this.value = value;
        this.metadata = metadata;
        this.pos = pos;
        if (metadata.length < 1 || (metadata[0] & 0xF) != 1) {
            throw BinaryVariantUtil.malformedVariant();
        }
        if (metadata.length > 0x1000000 || value.length > 0x1000000) {
            throw BinaryVariantUtil.variantConstructorSizeLimit();
        }
    }

    @Override
    public boolean isPrimitive() {
        return !this.isArray() && !this.isObject();
    }

    @Override
    public boolean isArray() {
        return this.getType() == Variant.Type.ARRAY;
    }

    @Override
    public boolean isObject() {
        return this.getType() == Variant.Type.OBJECT;
    }

    @Override
    public boolean isNull() {
        return this.getType() == Variant.Type.NULL;
    }

    @Override
    public Variant.Type getType() {
        return BinaryVariantUtil.getType(this.value, this.pos);
    }

    @Override
    public boolean getBoolean() throws VariantTypeException {
        this.checkType(Variant.Type.BOOLEAN, this.getType());
        return BinaryVariantUtil.getBoolean(this.value, this.pos);
    }

    @Override
    public byte getByte() throws VariantTypeException {
        this.checkType(Variant.Type.TINYINT, this.getType());
        return (byte)BinaryVariantUtil.getLong(this.value, this.pos);
    }

    @Override
    public short getShort() throws VariantTypeException {
        this.checkType(Variant.Type.SMALLINT, this.getType());
        return (short)BinaryVariantUtil.getLong(this.value, this.pos);
    }

    @Override
    public int getInt() throws VariantTypeException {
        this.checkType(Variant.Type.INT, this.getType());
        return (int)BinaryVariantUtil.getLong(this.value, this.pos);
    }

    @Override
    public long getLong() throws VariantTypeException {
        this.checkType(Variant.Type.BIGINT, this.getType());
        return BinaryVariantUtil.getLong(this.value, this.pos);
    }

    @Override
    public float getFloat() throws VariantTypeException {
        this.checkType(Variant.Type.FLOAT, this.getType());
        return BinaryVariantUtil.getFloat(this.value, this.pos);
    }

    @Override
    public BigDecimal getDecimal() throws VariantTypeException {
        this.checkType(Variant.Type.DECIMAL, this.getType());
        return BinaryVariantUtil.getDecimal(this.value, this.pos);
    }

    @Override
    public double getDouble() throws VariantTypeException {
        this.checkType(Variant.Type.DOUBLE, this.getType());
        return BinaryVariantUtil.getDouble(this.value, this.pos);
    }

    @Override
    public String getString() throws VariantTypeException {
        this.checkType(Variant.Type.STRING, this.getType());
        return BinaryVariantUtil.getString(this.value, this.pos);
    }

    @Override
    public LocalDate getDate() throws VariantTypeException {
        this.checkType(Variant.Type.DATE, this.getType());
        return LocalDate.ofEpochDay(BinaryVariantUtil.getLong(this.value, this.pos));
    }

    @Override
    public LocalDateTime getDateTime() throws VariantTypeException {
        this.checkType(Variant.Type.TIMESTAMP, this.getType());
        return BinaryVariant.microsToInstant(BinaryVariantUtil.getLong(this.value, this.pos)).atZone(ZoneOffset.UTC).toLocalDateTime();
    }

    @Override
    public Instant getInstant() throws VariantTypeException {
        this.checkType(Variant.Type.TIMESTAMP_LTZ, this.getType());
        return BinaryVariant.microsToInstant(BinaryVariantUtil.getLong(this.value, this.pos));
    }

    @Override
    public byte[] getBytes() throws VariantTypeException {
        this.checkType(Variant.Type.BYTES, this.getType());
        return BinaryVariantUtil.getBinary(this.value, this.pos);
    }

    @Override
    public Object get() throws VariantTypeException {
        switch (this.getType()) {
            case NULL: {
                return null;
            }
            case BOOLEAN: {
                return this.getBoolean();
            }
            case TINYINT: {
                return this.getByte();
            }
            case SMALLINT: {
                return this.getShort();
            }
            case INT: {
                return this.getInt();
            }
            case BIGINT: {
                return this.getLong();
            }
            case FLOAT: {
                return Float.valueOf(this.getFloat());
            }
            case DOUBLE: {
                return this.getDouble();
            }
            case DECIMAL: {
                return this.getDecimal();
            }
            case STRING: {
                return this.getString();
            }
            case DATE: {
                return this.getDate();
            }
            case TIMESTAMP: {
                return this.getDateTime();
            }
            case TIMESTAMP_LTZ: {
                return this.getInstant();
            }
            case BYTES: {
                return this.getBytes();
            }
        }
        throw new VariantTypeException(String.format("Expecting a primitive variant but got %s", new Object[]{this.getType()}));
    }

    @Override
    public <T> T getAs() throws VariantTypeException {
        return (T)this.get();
    }

    @Override
    public Variant getElement(int index) throws VariantTypeException {
        return this.getElementAtIndex(index);
    }

    @Override
    public Variant getField(String fieldName) throws VariantTypeException {
        return this.getFieldByKey(fieldName);
    }

    @Override
    public String toJson() {
        StringBuilder sb = new StringBuilder();
        BinaryVariant.toJsonImpl(this.value, this.metadata, this.pos, sb, ZoneOffset.UTC);
        return sb.toString();
    }

    public byte[] getValue() {
        if (this.pos == 0) {
            return this.value;
        }
        int size = BinaryVariantUtil.valueSize(this.value, this.pos);
        BinaryVariantUtil.checkIndex(this.pos + size - 1, this.value.length);
        return Arrays.copyOfRange(this.value, this.pos, this.pos + size);
    }

    public byte[] getMetadata() {
        return this.metadata;
    }

    public int getPos() {
        return this.pos;
    }

    private static void toJsonImpl(byte[] value, byte[] metadata, int pos, StringBuilder sb, ZoneId zoneId) {
        switch (BinaryVariantUtil.getType(value, pos)) {
            case OBJECT: {
                BinaryVariantUtil.handleObject(value, pos, (size, idSize, offsetSize, idStart, offsetStart, dataStart) -> {
                    sb.append('{');
                    for (int i = 0; i < size; ++i) {
                        int id = BinaryVariantUtil.readUnsigned(value, idStart + idSize * i, idSize);
                        int offset = BinaryVariantUtil.readUnsigned(value, offsetStart + offsetSize * i, offsetSize);
                        int elementPos = dataStart + offset;
                        if (i != 0) {
                            sb.append(',');
                        }
                        sb.append(BinaryVariant.escapeJson(BinaryVariantUtil.getMetadataKey(metadata, id)));
                        sb.append(':');
                        BinaryVariant.toJsonImpl(value, metadata, elementPos, sb, zoneId);
                    }
                    sb.append('}');
                    return null;
                });
                break;
            }
            case ARRAY: {
                BinaryVariantUtil.handleArray(value, pos, (size, offsetSize, offsetStart, dataStart) -> {
                    sb.append('[');
                    for (int i = 0; i < size; ++i) {
                        int offset = BinaryVariantUtil.readUnsigned(value, offsetStart + offsetSize * i, offsetSize);
                        int elementPos = dataStart + offset;
                        if (i != 0) {
                            sb.append(',');
                        }
                        BinaryVariant.toJsonImpl(value, metadata, elementPos, sb, zoneId);
                    }
                    sb.append(']');
                    return null;
                });
                break;
            }
            case NULL: {
                sb.append("null");
                break;
            }
            case BOOLEAN: {
                sb.append(BinaryVariantUtil.getBoolean(value, pos));
                break;
            }
            case TINYINT: 
            case SMALLINT: 
            case INT: 
            case BIGINT: {
                sb.append(BinaryVariantUtil.getLong(value, pos));
                break;
            }
            case STRING: {
                sb.append(BinaryVariant.escapeJson(BinaryVariantUtil.getString(value, pos)));
                break;
            }
            case DOUBLE: {
                sb.append(BinaryVariantUtil.getDouble(value, pos));
                break;
            }
            case DECIMAL: {
                sb.append(BinaryVariantUtil.getDecimal(value, pos).toPlainString());
                break;
            }
            case DATE: {
                BinaryVariant.appendQuoted(sb, LocalDate.ofEpochDay((int)BinaryVariantUtil.getLong(value, pos)).toString());
                break;
            }
            case TIMESTAMP_LTZ: {
                BinaryVariant.appendQuoted(sb, BinaryVariantUtil.TIMESTAMP_LTZ_FORMATTER.format(BinaryVariant.microsToInstant(BinaryVariantUtil.getLong(value, pos)).atZone(zoneId)));
                break;
            }
            case TIMESTAMP: {
                BinaryVariant.appendQuoted(sb, BinaryVariantUtil.TIMESTAMP_FORMATTER.format(BinaryVariant.microsToInstant(BinaryVariantUtil.getLong(value, pos)).atZone(ZoneOffset.UTC)));
                break;
            }
            case FLOAT: {
                sb.append(BinaryVariantUtil.getFloat(value, pos));
                break;
            }
            case BYTES: {
                BinaryVariant.appendQuoted(sb, Base64.getEncoder().encodeToString(BinaryVariantUtil.getBinary(value, pos)));
                break;
            }
            default: {
                throw BinaryVariantUtil.unexpectedType(BinaryVariantUtil.getType(value, pos));
            }
        }
    }

    private static Instant microsToInstant(long timestamp) {
        return Instant.EPOCH.plus(timestamp, ChronoUnit.MICROS);
    }

    private void checkType(Variant.Type expected, Variant.Type actual) {
        if (expected != actual) {
            throw new VariantTypeException(String.format("Expected type %s but got %s", new Object[]{expected, actual}));
        }
    }

    private BinaryVariant getFieldByKey(String key) {
        return BinaryVariantUtil.handleObject(this.value, this.pos, (size, idSize, offsetSize, idStart, offsetStart, dataStart) -> {
            if (size < 32) {
                for (int i = 0; i < size; ++i) {
                    int id = BinaryVariantUtil.readUnsigned(this.value, idStart + idSize * i, idSize);
                    if (!key.equals(BinaryVariantUtil.getMetadataKey(this.metadata, id))) continue;
                    int offset = BinaryVariantUtil.readUnsigned(this.value, offsetStart + offsetSize * i, offsetSize);
                    return new BinaryVariant(this.value, this.metadata, dataStart + offset);
                }
            } else {
                int low = 0;
                int high = size - 1;
                while (low <= high) {
                    int mid = low + high >>> 1;
                    int id = BinaryVariantUtil.readUnsigned(this.value, idStart + idSize * mid, idSize);
                    int cmp = BinaryVariantUtil.getMetadataKey(this.metadata, id).compareTo(key);
                    if (cmp < 0) {
                        low = mid + 1;
                        continue;
                    }
                    if (cmp > 0) {
                        high = mid - 1;
                        continue;
                    }
                    int offset = BinaryVariantUtil.readUnsigned(this.value, offsetStart + offsetSize * mid, offsetSize);
                    return new BinaryVariant(this.value, this.metadata, dataStart + offset);
                }
            }
            return null;
        });
    }

    private BinaryVariant getElementAtIndex(int index) {
        return BinaryVariantUtil.handleArray(this.value, this.pos, (size, offsetSize, offsetStart, dataStart) -> {
            if (index < 0 || index >= size) {
                return null;
            }
            int offset = BinaryVariantUtil.readUnsigned(this.value, offsetStart + offsetSize * index, offsetSize);
            return new BinaryVariant(this.value, this.metadata, dataStart + offset);
        });
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static String escapeJson(String str) {
        try (CharArrayWriter writer = new CharArrayWriter();){
            String string;
            block13: {
                JsonGenerator gen = new JsonFactory().createGenerator(writer);
                try {
                    gen.writeString(str);
                    gen.flush();
                    string = writer.toString();
                    if (gen == null) break block13;
                }
                catch (Throwable throwable) {
                    if (gen != null) {
                        try {
                            gen.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                gen.close();
            }
            return string;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static void appendQuoted(StringBuilder sb, String str) {
        sb.append('\"');
        sb.append(str);
        sb.append('\"');
    }

    public String toString() {
        return this.toJson();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof BinaryVariant)) {
            return false;
        }
        BinaryVariant variant = (BinaryVariant)o;
        return this.getPos() == variant.getPos() && Objects.deepEquals(this.getValue(), variant.getValue()) && Objects.deepEquals(this.getMetadata(), variant.getMetadata());
    }

    public int hashCode() {
        return Objects.hash(Arrays.hashCode(this.value), Arrays.hashCode(this.metadata), this.pos);
    }
}

