/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.state.filesystem;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import javax.annotation.Nullable;
import org.apache.flink.core.fs.DuplicatingFileSystem;
import org.apache.flink.core.fs.EntropyInjector;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.OutputStreamAndPath;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.state.CheckpointStateOutputStream;
import org.apache.flink.runtime.state.CheckpointStreamFactory;
import org.apache.flink.runtime.state.CheckpointedStateScope;
import org.apache.flink.runtime.state.StreamStateHandle;
import org.apache.flink.runtime.state.filesystem.FileStateHandle;
import org.apache.flink.runtime.state.filesystem.FsCheckpointStateToolset;
import org.apache.flink.runtime.state.filesystem.RelativeFileStateHandle;
import org.apache.flink.runtime.state.memory.ByteStreamStateHandle;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FsCheckpointStreamFactory
implements CheckpointStreamFactory {
    private static final Logger LOG = LoggerFactory.getLogger(FsCheckpointStreamFactory.class);
    public static final int MAX_FILE_STATE_THRESHOLD = 0x100000;
    private final int writeBufferSize;
    private final int fileStateThreshold;
    private final Path checkpointDirectory;
    private final Path sharedStateDirectory;
    private final FileSystem filesystem;
    private final FsCheckpointStateToolset privateStateToolset;
    private final FsCheckpointStateToolset sharedStateToolset;

    public FsCheckpointStreamFactory(FileSystem fileSystem, Path checkpointDirectory, Path sharedStateDirectory, int fileStateSizeThreshold, int writeBufferSize) {
        if (fileStateSizeThreshold < 0) {
            throw new IllegalArgumentException("The threshold for file state size must be zero or larger.");
        }
        if (writeBufferSize < 0) {
            throw new IllegalArgumentException("The write buffer size must be zero or larger.");
        }
        if (fileStateSizeThreshold > 0x100000) {
            throw new IllegalArgumentException("The threshold for file state size cannot be larger than 1048576");
        }
        this.filesystem = Preconditions.checkNotNull(fileSystem);
        this.checkpointDirectory = Preconditions.checkNotNull(checkpointDirectory);
        this.sharedStateDirectory = Preconditions.checkNotNull(sharedStateDirectory);
        this.fileStateThreshold = fileStateSizeThreshold;
        this.writeBufferSize = writeBufferSize;
        if (fileSystem instanceof DuplicatingFileSystem) {
            DuplicatingFileSystem duplicatingFileSystem = (DuplicatingFileSystem)((Object)fileSystem);
            this.privateStateToolset = new FsCheckpointStateToolset(checkpointDirectory, duplicatingFileSystem);
            this.sharedStateToolset = new FsCheckpointStateToolset(sharedStateDirectory, duplicatingFileSystem);
        } else {
            this.privateStateToolset = null;
            this.sharedStateToolset = null;
        }
    }

    @Override
    public FsCheckpointStateOutputStream createCheckpointStateOutputStream(CheckpointedStateScope scope) throws IOException {
        Path target = this.getTargetPath(scope);
        int bufferSize = Math.max(this.writeBufferSize, this.fileStateThreshold);
        boolean entropyInjecting = EntropyInjector.isEntropyInjecting(this.filesystem, target);
        boolean absolutePath = entropyInjecting || scope == CheckpointedStateScope.SHARED;
        return new FsCheckpointStateOutputStream(target, this.filesystem, bufferSize, this.fileStateThreshold, !absolutePath);
    }

    private Path getTargetPath(CheckpointedStateScope scope) {
        return scope == CheckpointedStateScope.EXCLUSIVE ? this.checkpointDirectory : this.sharedStateDirectory;
    }

    @Override
    public boolean canFastDuplicate(StreamStateHandle stateHandle, CheckpointedStateScope scope) throws IOException {
        if (this.privateStateToolset == null || this.sharedStateToolset == null) {
            return false;
        }
        switch (scope) {
            case EXCLUSIVE: {
                return this.privateStateToolset.canFastDuplicate(stateHandle);
            }
            case SHARED: {
                return this.sharedStateToolset.canFastDuplicate(stateHandle);
            }
        }
        return false;
    }

    @Override
    public List<StreamStateHandle> duplicate(List<StreamStateHandle> stateHandles, CheckpointedStateScope scope) throws IOException {
        if (this.privateStateToolset == null || this.sharedStateToolset == null) {
            throw new IllegalArgumentException("The underlying FS does not support duplication.");
        }
        switch (scope) {
            case EXCLUSIVE: {
                return this.privateStateToolset.duplicate(stateHandles);
            }
            case SHARED: {
                return this.sharedStateToolset.duplicate(stateHandles);
            }
        }
        throw new IllegalArgumentException("Unknown state scope: " + (Object)((Object)scope));
    }

    public String toString() {
        return "File Stream Factory @ " + this.checkpointDirectory;
    }

    public static class FsCheckpointStateOutputStream
    extends CheckpointStateOutputStream {
        private final byte[] writeBuffer;
        private int pos;
        private FSDataOutputStream outStream;
        private final int localStateThreshold;
        private final Path basePath;
        private final FileSystem fs;
        private Path statePath;
        private String relativeStatePath;
        private volatile boolean closed;
        private final boolean allowRelativePaths;

        public FsCheckpointStateOutputStream(Path basePath, FileSystem fs, int bufferSize, int localStateThreshold) {
            this(basePath, fs, bufferSize, localStateThreshold, false);
        }

        public FsCheckpointStateOutputStream(Path basePath, FileSystem fs, int bufferSize, int localStateThreshold, boolean allowRelativePaths) {
            if (bufferSize < localStateThreshold) {
                throw new IllegalArgumentException();
            }
            this.basePath = basePath;
            this.fs = fs;
            this.writeBuffer = new byte[bufferSize];
            this.localStateThreshold = localStateThreshold;
            this.allowRelativePaths = allowRelativePaths;
        }

        @Override
        public void write(int b) throws IOException {
            if (this.pos >= this.writeBuffer.length) {
                this.flushToFile();
            }
            this.writeBuffer[this.pos++] = (byte)b;
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            if (len < this.writeBuffer.length) {
                int remaining = this.writeBuffer.length - this.pos;
                if (len > remaining) {
                    System.arraycopy(b, off, this.writeBuffer, this.pos, remaining);
                    off += remaining;
                    len -= remaining;
                    this.pos += remaining;
                    this.flushToFile();
                }
                System.arraycopy(b, off, this.writeBuffer, this.pos, len);
                this.pos += len;
            } else {
                this.flushToFile();
                this.outStream.write(b, off, len);
            }
        }

        @Override
        public long getPos() throws IOException {
            return (long)this.pos + (this.outStream == null ? 0L : this.outStream.getPos());
        }

        public void flushToFile() throws IOException {
            if (!this.closed) {
                if (this.outStream == null) {
                    this.createStream();
                }
                if (this.pos > 0) {
                    this.outStream.write(this.writeBuffer, 0, this.pos);
                    this.pos = 0;
                }
            } else {
                throw new IOException("closed");
            }
        }

        @Override
        public void flush() throws IOException {
            if (this.outStream != null || this.pos > this.localStateThreshold) {
                this.flushToFile();
            }
        }

        @Override
        public void sync() throws IOException {
            this.outStream.sync();
        }

        public boolean isClosed() {
            return this.closed;
        }

        @Override
        public void close() {
            if (!this.closed) {
                this.closed = true;
                this.pos = this.writeBuffer.length;
                if (this.outStream != null) {
                    try {
                        this.outStream.close();
                    }
                    catch (Throwable throwable) {
                        LOG.warn("Could not close the state stream for {}.", (Object)this.statePath, (Object)throwable);
                    }
                    finally {
                        try {
                            this.fs.delete(this.statePath, false);
                        }
                        catch (Exception e) {
                            LOG.warn("Cannot delete closed and discarded state stream for {}.", (Object)this.statePath, (Object)e);
                        }
                    }
                }
            }
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        @Nullable
        public StreamStateHandle closeAndGetHandle() throws IOException {
            if (this.outStream == null && this.pos == 0) {
                return null;
            }
            FsCheckpointStateOutputStream fsCheckpointStateOutputStream = this;
            synchronized (fsCheckpointStateOutputStream) {
                if (this.closed) throw new IOException("Stream has already been closed and discarded.");
                if (this.outStream == null && this.pos <= this.localStateThreshold) {
                    this.closed = true;
                    byte[] bytes = Arrays.copyOf(this.writeBuffer, this.pos);
                    this.pos = this.writeBuffer.length;
                    return new ByteStreamStateHandle(this.createStatePath().toString(), bytes);
                }
                try {
                    this.flushToFile();
                    this.pos = this.writeBuffer.length;
                    long size = -1L;
                    try {
                        size = this.outStream.getPos();
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                    this.sync();
                    this.outStream.close();
                    FileStateHandle fileStateHandle = this.allowRelativePaths ? new RelativeFileStateHandle(this.statePath, this.relativeStatePath, size) : new FileStateHandle(this.statePath, size);
                    return fileStateHandle;
                }
                catch (Exception exception) {
                    try {
                        if (this.statePath == null) throw new IOException("Could not flush to file and close the file system output stream to " + this.statePath + " in order to obtain the stream state handle", exception);
                        this.fs.delete(this.statePath, false);
                        throw new IOException("Could not flush to file and close the file system output stream to " + this.statePath + " in order to obtain the stream state handle", exception);
                    }
                    catch (Exception deleteException) {
                        LOG.warn("Could not delete the checkpoint stream file {}.", (Object)this.statePath, (Object)deleteException);
                    }
                    throw new IOException("Could not flush to file and close the file system output stream to " + this.statePath + " in order to obtain the stream state handle", exception);
                }
                finally {
                    this.closed = true;
                }
            }
        }

        private Path createStatePath() {
            String fileName;
            this.relativeStatePath = fileName = UUID.randomUUID().toString();
            return new Path(this.basePath, fileName);
        }

        private void createStream() throws IOException {
            Exception latestException = null;
            for (int attempt = 0; attempt < 10; ++attempt) {
                try {
                    OutputStreamAndPath streamAndPath = EntropyInjector.createEntropyAware(this.fs, this.createStatePath(), FileSystem.WriteMode.NO_OVERWRITE);
                    this.outStream = streamAndPath.stream();
                    this.statePath = streamAndPath.path();
                    return;
                }
                catch (Exception e) {
                    latestException = e;
                    continue;
                }
            }
            throw new IOException("Could not open output stream for state backend", latestException);
        }
    }
}

