/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.instance;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.JobID;
import org.apache.flink.api.common.time.Time;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.clusterframework.types.AllocationID;
import org.apache.flink.runtime.clusterframework.types.ResourceID;
import org.apache.flink.runtime.clusterframework.types.ResourceProfile;
import org.apache.flink.runtime.concurrent.FutureUtils;
import org.apache.flink.runtime.instance.SimpleSlot;
import org.apache.flink.runtime.instance.Slot;
import org.apache.flink.runtime.instance.SlotPoolGateway;
import org.apache.flink.runtime.instance.SlotProvider;
import org.apache.flink.runtime.jobmanager.scheduler.Locality;
import org.apache.flink.runtime.jobmanager.scheduler.NoResourceAvailableException;
import org.apache.flink.runtime.jobmanager.scheduler.ScheduledUnit;
import org.apache.flink.runtime.jobmanager.slots.AllocatedSlot;
import org.apache.flink.runtime.jobmanager.slots.SlotAndLocality;
import org.apache.flink.runtime.jobmanager.slots.SlotOwner;
import org.apache.flink.runtime.jobmaster.JobMasterId;
import org.apache.flink.runtime.messages.Acknowledge;
import org.apache.flink.runtime.resourcemanager.ResourceManagerGateway;
import org.apache.flink.runtime.resourcemanager.SlotRequest;
import org.apache.flink.runtime.rpc.RpcEndpoint;
import org.apache.flink.runtime.rpc.RpcService;
import org.apache.flink.runtime.taskexecutor.slot.SlotOffer;
import org.apache.flink.runtime.taskmanager.TaskManagerLocation;
import org.apache.flink.runtime.util.clock.Clock;
import org.apache.flink.runtime.util.clock.SystemClock;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SlotPool
extends RpcEndpoint
implements SlotPoolGateway {
    static final Logger LOG = LoggerFactory.getLogger(SlotPool.class);
    private static final Time DEFAULT_SLOT_REQUEST_TIMEOUT = Time.minutes((long)5L);
    private static final Time DEFAULT_RM_ALLOCATION_TIMEOUT = Time.minutes((long)10L);
    private static final Time DEFAULT_RM_REQUEST_TIMEOUT = Time.seconds((long)10L);
    private final JobID jobId;
    private final ProviderAndOwner providerAndOwner;
    private final HashSet<ResourceID> registeredTaskManagers;
    private final AllocatedSlots allocatedSlots;
    private final AvailableSlots availableSlots;
    private final HashMap<AllocationID, PendingRequest> pendingRequests;
    private final HashMap<AllocationID, PendingRequest> waitingForResourceManager;
    private final Time resourceManagerRequestsTimeout;
    private final Time resourceManagerAllocationTimeout;
    private final Clock clock;
    private JobMasterId jobMasterId;
    private ResourceManagerGateway resourceManagerGateway;
    private String jobManagerAddress;

    public SlotPool(RpcService rpcService, JobID jobId) {
        this(rpcService, jobId, SystemClock.getInstance(), DEFAULT_SLOT_REQUEST_TIMEOUT, DEFAULT_RM_ALLOCATION_TIMEOUT, DEFAULT_RM_REQUEST_TIMEOUT);
    }

    public SlotPool(RpcService rpcService, JobID jobId, Clock clock, Time slotRequestTimeout, Time resourceManagerAllocationTimeout, Time resourceManagerRequestTimeout) {
        super(rpcService);
        this.jobId = (JobID)Preconditions.checkNotNull((Object)jobId);
        this.clock = (Clock)Preconditions.checkNotNull((Object)clock);
        this.resourceManagerRequestsTimeout = (Time)Preconditions.checkNotNull((Object)resourceManagerRequestTimeout);
        this.resourceManagerAllocationTimeout = (Time)Preconditions.checkNotNull((Object)resourceManagerAllocationTimeout);
        this.registeredTaskManagers = new HashSet();
        this.allocatedSlots = new AllocatedSlots();
        this.availableSlots = new AvailableSlots();
        this.pendingRequests = new HashMap();
        this.waitingForResourceManager = new HashMap();
        this.providerAndOwner = new ProviderAndOwner(this.getSelfGateway(SlotPoolGateway.class), slotRequestTimeout);
        this.jobMasterId = null;
        this.resourceManagerGateway = null;
        this.jobManagerAddress = null;
    }

    @Override
    public void start() {
        throw new UnsupportedOperationException("Should never call start() without leader ID");
    }

    public void start(JobMasterId jobMasterId, String newJobManagerAddress) throws Exception {
        this.jobMasterId = (JobMasterId)((Object)Preconditions.checkNotNull((Object)((Object)jobMasterId)));
        this.jobManagerAddress = (String)Preconditions.checkNotNull((Object)newJobManagerAddress);
        try {
            super.start();
        }
        catch (Exception e) {
            throw new RuntimeException("This should never happen", e);
        }
    }

    @Override
    public void suspend() {
        this.validateRunsInMainThread();
        this.stop();
        this.jobMasterId = null;
        this.resourceManagerGateway = null;
        this.availableSlots.clear();
        this.allocatedSlots.clear();
        this.pendingRequests.clear();
    }

    public SlotOwner getSlotOwner() {
        return this.providerAndOwner;
    }

    public SlotProvider getSlotProvider() {
        return this.providerAndOwner;
    }

    @Override
    public void connectToResourceManager(ResourceManagerGateway resourceManagerGateway) {
        this.resourceManagerGateway = (ResourceManagerGateway)Preconditions.checkNotNull((Object)resourceManagerGateway);
        for (PendingRequest pending : this.waitingForResourceManager.values()) {
            this.requestSlotFromResourceManager(pending.allocationID(), pending.getFuture(), pending.resourceProfile());
        }
        this.waitingForResourceManager.clear();
    }

    @Override
    public void disconnectResourceManager() {
        this.resourceManagerGateway = null;
    }

    @Override
    public CompletableFuture<SimpleSlot> allocateSlot(ScheduledUnit task, ResourceProfile resources, Iterable<TaskManagerLocation> locationPreferences, Time timeout) {
        return this.internalAllocateSlot(task, resources, locationPreferences);
    }

    @Override
    public void returnAllocatedSlot(Slot slot) {
        this.internalReturnAllocatedSlot(slot);
    }

    CompletableFuture<SimpleSlot> internalAllocateSlot(ScheduledUnit task, ResourceProfile resources, Iterable<TaskManagerLocation> locationPreferences) {
        SlotAndLocality slotFromPool = this.availableSlots.poll(resources, locationPreferences);
        if (slotFromPool != null) {
            SimpleSlot slot = this.createSimpleSlot(slotFromPool.slot(), slotFromPool.locality());
            this.allocatedSlots.add(slot);
            return CompletableFuture.completedFuture(slot);
        }
        AllocationID allocationID = new AllocationID();
        CompletableFuture<SimpleSlot> future = new CompletableFuture<SimpleSlot>();
        if (this.resourceManagerGateway == null) {
            this.stashRequestWaitingForResourceManager(allocationID, resources, future);
        } else {
            this.requestSlotFromResourceManager(allocationID, future, resources);
        }
        return future;
    }

    private void requestSlotFromResourceManager(AllocationID allocationID, CompletableFuture<SimpleSlot> future, ResourceProfile resources) {
        LOG.info("Requesting slot with profile {} from resource manager (request = {}).", (Object)resources, (Object)allocationID);
        this.pendingRequests.put(allocationID, new PendingRequest(allocationID, future, resources));
        CompletableFuture<Acknowledge> rmResponse = this.resourceManagerGateway.requestSlot(this.jobMasterId, new SlotRequest(this.jobId, allocationID, resources, this.jobManagerAddress), this.resourceManagerRequestsTimeout);
        CompletionStage slotRequestProcessingFuture = rmResponse.thenAcceptAsync(value -> this.slotRequestToResourceManagerSuccess(allocationID), (Executor)this.getMainThreadExecutor());
        ((CompletableFuture)slotRequestProcessingFuture).whenCompleteAsync((v, failure) -> {
            if (failure != null) {
                this.slotRequestToResourceManagerFailed(allocationID, (Throwable)failure);
            }
        }, (Executor)this.getMainThreadExecutor());
    }

    private void slotRequestToResourceManagerSuccess(final AllocationID allocationID) {
        this.scheduleRunAsync(new Runnable(){

            @Override
            public void run() {
                SlotPool.this.checkTimeoutSlotAllocation(allocationID);
            }
        }, this.resourceManagerAllocationTimeout);
    }

    private void slotRequestToResourceManagerFailed(AllocationID allocationID, Throwable failure) {
        PendingRequest request = this.pendingRequests.remove((Object)allocationID);
        if (request != null) {
            request.getFuture().completeExceptionally((Throwable)((Object)new NoResourceAvailableException("No pooled slot available and request to ResourceManager for new slot failed", failure)));
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Unregistered slot request {} failed.", (Object)allocationID, (Object)failure);
        }
    }

    private void checkTimeoutSlotAllocation(AllocationID allocationID) {
        PendingRequest request = this.pendingRequests.remove((Object)allocationID);
        if (request != null && !request.getFuture().isDone()) {
            request.getFuture().completeExceptionally(new TimeoutException("Slot allocation request timed out"));
        }
    }

    private void stashRequestWaitingForResourceManager(final AllocationID allocationID, ResourceProfile resources, CompletableFuture<SimpleSlot> future) {
        LOG.info("Cannot serve slot request, no ResourceManager connected. Adding as pending request {}", (Object)allocationID);
        this.waitingForResourceManager.put(allocationID, new PendingRequest(allocationID, future, resources));
        this.scheduleRunAsync(new Runnable(){

            @Override
            public void run() {
                SlotPool.this.checkTimeoutRequestWaitingForResourceManager(allocationID);
            }
        }, this.resourceManagerRequestsTimeout);
    }

    private void checkTimeoutRequestWaitingForResourceManager(AllocationID allocationID) {
        PendingRequest request = this.waitingForResourceManager.remove((Object)allocationID);
        if (request != null && !request.getFuture().isDone()) {
            request.getFuture().completeExceptionally((Throwable)((Object)new NoResourceAvailableException("No slot available and no connection to Resource Manager established.")));
        }
    }

    private void internalReturnAllocatedSlot(Slot slot) {
        Preconditions.checkNotNull((Object)slot);
        Preconditions.checkArgument((!slot.isAlive() ? 1 : 0) != 0, (Object)"slot is still alive");
        Preconditions.checkArgument((slot.getOwner() == this.providerAndOwner ? 1 : 0) != 0, (Object)"slot belongs to the wrong pool.");
        if (slot.markReleased()) {
            if (this.allocatedSlots.remove(slot)) {
                AllocatedSlot taskManagerSlot = slot.getAllocatedSlot();
                PendingRequest pendingRequest = this.pollMatchingPendingRequest(taskManagerSlot);
                if (pendingRequest != null) {
                    LOG.debug("Fulfilling pending request [{}] early with returned slot [{}]", (Object)pendingRequest.allocationID(), (Object)taskManagerSlot.getSlotAllocationId());
                    SimpleSlot newSlot = this.createSimpleSlot(taskManagerSlot, Locality.UNKNOWN);
                    this.allocatedSlots.add(newSlot);
                    pendingRequest.getFuture().complete(newSlot);
                } else {
                    LOG.debug("Adding returned slot [{}] to available slots", (Object)taskManagerSlot.getSlotAllocationId());
                    this.availableSlots.add(taskManagerSlot, this.clock.relativeTimeMillis());
                }
            } else {
                LOG.debug("Returned slot's allocation has been failed. Dropping slot.");
            }
        }
    }

    private PendingRequest pollMatchingPendingRequest(AllocatedSlot slot) {
        ResourceProfile slotResources = slot.getResourceProfile();
        for (PendingRequest request : this.pendingRequests.values()) {
            if (!slotResources.isMatching(request.resourceProfile())) continue;
            this.pendingRequests.remove((Object)request.allocationID());
            return request;
        }
        for (PendingRequest request : this.waitingForResourceManager.values()) {
            if (!slotResources.isMatching(request.resourceProfile())) continue;
            this.waitingForResourceManager.remove((Object)request.allocationID());
            return request;
        }
        return null;
    }

    @Override
    public CompletableFuture<Collection<SlotOffer>> offerSlots(Collection<Tuple2<AllocatedSlot, SlotOffer>> offers) {
        this.validateRunsInMainThread();
        List acceptedSlotOffers = offers.stream().map(offer -> {
            CompletionStage acceptedSlotOffer = this.offerSlot((AllocatedSlot)offer.f0).thenApply(acceptedSlot -> {
                if (acceptedSlot.booleanValue()) {
                    return Optional.of(offer.f1);
                }
                return Optional.empty();
            });
            return acceptedSlotOffer;
        }).collect(Collectors.toList());
        FutureUtils.ConjunctFuture optionalSlotOffers = FutureUtils.combineAll(acceptedSlotOffers);
        CompletionStage resultingSlotOffers = optionalSlotOffers.thenApply(collection -> {
            Collection slotOffers = collection.stream().flatMap(opt -> opt.map(Stream::of).orElseGet(Stream::empty)).collect(Collectors.toList());
            return slotOffers;
        });
        return resultingSlotOffers;
    }

    @Override
    public CompletableFuture<Boolean> offerSlot(AllocatedSlot slot) {
        this.validateRunsInMainThread();
        ResourceID resourceID = slot.getTaskManagerId();
        AllocationID allocationID = slot.getSlotAllocationId();
        if (!this.registeredTaskManagers.contains(resourceID)) {
            LOG.debug("Received outdated slot offering [{}] from unregistered TaskManager: {}", (Object)slot.getSlotAllocationId(), (Object)slot);
            return CompletableFuture.completedFuture(false);
        }
        if (this.allocatedSlots.contains(allocationID) || this.availableSlots.contains(allocationID)) {
            LOG.debug("Received repeated offer for slot [{}]. Ignoring.", (Object)allocationID);
            return CompletableFuture.completedFuture(true);
        }
        PendingRequest pendingRequest = this.pendingRequests.remove((Object)allocationID);
        if (pendingRequest != null) {
            SimpleSlot resultSlot = this.createSimpleSlot(slot, Locality.UNKNOWN);
            pendingRequest.getFuture().complete(resultSlot);
            this.allocatedSlots.add(resultSlot);
        } else {
            this.availableSlots.add(slot, this.clock.relativeTimeMillis());
        }
        return CompletableFuture.completedFuture(true);
    }

    @Override
    public void failAllocation(AllocationID allocationID, Exception cause) {
        PendingRequest pendingRequest = this.pendingRequests.remove((Object)allocationID);
        if (pendingRequest != null) {
            LOG.debug("Failed pending request [{}] with ", (Object)allocationID, (Object)cause);
            pendingRequest.getFuture().completeExceptionally(cause);
        } else if (this.availableSlots.tryRemove(allocationID)) {
            LOG.debug("Failed available slot [{}] with ", (Object)allocationID, (Object)cause);
        } else {
            Slot slot = this.allocatedSlots.remove(allocationID);
            if (slot != null) {
                slot.releaseSlot();
            } else {
                LOG.debug("Outdated request to fail slot [{}] with ", (Object)allocationID, (Object)cause);
            }
        }
    }

    @Override
    public void registerTaskManager(ResourceID resourceID) {
        this.registeredTaskManagers.add(resourceID);
    }

    @Override
    public CompletableFuture<Acknowledge> releaseTaskManager(ResourceID resourceID) {
        if (this.registeredTaskManagers.remove(resourceID)) {
            this.availableSlots.removeAllForTaskManager(resourceID);
            Set<Slot> allocatedSlotsForResource = this.allocatedSlots.removeSlotsForTaskManager(resourceID);
            for (Slot slot : allocatedSlotsForResource) {
                slot.releaseSlot();
            }
        }
        return CompletableFuture.completedFuture(Acknowledge.get());
    }

    private SimpleSlot createSimpleSlot(AllocatedSlot slot, Locality locality) {
        SimpleSlot result = new SimpleSlot(slot, this.providerAndOwner, slot.getSlotNumber());
        if (locality != null) {
            result.setLocality(locality);
        }
        return result;
    }

    @VisibleForTesting
    AllocatedSlots getAllocatedSlots() {
        return this.allocatedSlots;
    }

    private static class SlotAndTimestamp {
        private final AllocatedSlot slot;
        private final long timestamp;

        SlotAndTimestamp(AllocatedSlot slot, long timestamp) {
            this.slot = slot;
            this.timestamp = timestamp;
        }

        public AllocatedSlot slot() {
            return this.slot;
        }

        public long timestamp() {
            return this.timestamp;
        }

        public String toString() {
            return this.slot + " @ " + this.timestamp;
        }
    }

    private static class PendingRequest {
        private final AllocationID allocationID;
        private final CompletableFuture<SimpleSlot> future;
        private final ResourceProfile resourceProfile;

        PendingRequest(AllocationID allocationID, CompletableFuture<SimpleSlot> future, ResourceProfile resourceProfile) {
            this.allocationID = allocationID;
            this.future = future;
            this.resourceProfile = resourceProfile;
        }

        public AllocationID allocationID() {
            return this.allocationID;
        }

        public CompletableFuture<SimpleSlot> getFuture() {
            return this.future;
        }

        public ResourceProfile resourceProfile() {
            return this.resourceProfile;
        }
    }

    private static class ProviderAndOwner
    implements SlotOwner,
    SlotProvider {
        private final SlotPoolGateway gateway;
        private final Time timeout;

        ProviderAndOwner(SlotPoolGateway gateway, Time timeout) {
            this.gateway = gateway;
            this.timeout = timeout;
        }

        @Override
        public boolean returnAllocatedSlot(Slot slot) {
            this.gateway.returnAllocatedSlot(slot);
            return true;
        }

        @Override
        public CompletableFuture<SimpleSlot> allocateSlot(ScheduledUnit task, boolean allowQueued, Collection<TaskManagerLocation> preferredLocations) {
            return this.gateway.allocateSlot(task, ResourceProfile.UNKNOWN, preferredLocations, this.timeout);
        }
    }

    static class AvailableSlots {
        private final HashMap<ResourceID, Set<AllocatedSlot>> availableSlotsByTaskManager = new HashMap();
        private final HashMap<String, Set<AllocatedSlot>> availableSlotsByHost = new HashMap();
        private final HashMap<AllocationID, SlotAndTimestamp> availableSlots = new HashMap();

        AvailableSlots() {
        }

        void add(AllocatedSlot slot, long timestamp) {
            Set<AllocatedSlot> slotsForHost;
            Preconditions.checkNotNull((Object)slot);
            SlotAndTimestamp previous = this.availableSlots.put(slot.getSlotAllocationId(), new SlotAndTimestamp(slot, timestamp));
            if (previous == null) {
                ResourceID resourceID = slot.getTaskManagerLocation().getResourceID();
                String host = slot.getTaskManagerLocation().getFQDNHostname();
                Set<AllocatedSlot> slotsForTaskManager = this.availableSlotsByTaskManager.get(resourceID);
                if (slotsForTaskManager == null) {
                    slotsForTaskManager = new HashSet<AllocatedSlot>();
                    this.availableSlotsByTaskManager.put(resourceID, slotsForTaskManager);
                }
                slotsForTaskManager.add(slot);
                slotsForHost = this.availableSlotsByHost.get(host);
                if (slotsForHost == null) {
                    slotsForHost = new HashSet<AllocatedSlot>();
                    this.availableSlotsByHost.put(host, slotsForHost);
                }
            } else {
                throw new IllegalStateException("slot already contained");
            }
            slotsForHost.add(slot);
        }

        boolean contains(AllocationID slotId) {
            return this.availableSlots.containsKey((Object)slotId);
        }

        SlotAndLocality poll(ResourceProfile resourceProfile, Iterable<TaskManagerLocation> locationPreferences) {
            if (this.availableSlots.isEmpty()) {
                return null;
            }
            boolean hadLocationPreference = false;
            if (locationPreferences != null) {
                for (TaskManagerLocation location : locationPreferences) {
                    hadLocationPreference = true;
                    Set<AllocatedSlot> onTaskManager = this.availableSlotsByTaskManager.get(location.getResourceID());
                    if (onTaskManager == null) continue;
                    for (AllocatedSlot candidate : onTaskManager) {
                        if (!candidate.getResourceProfile().isMatching(resourceProfile)) continue;
                        this.remove(candidate.getSlotAllocationId());
                        return new SlotAndLocality(candidate, Locality.LOCAL);
                    }
                }
                for (TaskManagerLocation location : locationPreferences) {
                    Set<AllocatedSlot> onHost = this.availableSlotsByHost.get(location.getFQDNHostname());
                    if (onHost == null) continue;
                    for (AllocatedSlot candidate : onHost) {
                        if (!candidate.getResourceProfile().isMatching(resourceProfile)) continue;
                        this.remove(candidate.getSlotAllocationId());
                        return new SlotAndLocality(candidate, Locality.HOST_LOCAL);
                    }
                }
            }
            for (SlotAndTimestamp candidate : this.availableSlots.values()) {
                AllocatedSlot slot = candidate.slot();
                if (!slot.getResourceProfile().isMatching(resourceProfile)) continue;
                this.remove(slot.getSlotAllocationId());
                return new SlotAndLocality(slot, hadLocationPreference ? Locality.NON_LOCAL : Locality.UNCONSTRAINED);
            }
            return null;
        }

        void removeAllForTaskManager(ResourceID taskManager) {
            Set<AllocatedSlot> slotsForTm = this.availableSlotsByTaskManager.remove(taskManager);
            if (slotsForTm != null && slotsForTm.size() > 0) {
                String host = slotsForTm.iterator().next().getTaskManagerLocation().getFQDNHostname();
                Set<AllocatedSlot> slotsForHost = this.availableSlotsByHost.get(host);
                for (AllocatedSlot slot : slotsForTm) {
                    this.availableSlots.remove((Object)slot.getSlotAllocationId());
                    slotsForHost.remove(slot);
                }
                if (slotsForHost.isEmpty()) {
                    this.availableSlotsByHost.remove(host);
                }
            }
        }

        boolean tryRemove(AllocationID slotId) {
            SlotAndTimestamp sat = this.availableSlots.remove((Object)slotId);
            if (sat != null) {
                AllocatedSlot slot = sat.slot();
                ResourceID resourceID = slot.getTaskManagerLocation().getResourceID();
                String host = slot.getTaskManagerLocation().getFQDNHostname();
                Set<AllocatedSlot> slotsForTm = this.availableSlotsByTaskManager.get(resourceID);
                Set<AllocatedSlot> slotsForHost = this.availableSlotsByHost.get(host);
                slotsForTm.remove(slot);
                slotsForHost.remove(slot);
                if (slotsForTm.isEmpty()) {
                    this.availableSlotsByTaskManager.remove(resourceID);
                }
                if (slotsForHost.isEmpty()) {
                    this.availableSlotsByHost.remove(host);
                }
                return true;
            }
            return false;
        }

        private void remove(AllocationID slotId) throws IllegalStateException {
            if (!this.tryRemove(slotId)) {
                throw new IllegalStateException("slot not contained");
            }
        }

        @VisibleForTesting
        boolean containsTaskManager(ResourceID resourceID) {
            return this.availableSlotsByTaskManager.containsKey(resourceID);
        }

        @VisibleForTesting
        int size() {
            return this.availableSlots.size();
        }

        @VisibleForTesting
        void clear() {
            this.availableSlots.clear();
            this.availableSlotsByTaskManager.clear();
            this.availableSlotsByHost.clear();
        }
    }

    static class AllocatedSlots {
        private final Map<ResourceID, Set<Slot>> allocatedSlotsByTaskManager = new HashMap<ResourceID, Set<Slot>>();
        private final Map<AllocationID, Slot> allocatedSlotsById = new HashMap<AllocationID, Slot>();

        AllocatedSlots() {
        }

        void add(Slot slot) {
            this.allocatedSlotsById.put(slot.getAllocatedSlot().getSlotAllocationId(), slot);
            ResourceID resourceID = slot.getTaskManagerID();
            Set<Slot> slotsForTaskManager = this.allocatedSlotsByTaskManager.get(resourceID);
            if (slotsForTaskManager == null) {
                slotsForTaskManager = new HashSet<Slot>();
                this.allocatedSlotsByTaskManager.put(resourceID, slotsForTaskManager);
            }
            slotsForTaskManager.add(slot);
        }

        Slot get(AllocationID allocationID) {
            return this.allocatedSlotsById.get((Object)allocationID);
        }

        boolean contains(AllocationID slotAllocationId) {
            return this.allocatedSlotsById.containsKey((Object)slotAllocationId);
        }

        boolean remove(Slot slot) {
            return this.remove(slot.getAllocatedSlot().getSlotAllocationId()) != null;
        }

        Slot remove(AllocationID slotId) {
            Slot slot = this.allocatedSlotsById.remove((Object)slotId);
            if (slot != null) {
                ResourceID taskManagerId = slot.getTaskManagerID();
                Set<Slot> slotsForTM = this.allocatedSlotsByTaskManager.get(taskManagerId);
                slotsForTM.remove(slot);
                if (slotsForTM.isEmpty()) {
                    this.allocatedSlotsByTaskManager.remove(taskManagerId);
                }
                return slot;
            }
            return null;
        }

        Set<Slot> removeSlotsForTaskManager(ResourceID resourceID) {
            Set<Slot> slotsForTaskManager = this.allocatedSlotsByTaskManager.remove(resourceID);
            if (slotsForTaskManager != null) {
                for (Slot slot : slotsForTaskManager) {
                    this.allocatedSlotsById.remove((Object)slot.getAllocatedSlot().getSlotAllocationId());
                }
                return slotsForTaskManager;
            }
            return Collections.emptySet();
        }

        void clear() {
            this.allocatedSlotsById.clear();
            this.allocatedSlotsByTaskManager.clear();
        }

        @VisibleForTesting
        boolean containResource(ResourceID resourceID) {
            return this.allocatedSlotsByTaskManager.containsKey(resourceID);
        }

        @VisibleForTesting
        int size() {
            return this.allocatedSlotsById.size();
        }

        @VisibleForTesting
        Set<Slot> getSlotsForTaskManager(ResourceID resourceId) {
            if (this.allocatedSlotsByTaskManager.containsKey(resourceId)) {
                return this.allocatedSlotsByTaskManager.get(resourceId);
            }
            return Collections.emptySet();
        }
    }
}

