/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.blob;

import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.UUID;
import javax.annotation.Nullable;
import org.apache.flink.api.common.JobID;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.HighAvailabilityOptions;
import org.apache.flink.configuration.IllegalConfigurationException;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.blob.BlobKey;
import org.apache.flink.runtime.blob.BlobStore;
import org.apache.flink.runtime.blob.BlobStoreService;
import org.apache.flink.runtime.blob.FileSystemBlobStore;
import org.apache.flink.runtime.blob.VoidBlobStore;
import org.apache.flink.runtime.jobmanager.HighAvailabilityMode;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.StringUtils;
import org.slf4j.Logger;

public class BlobUtils {
    private static final String HASHING_ALGORITHM = "SHA-1";
    private static final String BLOB_FILE_PREFIX = "blob_";
    static final String JOB_DIR_PREFIX = "job_";
    static final String NO_JOB_DIR_PREFIX = "no_job";

    public static BlobStoreService createBlobStoreFromConfig(Configuration config) throws IOException {
        HighAvailabilityMode highAvailabilityMode = HighAvailabilityMode.fromConfig(config);
        if (highAvailabilityMode == HighAvailabilityMode.NONE) {
            return new VoidBlobStore();
        }
        if (highAvailabilityMode == HighAvailabilityMode.ZOOKEEPER) {
            return BlobUtils.createFileSystemBlobStore(config);
        }
        throw new IllegalConfigurationException("Unexpected high availability mode '" + (Object)((Object)highAvailabilityMode) + "'.");
    }

    private static BlobStoreService createFileSystemBlobStore(Configuration configuration) throws IOException {
        FileSystem fileSystem;
        Path path;
        String storagePath = configuration.getValue(HighAvailabilityOptions.HA_STORAGE_PATH);
        if (StringUtils.isNullOrWhitespaceOnly((String)storagePath)) {
            throw new IllegalConfigurationException("Configuration is missing the mandatory parameter: " + HighAvailabilityOptions.HA_STORAGE_PATH);
        }
        try {
            path = new Path(storagePath);
        }
        catch (Exception e) {
            throw new IOException("Invalid path for highly available storage (" + HighAvailabilityOptions.HA_STORAGE_PATH.key() + ')', e);
        }
        try {
            fileSystem = path.getFileSystem();
        }
        catch (Exception e) {
            throw new IOException("Could not create FileSystem for highly available storage (" + HighAvailabilityOptions.HA_STORAGE_PATH.key() + ')', e);
        }
        String clusterId = configuration.getValue(HighAvailabilityOptions.HA_CLUSTER_ID);
        storagePath = storagePath + "/" + clusterId;
        return new FileSystemBlobStore(fileSystem, storagePath);
    }

    static File initLocalStorageDirectory(String basePath) throws IOException {
        File baseDir = StringUtils.isNullOrWhitespaceOnly((String)basePath) ? new File(System.getProperty("java.io.tmpdir")) : new File(basePath);
        int MAX_ATTEMPTS = 10;
        for (int attempt = 0; attempt < 10; ++attempt) {
            File storageDir = new File(baseDir, String.format("blobStore-%s", UUID.randomUUID().toString()));
            if (!storageDir.mkdirs()) continue;
            return storageDir;
        }
        throw new IOException("Could not create storage directory for BLOB store in '" + baseDir + "'.");
    }

    static File getIncomingDirectory(File storageDir) throws IOException {
        File incomingDir = new File(storageDir, "incoming");
        BlobUtils.mkdirTolerateExisting(incomingDir);
        return incomingDir;
    }

    private static void mkdirTolerateExisting(File dir) throws IOException {
        if (!dir.mkdirs() && !dir.exists()) {
            throw new IOException("Cannot create directory '" + dir.getAbsolutePath() + "'.");
        }
    }

    static File getStorageLocation(File storageDir, @Nullable JobID jobId, BlobKey key) throws IOException {
        File file = new File(BlobUtils.getStorageLocationPath(storageDir.getAbsolutePath(), jobId, key));
        BlobUtils.mkdirTolerateExisting(file.getParentFile());
        return file;
    }

    static String getStorageLocationPath(String storageDir, @Nullable JobID jobId) {
        if (jobId == null) {
            return String.format("%s/%s", storageDir, NO_JOB_DIR_PREFIX);
        }
        return String.format("%s/%s%s", storageDir, JOB_DIR_PREFIX, jobId.toString());
    }

    static String getStorageLocationPath(String storageDir, @Nullable JobID jobId, BlobKey key) {
        if (jobId == null) {
            return String.format("%s/%s/%s%s", storageDir, NO_JOB_DIR_PREFIX, BLOB_FILE_PREFIX, key.toString());
        }
        return String.format("%s/%s%s/%s%s", storageDir, JOB_DIR_PREFIX, jobId.toString(), BLOB_FILE_PREFIX, key.toString());
    }

    static MessageDigest createMessageDigest() {
        try {
            return MessageDigest.getInstance(HASHING_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Cannot instantiate the message digest algorithm SHA-1", e);
        }
    }

    static Thread addShutdownHook(final Closeable service, final Logger logger) {
        Preconditions.checkNotNull((Object)service);
        Preconditions.checkNotNull((Object)logger);
        Thread shutdownHook = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    service.close();
                }
                catch (Throwable t) {
                    logger.error("Error during shutdown of blob service via JVM shutdown hook.", t);
                }
            }
        });
        try {
            Runtime.getRuntime().addShutdownHook(shutdownHook);
            return shutdownHook;
        }
        catch (IllegalStateException e) {
            return null;
        }
        catch (Throwable t) {
            logger.error("Cannot register shutdown hook that cleanly terminates the BLOB service.");
            return null;
        }
    }

    static void writeLength(int length, OutputStream outputStream) throws IOException {
        byte[] buf = new byte[]{(byte)(length & 0xFF), (byte)(length >> 8 & 0xFF), (byte)(length >> 16 & 0xFF), (byte)(length >> 24 & 0xFF)};
        outputStream.write(buf, 0, 4);
    }

    static int readLength(InputStream inputStream) throws IOException {
        int bytesRead;
        int read;
        byte[] buf = new byte[4];
        for (bytesRead = 0; bytesRead < 4; bytesRead += read) {
            read = inputStream.read(buf, bytesRead, 4 - bytesRead);
            if (read >= 0) continue;
            throw new EOFException("Read an incomplete length");
        }
        bytesRead = buf[0] & 0xFF;
        bytesRead |= (buf[1] & 0xFF) << 8;
        bytesRead |= (buf[2] & 0xFF) << 16;
        return bytesRead |= (buf[3] & 0xFF) << 24;
    }

    static void readFully(InputStream inputStream, byte[] buf, int off, int len, String type) throws IOException {
        int read;
        for (int bytesRead = 0; bytesRead < len; bytesRead += read) {
            read = inputStream.read(buf, off + bytesRead, len - bytesRead);
            if (read >= 0) continue;
            throw new EOFException("Received an incomplete " + type);
        }
    }

    static void closeSilently(Socket socket, Logger LOG) {
        if (socket != null) {
            try {
                socket.close();
            }
            catch (Throwable t) {
                LOG.debug("Exception while closing BLOB server connection socket.", t);
            }
        }
    }

    private BlobUtils() {
        throw new RuntimeException();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void moveTempFileToStore(File incomingFile, @Nullable JobID jobId, BlobKey blobKey, File storageFile, Logger log, @Nullable BlobStore blobStore) throws IOException {
        block10: {
            try {
                if (!storageFile.exists()) {
                    try {
                        Files.move(incomingFile.toPath(), storageFile.toPath(), new CopyOption[0]);
                        incomingFile = null;
                    }
                    catch (FileAlreadyExistsException ignored) {
                        log.warn("Detected concurrent file modifications. This should only happen if multipleBlobServer use the same storage directory.");
                    }
                    if (blobStore != null) {
                        blobStore.put(storageFile, jobId, blobKey);
                    }
                } else {
                    log.warn("File upload for an existing file with key {} for job {}. This may indicate a duplicate upload or a hash collision. Ignoring newest upload.", (Object)blobKey, (Object)jobId);
                }
                if ((storageFile = null) == null || storageFile.delete() || !storageFile.exists()) break block10;
            }
            catch (Throwable throwable) {
                if (storageFile != null && !storageFile.delete() && storageFile.exists()) {
                    log.warn("Could not delete the storage file {}.", (Object)storageFile);
                }
                if (incomingFile != null && !incomingFile.delete() && incomingFile.exists()) {
                    log.warn("Could not delete the staging file {} for blob key {} and job {}.", new Object[]{incomingFile, blobKey, jobId});
                }
                throw throwable;
            }
            log.warn("Could not delete the storage file {}.", (Object)storageFile);
        }
        if (incomingFile != null && !incomingFile.delete() && incomingFile.exists()) {
            log.warn("Could not delete the staging file {} for blob key {} and job {}.", new Object[]{incomingFile, blobKey, jobId});
        }
    }
}

