/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.blob;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;
import org.apache.flink.api.common.JobID;
import org.apache.flink.configuration.BlobServerOptions;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.CoreOptions;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.local.LocalFileSystem;
import org.apache.flink.runtime.blob.BlobKey;
import org.apache.flink.runtime.blob.BlobStore;
import org.apache.flink.runtime.blob.BlobUtils;
import org.apache.flink.runtime.blob.FileSystemBlobStore;
import org.apache.flink.runtime.blob.PermanentBlobKey;
import org.apache.flink.runtime.blob.TestingBlobUtils;
import org.apache.flink.runtime.blob.TransientBlobKey;
import org.apache.flink.testutils.junit.utils.TempDirUtils;
import org.apache.flink.util.FileUtils;
import org.apache.flink.util.Reference;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class BlobUtilsTest {
    private static final Logger LOG = LoggerFactory.getLogger(BlobUtilsTest.class);
    @TempDir
    private Path tempDir;

    BlobUtilsTest() {
    }

    @Test
    void testDefaultBlobStorageDirectory() throws IOException {
        Configuration config = new Configuration();
        String blobStorageDir = TempDirUtils.newFolder((Path)this.tempDir).getAbsolutePath();
        config.set(BlobServerOptions.STORAGE_DIRECTORY, (Object)blobStorageDir);
        config.set(CoreOptions.TMP_DIRS, (Object)TempDirUtils.newFolder((Path)this.tempDir).getAbsolutePath());
        File dir = (File)BlobUtils.createBlobStorageDirectory((Configuration)config, null).deref();
        Assertions.assertThat((String)dir.getAbsolutePath()).startsWith((CharSequence)blobStorageDir);
    }

    @Test
    void testTaskManagerFallbackBlobStorageDirectory1() throws IOException {
        Configuration config = new Configuration();
        File fallbackDirectory = TempDirUtils.newFile((Path)this.tempDir, (String)"foobar");
        File dir = (File)BlobUtils.createBlobStorageDirectory((Configuration)config, (Reference)Reference.borrowed((Object)fallbackDirectory)).deref();
        Assertions.assertThat((File)dir).isEqualTo((Object)fallbackDirectory);
    }

    @Test
    void testBlobUtilsFailIfNoStorageDirectoryIsSpecified() {
        Assertions.assertThatThrownBy(() -> BlobUtils.createBlobStorageDirectory((Configuration)new Configuration(), null)).isInstanceOf(IOException.class);
    }

    @Test
    void testCheckAndDeleteCorruptedBlobsDeletesCorruptedBlobs() throws IOException {
        JobID jobId = new JobID();
        byte[] validContent = "valid".getBytes(StandardCharsets.UTF_8);
        PermanentBlobKey validPermanentBlobKey = TestingBlobUtils.writePermanentBlob(this.tempDir, jobId, validContent);
        TransientBlobKey validTransientBlobKey = TestingBlobUtils.writeTransientBlob(this.tempDir, jobId, validContent);
        PermanentBlobKey corruptedBlobKey = TestingBlobUtils.writePermanentBlob(this.tempDir, jobId, validContent);
        FileUtils.writeFileUtf8((File)new File(BlobUtils.getStorageLocationPath((String)this.tempDir.toString(), (JobID)jobId, (BlobKey)corruptedBlobKey)), (String)"corrupted");
        BlobUtils.checkAndDeleteCorruptedBlobs((Path)this.tempDir, (Logger)LOG);
        List blobKeys = BlobUtils.listBlobsInDirectory((Path)this.tempDir).stream().map(BlobUtils.Blob::getBlobKey).collect(Collectors.toList());
        Assertions.assertThat(blobKeys).containsExactlyInAnyOrder((Object[])new BlobKey[]{validPermanentBlobKey, validTransientBlobKey});
    }

    @Test
    void testMoveTempFileToStoreSucceeds() throws IOException {
        FileSystemBlobStore blobStore = new FileSystemBlobStore((FileSystem)new LocalFileSystem(), TempDirUtils.newFolder((Path)this.tempDir).toString());
        JobID jobId = new JobID();
        File storageFile = this.tempDir.resolve(UUID.randomUUID().toString()).toFile();
        File incomingFile = TempDirUtils.newFile((Path)this.tempDir);
        byte[] fileContent = new byte[]{1, 2, 3, 4};
        BlobKey blobKey = BlobKey.createKey((BlobKey.BlobType)BlobKey.BlobType.PERMANENT_BLOB, (byte[])BlobUtils.createMessageDigest().digest(fileContent));
        Files.write(incomingFile.toPath(), fileContent, new OpenOption[0]);
        BlobUtils.moveTempFileToStore((File)incomingFile, (JobID)jobId, (BlobKey)blobKey, (File)storageFile, (Logger)LOG, (BlobStore)blobStore);
        Assertions.assertThat((File)incomingFile).doesNotExist();
        Assertions.assertThat((File)storageFile).hasBinaryContent(fileContent);
        File blobStoreFile = this.tempDir.resolve(UUID.randomUUID().toString()).toFile();
        Assertions.assertThat((boolean)blobStore.get(jobId, blobKey, blobStoreFile)).isTrue();
        Assertions.assertThat((File)blobStoreFile).hasBinaryContent(fileContent);
    }

    @Test
    void testCleanupIfMoveTempFileToStoreFails() throws IOException {
        File storageFile = this.tempDir.resolve(UUID.randomUUID().toString()).toFile();
        File incomingFile = TempDirUtils.newFile((Path)this.tempDir);
        Files.write(incomingFile.toPath(), new byte[]{1, 2, 3, 4}, new OpenOption[0]);
        FileSystemBlobStore blobStore = new FileSystemBlobStore((FileSystem)new LocalFileSystem(), TempDirUtils.newFolder((Path)this.tempDir).toString());
        JobID jobId = new JobID();
        BlobKey blobKey = BlobKey.createKey((BlobKey.BlobType)BlobKey.BlobType.PERMANENT_BLOB);
        Assertions.assertThatThrownBy(() -> BlobUtils.internalMoveTempFileToStore((File)incomingFile, (JobID)jobId, (BlobKey)blobKey, (File)storageFile, (Logger)LOG, (BlobStore)blobStore, (source, target) -> {
            throw new IOException("Test Failure");
        })).isInstanceOf(IOException.class);
        Assertions.assertThatThrownBy(() -> blobStore.get(jobId, blobKey, this.tempDir.resolve(UUID.randomUUID().toString()).toFile())).isInstanceOf(FileNotFoundException.class);
        Assertions.assertThat((File)incomingFile).doesNotExist();
        Assertions.assertThat((File)storageFile).doesNotExist();
    }
}

