/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.buffer;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.flink.core.fs.CloseableRegistry;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.testutils.CheckedThread;
import org.apache.flink.core.testutils.OneShotLatch;
import org.apache.flink.runtime.execution.CancelTaskException;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferBuilder;
import org.apache.flink.runtime.io.network.buffer.BufferPool;
import org.apache.flink.runtime.io.network.buffer.NetworkBufferPool;
import org.apache.flink.util.Preconditions;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

class NetworkBufferPoolTest {
    NetworkBufferPoolTest() {
    }

    @Test
    void testCreatePoolAfterDestroy() {
        int bufferSize = 128;
        int numBuffers = 10;
        NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        Assertions.assertThat((int)globalPool.getNumberOfRegisteredBufferPools()).isZero();
        globalPool.destroy();
        Assertions.assertThat((boolean)globalPool.isDestroyed()).isTrue();
        Assertions.assertThatThrownBy(() -> globalPool.createBufferPool(2, 2)).isInstanceOf(IllegalStateException.class);
        Assertions.assertThatThrownBy(() -> globalPool.createBufferPool(2, 10)).isInstanceOf(IllegalStateException.class);
        Assertions.assertThatThrownBy(() -> globalPool.createBufferPool(2, Integer.MAX_VALUE)).isInstanceOf(IllegalStateException.class);
    }

    @Test
    void testMemoryUsageInTheContextOfMemoryPoolCreation() {
        int bufferSize = 128;
        int numBuffers = 10;
        NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        Assertions.assertThat((int)globalPool.getTotalNumberOfMemorySegments()).isEqualTo(10);
        Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).isEqualTo(10);
        Assertions.assertThat((int)globalPool.getNumberOfUsedMemorySegments()).isZero();
        Assertions.assertThat((long)globalPool.getTotalMemory()).isEqualTo(1280L);
        Assertions.assertThat((long)globalPool.getAvailableMemory()).isEqualTo(1280L);
        Assertions.assertThat((long)globalPool.getUsedMemory()).isZero();
    }

    @Test
    void testMemoryUsageInTheContextOfMemorySegmentAllocation() {
        int bufferSize = 128;
        int numBuffers = 10;
        NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        MemorySegment segment = globalPool.requestPooledMemorySegment();
        Assertions.assertThat((Object)segment).isNotNull();
        Assertions.assertThat((int)globalPool.getTotalNumberOfMemorySegments()).isEqualTo(10);
        Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).isEqualTo(9);
        Assertions.assertThat((int)globalPool.getNumberOfUsedMemorySegments()).isOne();
        Assertions.assertThat((long)globalPool.getTotalMemory()).isEqualTo(1280L);
        Assertions.assertThat((long)globalPool.getAvailableMemory()).isEqualTo(1152L);
        Assertions.assertThat((long)globalPool.getUsedMemory()).isEqualTo(128L);
    }

    @Test
    void testMemoryUsageInTheContextOfMemoryPoolDestruction() {
        int bufferSize = 128;
        int numBuffers = 10;
        NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        globalPool.destroy();
        Assertions.assertThat((int)globalPool.getTotalNumberOfMemorySegments()).isZero();
        Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).isZero();
        Assertions.assertThat((int)globalPool.getNumberOfUsedMemorySegments()).isZero();
        Assertions.assertThat((long)globalPool.getTotalMemory()).isZero();
        Assertions.assertThat((long)globalPool.getAvailableMemory()).isZero();
        Assertions.assertThat((long)globalPool.getUsedMemory()).isZero();
    }

    @Test
    void testDestroyAll() throws IOException {
        NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        BufferPool fixedPool = globalPool.createBufferPool(2, 2);
        BufferPool boundedPool = globalPool.createBufferPool(1, 1);
        BufferPool nonFixedPool = globalPool.createBufferPool(5, Integer.MAX_VALUE);
        Assertions.assertThat((int)fixedPool.getNumberOfRequiredMemorySegments()).isEqualTo(2);
        Assertions.assertThat((int)boundedPool.getNumberOfRequiredMemorySegments()).isOne();
        Assertions.assertThat((int)nonFixedPool.getNumberOfRequiredMemorySegments()).isEqualTo(5);
        ArrayList<Buffer> buffers = new ArrayList<Buffer>(globalPool.getTotalNumberOfMemorySegments());
        block0: for (int i = 0; i < 10; ++i) {
            for (BufferPool bp : new BufferPool[]{fixedPool, boundedPool, nonFixedPool}) {
                Buffer buffer = bp.requestBuffer();
                if (buffer == null) continue;
                Assertions.assertThat((Object)buffer.getMemorySegment()).isNotNull();
                buffers.add(buffer);
                continue block0;
            }
        }
        Assertions.assertThat(buffers).hasSize(globalPool.getTotalNumberOfMemorySegments());
        Assertions.assertThat((Object)fixedPool.requestBuffer()).isNull();
        Assertions.assertThat((Object)boundedPool.requestBuffer()).isNull();
        Assertions.assertThat((Object)nonFixedPool.requestBuffer()).isNull();
        globalPool.destroyAllBufferPools();
        Assertions.assertThat((boolean)globalPool.isDestroyed()).isFalse();
        Assertions.assertThat((boolean)fixedPool.isDestroyed()).isTrue();
        Assertions.assertThat((boolean)boundedPool.isDestroyed()).isTrue();
        Assertions.assertThat((boolean)nonFixedPool.isDestroyed()).isTrue();
        Assertions.assertThat((int)globalPool.getNumberOfRegisteredBufferPools()).isZero();
        Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).isZero();
        for (Buffer b : buffers) {
            b.recycleBuffer();
        }
        Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).isEqualTo(globalPool.getTotalNumberOfMemorySegments());
        Assertions.assertThatThrownBy(() -> ((BufferPool)fixedPool).requestBuffer()).isInstanceOf(CancelTaskException.class);
        Assertions.assertThatThrownBy(() -> ((BufferPool)boundedPool).requestBuffer()).isInstanceOf(CancelTaskException.class);
        Assertions.assertThatThrownBy(() -> ((BufferPool)nonFixedPool).requestBuffer()).isInstanceOf(CancelTaskException.class);
        Assertions.assertThat((Object)globalPool.createBufferPool(10, Integer.MAX_VALUE)).isNotNull();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testRequestMemorySegmentsLessThanTotalBuffers() throws IOException {
        int numBuffers = 10;
        NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        List memorySegments = Collections.emptyList();
        try {
            memorySegments = globalPool.requestUnpooledMemorySegments(5);
            Assertions.assertThat((List)memorySegments).hasSize(5);
            globalPool.recycleUnpooledMemorySegments((Collection)memorySegments);
            memorySegments.clear();
            Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).isEqualTo(10);
        }
        finally {
            globalPool.recycleUnpooledMemorySegments((Collection)memorySegments);
            globalPool.destroy();
        }
    }

    @Test
    void testRequestMemorySegmentsMoreThanTotalBuffers() {
        int numBuffers = 10;
        NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        try {
            Assertions.assertThatThrownBy(() -> globalPool.requestUnpooledMemorySegments(11)).isInstanceOf(IOException.class);
            Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).isEqualTo(10);
        }
        finally {
            globalPool.destroy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testInsufficientNumberOfBuffers() throws Exception {
        int numberOfSegmentsToRequest = 5;
        NetworkBufferPool globalPool = new NetworkBufferPool(5, 128);
        try {
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isDone();
            List segments1 = globalPool.requestUnpooledMemorySegments(5);
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isNotDone();
            Assertions.assertThat((List)segments1).hasSize(5);
            Assertions.assertThatThrownBy(() -> globalPool.requestUnpooledMemorySegments(1)).hasMessageContaining("Insufficient number of network buffers").isInstanceOf(IOException.class);
            CompletableFuture availableFuture = globalPool.getAvailableFuture();
            globalPool.recycleUnpooledMemorySegments((Collection)segments1);
            Assertions.assertThat((CompletableFuture)availableFuture).isDone();
            List segments2 = globalPool.requestUnpooledMemorySegments(5);
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isNotDone();
            Assertions.assertThat((List)segments2).hasSize(5);
        }
        finally {
            globalPool.destroy();
        }
    }

    @Test
    void testEmptyPoolSegmentsUsage() throws IOException {
        try (CloseableRegistry closeableRegistry = new CloseableRegistry();){
            NetworkBufferPool globalPool = new NetworkBufferPool(0, 128);
            closeableRegistry.registerCloseable(() -> ((NetworkBufferPool)globalPool).destroy());
            Assertions.assertThat((int)globalPool.getEstimatedRequestedSegmentsUsage()).isZero();
        }
    }

    @Test
    void testSegmentsUsage() throws IOException {
        try (CloseableRegistry closeableRegistry = new CloseableRegistry();){
            NetworkBufferPool globalPool = new NetworkBufferPool(50, 128);
            closeableRegistry.registerCloseable(() -> ((NetworkBufferPool)globalPool).destroy());
            BufferPool bufferPool1 = globalPool.createBufferPool(10, 20);
            Assertions.assertThat((long)globalPool.getEstimatedNumberOfRequestedMemorySegments()).isEqualTo(20L);
            Assertions.assertThat((int)globalPool.getEstimatedRequestedSegmentsUsage()).isEqualTo(40);
            Assertions.assertThat((Optional)globalPool.getUsageWarning()).isEmpty();
            closeableRegistry.registerCloseable(() -> ((BufferPool)globalPool.createBufferPool(5, Integer.MAX_VALUE)).lazyDestroy());
            Assertions.assertThat((long)globalPool.getEstimatedNumberOfRequestedMemorySegments()).isEqualTo(30L);
            Assertions.assertThat((int)globalPool.getEstimatedRequestedSegmentsUsage()).isEqualTo(60);
            Assertions.assertThat((Optional)globalPool.getUsageWarning()).isEmpty();
            closeableRegistry.registerCloseable(() -> ((BufferPool)globalPool.createBufferPool(10, 30)).lazyDestroy());
            Assertions.assertThat((long)globalPool.getEstimatedNumberOfRequestedMemorySegments()).isEqualTo(60L);
            Assertions.assertThat((int)globalPool.getEstimatedRequestedSegmentsUsage()).isEqualTo(120);
            Assertions.assertThat((Optional)globalPool.getUsageWarning()).hasValue((Object)"Memory usage [120%] is too high to satisfy all of the requests. This can severely impact network throughput. Please consider increasing available network memory, or decreasing configured size of network buffer pools. (totalMemory=6.250kb (6400 bytes), requestedMemory=7.500kb (7680 bytes), missingMemory=1.250kb (1280 bytes))");
            Assertions.assertThat((Optional)globalPool.getUsageWarning()).isEmpty();
            BufferPool bufferPool2 = globalPool.createBufferPool(10, 20);
            Assertions.assertThat((long)globalPool.getEstimatedNumberOfRequestedMemorySegments()).isEqualTo(80L);
            Assertions.assertThat((int)globalPool.getEstimatedRequestedSegmentsUsage()).isEqualTo(160);
            Assertions.assertThat((Optional)globalPool.getUsageWarning()).hasValue((Object)"Memory usage [160%] is too high to satisfy all of the requests. This can severely impact network throughput. Please consider increasing available network memory, or decreasing configured size of network buffer pools. (totalMemory=6.250kb (6400 bytes), requestedMemory=10.000kb (10240 bytes), missingMemory=3.750kb (3840 bytes))");
            bufferPool2.lazyDestroy();
            bufferPool1.lazyDestroy();
            Assertions.assertThat((long)globalPool.getEstimatedNumberOfRequestedMemorySegments()).isEqualTo(40L);
            Assertions.assertThat((long)globalPool.getEstimatedRequestedMemory()).isEqualTo(5120L);
            Assertions.assertThat((int)globalPool.getEstimatedRequestedSegmentsUsage()).isEqualTo(80);
            Assertions.assertThat((Optional)globalPool.getUsageWarning()).hasValue((Object)"Memory usage [80%] went back to normal");
            Assertions.assertThat((Optional)globalPool.getUsageWarning()).isEmpty();
        }
    }

    @Test
    void testRequestMemorySegmentsWithInvalidArgument() {
        NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        Assertions.assertThatThrownBy(() -> globalPool.requestUnpooledMemorySegments(-1)).isInstanceOf(IllegalArgumentException.class);
        globalPool.destroy();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testRequestMemorySegmentsWithBuffersTaken() throws IOException, InterruptedException {
        BufferPool lbp1;
        List memorySegments;
        NetworkBufferPool networkBufferPool;
        block6: {
            int numBuffers = 10;
            networkBufferPool = new NetworkBufferPool(10, 128);
            ArrayList<Buffer> buffers = new ArrayList<Buffer>(10);
            memorySegments = Collections.emptyList();
            Thread bufferRecycler = null;
            lbp1 = null;
            try {
                lbp1 = networkBufferPool.createBufferPool(5, 10);
                for (int i = 0; i < 10; ++i) {
                    Buffer buffer = lbp1.requestBuffer();
                    buffers.add(buffer);
                    Assertions.assertThat((Object)buffer).isNotNull();
                }
                OneShotLatch isRunning = new OneShotLatch();
                bufferRecycler = new Thread(() -> {
                    try {
                        isRunning.trigger();
                        Thread.sleep(100L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                    for (Buffer buffer : buffers) {
                        buffer.recycleBuffer();
                    }
                });
                bufferRecycler.start();
                isRunning.await();
                memorySegments = networkBufferPool.requestUnpooledMemorySegments(5);
                Assertions.assertThat((List)memorySegments).doesNotContainNull();
                if (bufferRecycler == null) break block6;
            }
            catch (Throwable throwable) {
                if (bufferRecycler != null) {
                    bufferRecycler.join();
                }
                if (lbp1 != null) {
                    lbp1.lazyDestroy();
                }
                networkBufferPool.recycleUnpooledMemorySegments(memorySegments);
                networkBufferPool.destroy();
                throw throwable;
            }
            bufferRecycler.join();
        }
        if (lbp1 != null) {
            lbp1.lazyDestroy();
        }
        networkBufferPool.recycleUnpooledMemorySegments((Collection)memorySegments);
        networkBufferPool.destroy();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testRequestMemorySegmentsInterruptable() throws Exception {
        int numBuffers = 10;
        final NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        MemorySegment segment = globalPool.requestPooledMemorySegment();
        Assertions.assertThat((Object)segment).isNotNull();
        final OneShotLatch isRunning = new OneShotLatch();
        CheckedThread asyncRequest = new CheckedThread(){

            public void go() throws IOException {
                isRunning.trigger();
                globalPool.requestUnpooledMemorySegments(10);
            }
        };
        asyncRequest.start();
        isRunning.await();
        Thread.sleep(10L);
        globalPool.destroy();
        segment.free();
        try {
            Assertions.assertThatThrownBy(() -> ((CheckedThread)asyncRequest).sync()).hasMessageContaining("destroyed").isInstanceOf(IllegalStateException.class);
        }
        finally {
            globalPool.destroy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testRequestMemorySegmentsInterruptable2() throws Exception {
        int numBuffers = 10;
        final NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        MemorySegment segment = globalPool.requestPooledMemorySegment();
        Assertions.assertThat((Object)segment).isNotNull();
        final OneShotLatch isRunning = new OneShotLatch();
        CheckedThread asyncRequest = new CheckedThread(){

            public void go() throws IOException {
                isRunning.trigger();
                globalPool.requestUnpooledMemorySegments(10);
            }
        };
        asyncRequest.start();
        isRunning.await();
        Thread.sleep(10L);
        asyncRequest.interrupt();
        globalPool.recyclePooledMemorySegment(segment);
        try {
            asyncRequest.sync();
        }
        catch (IOException e) {
            Assertions.assertThat((Throwable)e).hasCauseInstanceOf(InterruptedException.class);
            globalPool.createBufferPool(10, 10);
        }
        finally {
            globalPool.destroy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testRequestMemorySegmentsTimeout() throws Exception {
        int numBuffers = 10;
        int numberOfSegmentsToRequest = 2;
        Duration requestSegmentsTimeout = Duration.ofMillis(50L);
        final NetworkBufferPool globalPool = new NetworkBufferPool(10, 128, requestSegmentsTimeout);
        BufferPool localBufferPool = globalPool.createBufferPool(1, 10);
        for (int i = 0; i < 10; ++i) {
            localBufferPool.requestBuffer();
        }
        Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).isZero();
        CheckedThread asyncRequest = new CheckedThread(){

            public void go() throws Exception {
                globalPool.requestUnpooledMemorySegments(2);
            }
        };
        asyncRequest.start();
        try {
            Assertions.assertThatThrownBy(() -> ((CheckedThread)asyncRequest).sync()).hasMessageContaining("Timeout").isInstanceOf(IOException.class);
        }
        finally {
            globalPool.destroy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testIsAvailableOrNotAfterRequestAndRecycleSingleSegment() {
        int numBuffers = 2;
        NetworkBufferPool globalPool = new NetworkBufferPool(2, 128);
        try {
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isDone();
            MemorySegment segment1 = (MemorySegment)Preconditions.checkNotNull((Object)globalPool.requestPooledMemorySegment());
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isDone();
            MemorySegment segment2 = (MemorySegment)Preconditions.checkNotNull((Object)globalPool.requestPooledMemorySegment());
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isNotDone();
            CompletableFuture availableFuture = globalPool.getAvailableFuture();
            globalPool.recyclePooledMemorySegment(segment1);
            Assertions.assertThat((CompletableFuture)availableFuture).isDone();
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isDone();
            globalPool.recyclePooledMemorySegment(segment2);
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isDone();
        }
        finally {
            globalPool.destroy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testIsAvailableOrNotAfterRequestAndRecycleMultiSegments() throws Exception {
        int numberOfSegmentsToRequest = 5;
        int numBuffers = 10;
        NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        try {
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isDone();
            List segments1 = globalPool.requestUnpooledMemorySegments(5);
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isDone();
            Assertions.assertThat((List)segments1).hasSize(5);
            List segments2 = globalPool.requestUnpooledMemorySegments(5);
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isNotDone();
            Assertions.assertThat((List)segments2).hasSize(5);
            CompletableFuture availableFuture = globalPool.getAvailableFuture();
            globalPool.recycleUnpooledMemorySegments((Collection)segments1);
            Assertions.assertThat((CompletableFuture)availableFuture).isDone();
            List segments3 = globalPool.requestUnpooledMemorySegments(5);
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isNotDone();
            Assertions.assertThat((List)segments3).hasSize(5);
            globalPool.recycleUnpooledMemorySegments((Collection)segments2);
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isDone();
            globalPool.recycleUnpooledMemorySegments((Collection)segments3);
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isDone();
        }
        finally {
            globalPool.destroy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testBlockingRequestFromMultiLocalBufferPool() throws IOException, InterruptedException {
        int localPoolRequiredSize = 5;
        int localPoolMaxSize = 10;
        int numLocalBufferPool = 2;
        int numberOfSegmentsToRequest = 10;
        int numBuffers = 20;
        ExecutorService executorService = Executors.newFixedThreadPool(2);
        NetworkBufferPool globalPool = new NetworkBufferPool(20, 128);
        ArrayList<BufferPool> localBufferPools = new ArrayList<BufferPool>(2);
        try {
            for (int i = 0; i < 2; ++i) {
                BufferPool localPool = globalPool.createBufferPool(5, 10);
                localBufferPools.add(localPool);
                Assertions.assertThat((CompletableFuture)localPool.getAvailableFuture()).isDone();
            }
            ArrayList<MemorySegment> segments = new ArrayList<MemorySegment>(9);
            for (int i = 0; i < 9; ++i) {
                segments.add(globalPool.requestPooledMemorySegment());
            }
            List exclusiveSegments = globalPool.requestUnpooledMemorySegments(globalPool.getNumberOfAvailableMemorySegments() - 1);
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isDone();
            for (BufferPool localPool : localBufferPools) {
                Assertions.assertThat((CompletableFuture)localPool.getAvailableFuture()).isDone();
            }
            CountDownLatch latch = new CountDownLatch(2);
            ArrayBlockingQueue segmentsRequested = new ArrayBlockingQueue(20);
            AtomicReference cause = new AtomicReference();
            for (BufferPool localPool : localBufferPools) {
                executorService.submit(() -> {
                    try {
                        for (int num = 10; num > 0; --num) {
                            segmentsRequested.add(localPool.requestBufferBuilderBlocking());
                        }
                    }
                    catch (Exception e) {
                        cause.set(e);
                    }
                    finally {
                        latch.countDown();
                    }
                });
            }
            while (segmentsRequested.size() + segments.size() + exclusiveSegments.size() < 20) {
                Thread.sleep(10L);
                Assertions.assertThat((Throwable)((Throwable)cause.get())).isNull();
            }
            CompletableFuture globalPoolAvailableFuture = globalPool.getAvailableFuture();
            Assertions.assertThat((CompletableFuture)globalPoolAvailableFuture).isNotDone();
            ArrayList<CompletableFuture> localPoolAvailableFutures = new ArrayList<CompletableFuture>(2);
            for (BufferPool localPool : localBufferPools) {
                CompletableFuture localPoolAvailableFuture = localPool.getAvailableFuture();
                localPoolAvailableFutures.add(localPoolAvailableFuture);
                Assertions.assertThat((CompletableFuture)localPoolAvailableFuture).isNotDone();
            }
            for (MemorySegment segment : segments) {
                globalPool.recyclePooledMemorySegment(segment);
            }
            globalPool.recycleUnpooledMemorySegments((Collection)exclusiveSegments);
            Assertions.assertThat((CompletableFuture)globalPoolAvailableFuture).isDone();
            for (CompletableFuture localPoolAvailableFuture : localPoolAvailableFutures) {
                Assertions.assertThat((CompletableFuture)localPoolAvailableFuture).isDone();
            }
            latch.await();
            Assertions.assertThat((Throwable)((Throwable)cause.get())).isNull();
            Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).isZero();
            Assertions.assertThat((CompletableFuture)globalPool.getAvailableFuture()).isNotDone();
            for (BufferPool localPool : localBufferPools) {
                Assertions.assertThat((CompletableFuture)localPool.getAvailableFuture()).isNotDone();
                Assertions.assertThat((int)localPool.bestEffortGetNumOfUsedBuffers()).isEqualTo(10);
            }
            for (BufferBuilder bufferBuilder : segmentsRequested) {
                bufferBuilder.close();
            }
        }
        finally {
            for (BufferPool bufferPool : localBufferPools) {
                bufferPool.lazyDestroy();
            }
            executorService.shutdown();
            globalPool.destroy();
        }
    }
}

