/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.kubernetes.shaded.io.fabric8.kubernetes.client.dsl.internal.uploadable;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.utils.CountingOutputStream;
import org.apache.flink.kubernetes.shaded.io.fabric8.kubernetes.client.dsl.ExecWatch;
import org.apache.flink.kubernetes.shaded.io.fabric8.kubernetes.client.dsl.internal.core.v1.PodOperationsImpl;
import org.apache.flink.kubernetes.shaded.io.fabric8.kubernetes.client.utils.InputStreamPumper;
import org.apache.flink.kubernetes.shaded.io.fabric8.kubernetes.client.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PodUpload {
    private static final Logger LOG = LoggerFactory.getLogger(PodUpload.class);
    private static final String TAR_PATH_DELIMITER = "/";

    private PodUpload() {
    }

    public static boolean upload(PodOperationsImpl operation, Path pathToUpload) throws IOException {
        if (Utils.isNotNullOrEmpty(operation.getContext().getFile()) && pathToUpload.toFile().isFile()) {
            return PodUpload.uploadTar(operation, PodUpload.getDirectoryFromFile(operation), tar -> PodUpload.addFileToTar(new File(operation.getContext().getFile()).getName(), pathToUpload.toFile(), tar));
        }
        if (Utils.isNotNullOrEmpty(operation.getContext().getDir()) && pathToUpload.toFile().isDirectory()) {
            return PodUpload.uploadTar(operation, operation.getContext().getDir(), tar -> {
                for (File file : pathToUpload.toFile().listFiles()) {
                    PodUpload.addFileToTar(file.getName(), file, tar);
                }
            });
        }
        throw new IllegalArgumentException("Provided arguments are not valid (file, directory, path)");
    }

    private static String getDirectoryFromFile(PodOperationsImpl operation) {
        String file = operation.getContext().getFile();
        String directoryTrimmedFromFilePath = file.substring(0, file.lastIndexOf(47));
        return directoryTrimmedFromFilePath.isEmpty() ? TAR_PATH_DELIMITER : directoryTrimmedFromFilePath;
    }

    private static boolean upload(PodOperationsImpl operation, String file, UploadProcessor<OutputStream> processor) throws IOException {
        CompletableFuture<Integer> exitFuture;
        String command = PodUpload.createExecCommandForUpload(file);
        int uploadRequestTimeout = operation.getRequestConfig().getUploadRequestTimeout();
        long uploadRequestTimeoutEnd = uploadRequestTimeout < 0 ? Long.MAX_VALUE : (long)uploadRequestTimeout + System.currentTimeMillis();
        long expected = 0L;
        try (ExecWatch execWatch = operation.redirectingInput().terminateOnError().exec("sh", "-c", command);){
            OutputStream out = execWatch.getInput();
            CountingOutputStream countingStream = new CountingOutputStream(out);
            processor.process((OutputStream)countingStream);
            out.close();
            expected = countingStream.getBytesWritten();
            exitFuture = execWatch.exitCode();
        }
        if (!Utils.waitUntilReady(exitFuture, Math.max(0L, uploadRequestTimeoutEnd - System.currentTimeMillis()), TimeUnit.MILLISECONDS)) {
            LOG.debug("failed to complete upload before timeout expired");
            return false;
        }
        Integer exitCode = exitFuture.getNow(null);
        if (exitCode != null && exitCode != 0) {
            LOG.debug("upload process failed with exit code {}", (Object)exitCode);
            return false;
        }
        ByteArrayOutputStream byteCount = new ByteArrayOutputStream();
        try (ExecWatch countWatch = operation.writingOutput(byteCount).exec("sh", "-c", String.format("wc -c < %s", PodOperationsImpl.shellQuote(file)));){
            CompletableFuture<Integer> countExitFuture = countWatch.exitCode();
            if (!Utils.waitUntilReady(countExitFuture, Math.max(0L, uploadRequestTimeoutEnd - System.currentTimeMillis()), TimeUnit.MILLISECONDS) || !Integer.valueOf(0).equals(countExitFuture.getNow(null))) {
                LOG.debug("failed to validate the upload size, exit code {}", countExitFuture.getNow(null));
                boolean bl = false;
                return bl;
            }
            String remoteSize = new String(byteCount.toByteArray(), StandardCharsets.UTF_8);
            if (!String.valueOf(expected).equals(remoteSize.trim())) {
                LOG.debug("upload file size validation failed, expected {}, but was {}", (Object)expected, (Object)remoteSize);
                boolean bl = false;
                return bl;
            }
        }
        return true;
    }

    public static boolean uploadFileData(PodOperationsImpl operation, InputStream inputStream) throws IOException {
        return PodUpload.upload(operation, operation.getContext().getFile(), os -> InputStreamPumper.transferTo(inputStream, os::write));
    }

    private static boolean uploadTar(PodOperationsImpl operation, String directory, UploadProcessor<TarArchiveOutputStream> processor) throws IOException {
        String fileName = String.format("/tmp/fabric8-%s.tar", UUID.randomUUID());
        boolean uploaded = PodUpload.upload(operation, fileName, os -> {
            try (TarArchiveOutputStream tar = new TarArchiveOutputStream(os);){
                tar.setLongFileMode(3);
                tar.setBigNumberMode(2);
                processor.process(tar);
            }
        });
        if (!uploaded) {
            try (ExecWatch rm = operation.writingOutput(new ByteArrayOutputStream()).exec("sh", "-c", String.format("rm %s", fileName));){
                if (!Utils.waitUntilReady(rm.exitCode(), operation.getRequestConfig().getUploadRequestTimeout(), TimeUnit.MILLISECONDS) || !Integer.valueOf(0).equals(rm.exitCode().getNow(null))) {
                    LOG.warn("delete of temporary tar file {} may not have completed", (Object)fileName);
                }
            }
            return false;
        }
        String command = PodUpload.extractTarCommand(directory, fileName);
        try (ExecWatch execWatch = operation.redirectingInput().exec("sh", "-c", command);){
            CompletableFuture<Integer> countExitFuture = execWatch.exitCode();
            boolean bl = Utils.waitUntilReady(countExitFuture, operation.getRequestConfig().getUploadRequestTimeout(), TimeUnit.MILLISECONDS) && Integer.valueOf(0).equals(countExitFuture.getNow(null));
            return bl;
        }
    }

    static String extractTarCommand(String directory, String tar) {
        return String.format("mkdir -p %1$s; tar -C %1$s -xmf %2$s; e=$?; rm %2$s; exit $e", PodOperationsImpl.shellQuote(directory), tar);
    }

    private static void addFileToTar(String fileName, File file, TarArchiveOutputStream tar) throws IOException {
        tar.putArchiveEntry((ArchiveEntry)new TarArchiveEntry(file, fileName));
        if (file.isFile()) {
            Files.copy(file.toPath(), (OutputStream)tar);
            tar.closeArchiveEntry();
        } else if (file.isDirectory()) {
            tar.closeArchiveEntry();
            for (File fileInDirectory : file.listFiles()) {
                PodUpload.addFileToTar(fileName + TAR_PATH_DELIMITER + fileInDirectory.getName(), fileInDirectory, tar);
            }
        }
    }

    static String createExecCommandForUpload(String file) {
        String directoryTrimmedFromFilePath = file.substring(0, file.lastIndexOf(47));
        String directory = directoryTrimmedFromFilePath.isEmpty() ? TAR_PATH_DELIMITER : directoryTrimmedFromFilePath;
        return String.format("mkdir -p %s && cat - > %s", PodOperationsImpl.shellQuote(directory), PodOperationsImpl.shellQuote(file));
    }

    private static interface UploadProcessor<T extends OutputStream> {
        public void process(T var1) throws IOException;
    }
}

