/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.felix.framework.util;

import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.WeakHashMap;

/**
 * A thread-safe variant of {@link Map} in which all mutative operations (the
 * "destructive" operations described by {@link Map} put, remove and so on) are
 * implemented by making a fresh copy of the underlying map.
 * <p>
 * This is ordinarily too costly, but may be <em>more</em> efficient than
 * alternatives when traversal operations vastly out-number mutations, and is
 * useful when you cannot or don't want to synchronize traversals, yet need to
 * preclude interference among concurrent threads. The "snapshot" style
 * iterators on the collections returned by {@link #entrySet()},
 * {@link #keySet()} and {@link #values()} use a reference to the internal map
 * at the point that the iterator was created. This map never changes during the
 * lifetime of the iterator, so interference is impossible and the iterator is
 * guaranteed not to throw <tt>ConcurrentModificationException</tt>. The
 * iterators will not reflect additions, removals, or changes to the list since
 * the iterator was created. Removing elements via these iterators is not
 * supported. The mutable operations on these collections (remove, retain etc.)
 * are supported but as with the {@link Map} interface, add and addAll are not
 * and throw {@link UnsupportedOperationException}.
 * <p>
 * The actual copy is performed by an abstract {@link #copy(Map)} method. The
 * method is responsible for the underlying Map implementation (for instance a
 * {@link HashMap}, {@link TreeMap} etc.) and therefore the semantics of what 
 * this map will cope with as far as null keys and values, iteration ordering 
 * etc. See the note below about suitable candidates for underlying Map 
 * implementations
 * <p>
 * There are supplied implementations for the common Collections {@link Map}
 * implementations via the {@link CopyOnWriteMaps} static factory methods.
 * <p>
 * Collection views of the keys, values and entries are modifiable and will
 * cause a copy.
 * <p>
 * <strong>Please note</strong> that the thread-safety guarantees are limited to
 * the thread-safety of the non-mutative (non-destructive) operations of the
 * underlying map implementation. For instance some implementations such as
 * {@link WeakHashMap} actually structurally modified by the {@link #get(Object)} 
 * method and are therefore not suitable candidates as delegates for this class.
 */
//@ThreadSafe
public abstract class CopyOnWriteMap extends AbstractCopyOnWriteMap
{
    private static final long serialVersionUID = 7935514534647505917L;

    /**
     * Creates a new {@link CopyOnWriteMap} with an underlying {@link HashMap}.
     */
    public static CopyOnWriteMap newHashMap()
    {
        return new CopyOnWriteMap()
        {
            private static final long serialVersionUID = 5221824943734164497L;

            public Map copy(final Map map)
            {
                return new HashMap(map);
            }
        };
    }

    /**
     * Creates a new {@link CopyOnWriteMap} with an underlying {@link HashMap}
     * using the supplied map as the initial values.
     */
    public static CopyOnWriteMap newHashMap(final Map map)
    {
        final CopyOnWriteMap result = new CopyOnWriteMap()
        {
            private static final long serialVersionUID = -7616159260882572421L;

            public Map copy(final Map map)
            {
                return new HashMap(map);
            }
        };
        result.putAll(map);
        return result;
    }

    //
    // constructors
    //

    /**
     * Create a new {@link CopyOnWriteMap} initialised with the supplied values.
     */
    CopyOnWriteMap(Map map)
    {
        super(map);
    }

    /**
     * Create a new empty {@link CopyOnWriteMap}.
     */
    CopyOnWriteMap()
    {}
}
