/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dubbo.common.cache;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.apache.dubbo.common.cache.FileCacheStore;
import org.apache.dubbo.common.logger.ErrorTypeAwareLogger;
import org.apache.dubbo.common.logger.LoggerFactory;
import org.apache.dubbo.common.utils.ConcurrentHashMapUtils;
import org.apache.dubbo.common.utils.SystemPropertyConfigUtils;

public final class FileCacheStoreFactory {
    private static final ErrorTypeAwareLogger logger = LoggerFactory.getErrorTypeAwareLogger(FileCacheStoreFactory.class);
    private static final ConcurrentMap<String, FileCacheStore> cacheMap = new ConcurrentHashMap<String, FileCacheStore>();
    private static final String SUFFIX = ".dubbo.cache";
    private static final char ESCAPE_MARK = '%';
    private static final Set<Character> LEGAL_CHARACTERS = Collections.unmodifiableSet(new HashSet<Character>(){
        {
            char c;
            this.add(Character.valueOf('-'));
            this.add(Character.valueOf('$'));
            this.add(Character.valueOf('.'));
            this.add(Character.valueOf('_'));
            for (c = '0'; c <= '9'; c = (char)(c + 1)) {
                this.add(Character.valueOf(c));
            }
            for (c = 'a'; c <= 'z'; c = (char)(c + '\u0001')) {
                this.add(Character.valueOf(c));
            }
            for (c = 'A'; c <= 'Z'; c = (char)(c + '\u0001')) {
                this.add(Character.valueOf(c));
            }
        }
    });

    private FileCacheStoreFactory() {
        throw new UnsupportedOperationException("No instance of 'FileCacheStoreFactory' for you! ");
    }

    public static FileCacheStore getInstance(String basePath, String cacheName) {
        return FileCacheStoreFactory.getInstance(basePath, cacheName, true);
    }

    public static FileCacheStore getInstance(String basePath, String cacheName, boolean enableFileCache) {
        if (basePath == null) {
            basePath = SystemPropertyConfigUtils.getSystemProperty("user.home") + File.separator + ".dubbo";
        }
        if (basePath.endsWith(File.separator)) {
            basePath = basePath.substring(0, basePath.length() - 1);
        }
        File candidate = new File(basePath);
        Path path = candidate.toPath();
        if (!candidate.isDirectory()) {
            try {
                Files.createDirectories(path, new FileAttribute[0]);
            }
            catch (IOException e) {
                logger.error("0-3", "inaccessible of cache path", "", "Cache store path can't be created: ", e);
                throw new RuntimeException("Cache store path can't be created: " + candidate, e);
            }
        }
        if (!(cacheName = FileCacheStoreFactory.safeName(cacheName)).endsWith(SUFFIX)) {
            cacheName = cacheName + SUFFIX;
        }
        String cacheFilePath = basePath + File.separator + cacheName;
        return ConcurrentHashMapUtils.computeIfAbsent(cacheMap, cacheFilePath, k -> FileCacheStoreFactory.getFile(k, enableFileCache));
    }

    private static String safeName(String name) {
        int len = name.length();
        StringBuilder sb = new StringBuilder(len);
        for (int i = 0; i < len; ++i) {
            char c = name.charAt(i);
            if (LEGAL_CHARACTERS.contains(Character.valueOf(c))) {
                sb.append(c);
                continue;
            }
            sb.append('%');
            sb.append(String.format("%04x", c));
        }
        return sb.toString();
    }

    private static FileCacheStore getFile(String name, boolean enableFileCache) {
        if (!enableFileCache) {
            return FileCacheStore.Empty.getInstance(name);
        }
        try {
            FileCacheStore.Builder builder = FileCacheStore.newBuilder();
            FileCacheStoreFactory.tryFileLock(builder, name);
            File file = new File(name);
            if (!file.exists()) {
                Path pathObjectOfFile = file.toPath();
                Files.createFile(pathObjectOfFile, new FileAttribute[0]);
            }
            builder.cacheFilePath(name).cacheFile(file);
            return builder.build();
        }
        catch (Throwable t) {
            logger.warn("0-3", "inaccessible of cache path", "", "Failed to create file store cache. Local file cache will be disabled. Cache file name: " + name, t);
            return FileCacheStore.Empty.getInstance(name);
        }
    }

    private static void tryFileLock(FileCacheStore.Builder builder, String fileName) throws PathNotExclusiveException {
        FileLock dirLock;
        File lockFile = new File(fileName + ".lock");
        try {
            lockFile.createNewFile();
            if (!lockFile.exists()) {
                throw new AssertionError((Object)("Failed to create lock file " + lockFile));
            }
            FileChannel lockFileChannel = new RandomAccessFile(lockFile, "rw").getChannel();
            dirLock = lockFileChannel.tryLock();
        }
        catch (OverlappingFileLockException ofle) {
            dirLock = null;
        }
        catch (IOException ioe) {
            throw new RuntimeException(ioe);
        }
        if (dirLock == null) {
            throw new PathNotExclusiveException(fileName + " is not exclusive. Maybe multiple Dubbo instances are using the same folder.");
        }
        lockFile.deleteOnExit();
        builder.directoryLock(dirLock).lockFile(lockFile);
    }

    static void removeCache(String cacheFileName) {
        cacheMap.remove(cacheFileName);
    }

    private static class PathNotExclusiveException
    extends Exception {
        public PathNotExclusiveException(String msg) {
            super(msg);
        }
    }
}

