/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.emitter.prometheus;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import io.prometheus.client.Collector;
import io.prometheus.client.CollectorRegistry;
import io.prometheus.client.Counter;
import io.prometheus.client.Gauge;
import io.prometheus.client.Histogram;
import io.prometheus.client.exporter.HTTPServer;
import io.prometheus.client.exporter.PushGateway;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.druid.emitter.prometheus.DimensionsAndCollector;
import org.apache.druid.emitter.prometheus.Metrics;
import org.apache.druid.emitter.prometheus.PrometheusEmitterConfig;
import org.apache.druid.java.util.common.concurrent.ScheduledExecutors;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.java.util.emitter.core.Emitter;
import org.apache.druid.java.util.emitter.core.Event;
import org.apache.druid.java.util.emitter.service.ServiceMetricEvent;

public class PrometheusEmitter
implements Emitter {
    private static final Logger log = new Logger(PrometheusEmitter.class);
    private final Metrics metrics;
    private final PrometheusEmitterConfig config;
    private final PrometheusEmitterConfig.Strategy strategy;
    private static final Pattern PATTERN = Pattern.compile("[^a-zA-Z0-9_][^a-zA-Z0-9_]*");
    private static final String TAG_HOSTNAME = "host_name";
    private static final String TAG_SERVICE = "druid_service";
    private HTTPServer server;
    private PushGateway pushGateway;
    private volatile String identifier;
    private ScheduledExecutorService exec;

    static PrometheusEmitter of(PrometheusEmitterConfig config) {
        return new PrometheusEmitter(config);
    }

    public PrometheusEmitter(PrometheusEmitterConfig config) {
        this.config = config;
        this.strategy = config.getStrategy();
        this.metrics = new Metrics(config);
    }

    public PrometheusEmitter(PrometheusEmitterConfig config, ScheduledExecutorService exec) {
        this(config);
        this.exec = exec;
    }

    public void start() {
        if (this.strategy.equals((Object)PrometheusEmitterConfig.Strategy.exporter)) {
            if (this.server == null) {
                try {
                    this.server = new HTTPServer(this.config.getPort());
                }
                catch (IOException e) {
                    log.error((Throwable)e, "Unable to start prometheus HTTPServer", new Object[0]);
                }
            } else {
                log.error("HTTPServer is already started", new Object[0]);
            }
            if (this.config.getFlushPeriod() != null && this.exec == null) {
                this.exec = ScheduledExecutors.fixed((int)1, (String)"PrometheusTTLExecutor-%s");
                this.exec.scheduleAtFixedRate(this::cleanUpStaleMetrics, this.config.getFlushPeriod().intValue(), this.config.getFlushPeriod().intValue(), TimeUnit.SECONDS);
                log.info("Started TTL scheduler with TTL of [%d] seconds.", new Object[]{this.config.getFlushPeriod()});
            }
        } else if (this.strategy.equals((Object)PrometheusEmitterConfig.Strategy.pushgateway)) {
            String address = this.config.getPushGatewayAddress();
            if (address.startsWith("https") || address.startsWith("http")) {
                URL myURL = PrometheusEmitter.createURLSneakily(address);
                this.pushGateway = new PushGateway(myURL);
            } else {
                this.pushGateway = new PushGateway(address);
            }
            this.exec = ScheduledExecutors.fixed((int)1, (String)"PrometheusPushGatewayEmitter-%s");
            this.exec.scheduleAtFixedRate(() -> this.flush(), this.config.getFlushPeriod().intValue(), this.config.getFlushPeriod().intValue(), TimeUnit.SECONDS);
        }
    }

    private static URL createURLSneakily(String urlString) {
        try {
            return new URL(urlString);
        }
        catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
    }

    public void emit(Event event) {
        if (event instanceof ServiceMetricEvent) {
            this.emitMetric((ServiceMetricEvent)event);
        }
    }

    private void emitMetric(ServiceMetricEvent metricEvent) {
        String name = metricEvent.getMetric();
        String service = metricEvent.getService();
        String host = metricEvent.getHost();
        Map userDims = metricEvent.getUserDims();
        this.identifier = userDims.get("task") == null ? metricEvent.getHost() : (String)userDims.get("task");
        Number value = metricEvent.getValue();
        DimensionsAndCollector metric = this.metrics.getByName(name, service);
        if (metric != null) {
            String[] labelValues = new String[metric.getDimensions().length];
            String[] labelNames = metric.getDimensions();
            Map<String, String> extraLabels = this.config.getExtraLabels();
            for (int i = 0; i < labelValues.length; ++i) {
                String labelName = labelNames[i];
                Object userDim = userDims.get(labelName);
                labelValues[i] = userDim != null ? PATTERN.matcher(userDim.toString()).replaceAll("_") : (this.config.isAddHostAsLabel() && TAG_HOSTNAME.equals(labelName) ? host : (this.config.isAddServiceAsLabel() && TAG_SERVICE.equals(labelName) ? service : (extraLabels.containsKey(labelName) ? this.config.getExtraLabels().get(labelName) : "unknown")));
            }
            if (metric.getCollector() instanceof Counter) {
                double metricValue = value.doubleValue();
                if (metricValue >= 0.0) {
                    ((Counter.Child)((Counter)metric.getCollector()).labels(labelValues)).inc(metricValue);
                    metric.resetLastUpdateTime(Arrays.asList(labelValues));
                } else {
                    log.warn("Counter increment amount must be non-negative, but got value[%f] for metric[%s]. If this is valid, metric should be defined as a gauge instead of counter.", new Object[]{metricValue, name});
                }
            } else if (metric.getCollector() instanceof Gauge) {
                ((Gauge.Child)((Gauge)metric.getCollector()).labels(labelValues)).set(value.doubleValue());
                metric.resetLastUpdateTime(Arrays.asList(labelValues));
            } else if (metric.getCollector() instanceof Histogram) {
                ((Histogram.Child)((Histogram)metric.getCollector()).labels(labelValues)).observe(value.doubleValue() / metric.getConversionFactor());
                metric.resetLastUpdateTime(Arrays.asList(labelValues));
            } else {
                log.error("Unrecognized metric type [%s]", new Object[]{metric.getCollector().getClass()});
            }
        } else {
            log.debug("Unmapped metric [%s]", new Object[]{name});
        }
    }

    private void pushMetric() {
        if (this.pushGateway == null || this.identifier == null) {
            return;
        }
        Map<String, DimensionsAndCollector> map = this.metrics.getRegisteredMetrics();
        CollectorRegistry metrics = new CollectorRegistry();
        try {
            for (DimensionsAndCollector collector : map.values()) {
                metrics.register((Collector)collector.getCollector());
            }
            this.pushGateway.push(metrics, this.config.getNamespace(), (Map)ImmutableMap.of((Object)this.config.getNamespace(), (Object)this.identifier));
        }
        catch (IOException e) {
            log.error((Throwable)e, "Unable to push prometheus metrics to pushGateway", new Object[0]);
        }
    }

    public void flush() {
        this.pushMetric();
    }

    public void close() {
        if (this.strategy.equals((Object)PrometheusEmitterConfig.Strategy.exporter)) {
            if (this.exec != null) {
                this.exec.shutdownNow();
            }
            if (this.server != null) {
                this.server.close();
            }
        } else {
            this.exec.shutdownNow();
            this.flush();
            try {
                if (this.config.getWaitForShutdownDelay().getMillis() > 0L) {
                    log.info("Waiting [%s]ms before deleting metrics from the push gateway.", new Object[]{this.config.getWaitForShutdownDelay().getMillis()});
                    Thread.sleep(this.config.getWaitForShutdownDelay().getMillis());
                }
            }
            catch (InterruptedException e) {
                log.error((Throwable)e, "Interrupted while waiting for shutdown delay. Deleting metrics from the push gateway now.", new Object[0]);
            }
            finally {
                this.deletePushGatewayMetrics();
            }
        }
    }

    private void deletePushGatewayMetrics() {
        if (this.pushGateway != null && this.config.isDeletePushGatewayMetricsOnShutdown()) {
            try {
                this.pushGateway.delete(this.config.getNamespace(), (Map)ImmutableMap.of((Object)this.config.getNamespace(), (Object)this.identifier));
            }
            catch (IOException e) {
                log.error((Throwable)e, "Unable to delete prometheus metrics from push gateway", new Object[0]);
            }
        }
    }

    public HTTPServer getServer() {
        return this.server;
    }

    public Metrics getMetrics() {
        return this.metrics;
    }

    public PushGateway getPushGateway() {
        return this.pushGateway;
    }

    public void setPushGateway(PushGateway pushGateway) {
        this.pushGateway = pushGateway;
    }

    @VisibleForTesting
    protected void cleanUpStaleMetrics() {
        if (this.config.getFlushPeriod() == null) {
            return;
        }
        Map<String, DimensionsAndCollector> map = this.metrics.getRegisteredMetrics();
        for (Map.Entry<String, DimensionsAndCollector> entry : map.entrySet()) {
            DimensionsAndCollector metric = entry.getValue();
            for (List labelValues : metric.getLabelValuesToStopwatch().keySet()) {
                if (!metric.shouldRemoveIfExpired(labelValues)) continue;
                log.debug("Metric [%s] with labels [%s] has expired", new Object[]{entry.getKey(), labelValues});
                metric.getCollector().remove(labelValues.toArray(new String[0]));
            }
        }
    }
}

