/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.indexing.overlord.autoscaling.gce;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonTypeName;
import com.google.api.client.googleapis.auth.oauth2.GoogleCredential;
import com.google.api.client.googleapis.javanet.GoogleNetHttpTransport;
import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.http.javanet.NetHttpTransport;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.jackson2.JacksonFactory;
import com.google.api.services.compute.Compute;
import com.google.api.services.compute.model.Instance;
import com.google.api.services.compute.model.InstanceGroupManagersDeleteInstancesRequest;
import com.google.api.services.compute.model.InstanceGroupManagersListManagedInstancesResponse;
import com.google.api.services.compute.model.InstanceList;
import com.google.api.services.compute.model.ManagedInstance;
import com.google.api.services.compute.model.NetworkInterface;
import com.google.api.services.compute.model.Operation;
import com.google.common.base.Preconditions;
import com.google.common.net.InetAddresses;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import javax.annotation.Nullable;
import org.apache.druid.indexing.overlord.autoscaling.AutoScaler;
import org.apache.druid.indexing.overlord.autoscaling.AutoScalingData;
import org.apache.druid.indexing.overlord.autoscaling.gce.GceEnvironmentConfig;
import org.apache.druid.indexing.overlord.autoscaling.gce.GceServiceException;
import org.apache.druid.indexing.overlord.autoscaling.gce.GceUtils;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.emitter.EmittingLogger;

@JsonTypeName(value="gce")
public class GceAutoScaler
implements AutoScaler<GceEnvironmentConfig> {
    private static final EmittingLogger log = new EmittingLogger(GceAutoScaler.class);
    private final GceEnvironmentConfig envConfig;
    private final int minNumWorkers;
    private final int maxNumWorkers;
    private Compute cachedComputeService = null;
    private static final long POLL_INTERVAL_MS = 5000L;
    private static final int RUNNING_INSTANCES_MAX_RETRIES = 10;
    private static final int OPERATION_END_MAX_RETRIES = 10;

    @JsonCreator
    public GceAutoScaler(@JsonProperty(value="minNumWorkers") int minNumWorkers, @JsonProperty(value="maxNumWorkers") int maxNumWorkers, @JsonProperty(value="envConfig") GceEnvironmentConfig envConfig) {
        Preconditions.checkArgument((minNumWorkers >= 0 ? 1 : 0) != 0, (Object)"minNumWorkers must be greater than or equal to 0");
        this.minNumWorkers = minNumWorkers;
        Preconditions.checkArgument((maxNumWorkers > 0 ? 1 : 0) != 0, (Object)"maxNumWorkers must be greater than 0");
        Preconditions.checkArgument((maxNumWorkers > minNumWorkers ? 1 : 0) != 0, (Object)"maxNumWorkers must be greater than minNumWorkers");
        this.maxNumWorkers = maxNumWorkers;
        this.envConfig = envConfig;
    }

    @JsonProperty
    public int getMinNumWorkers() {
        return this.minNumWorkers;
    }

    @JsonProperty
    public int getMaxNumWorkers() {
        return this.maxNumWorkers;
    }

    @JsonProperty
    public GceEnvironmentConfig getEnvConfig() {
        return this.envConfig;
    }

    @Nullable
    Compute createComputeServiceImpl() throws IOException, GeneralSecurityException, GceServiceException {
        JacksonFactory jsonFactory;
        NetHttpTransport httpTransport = GoogleNetHttpTransport.newTrustedTransport();
        GoogleCredential credential = GoogleCredential.getApplicationDefault((HttpTransport)httpTransport, (JsonFactory)(jsonFactory = JacksonFactory.getDefaultInstance()));
        if (credential.createScopedRequired()) {
            ArrayList<String> scopes = new ArrayList<String>();
            scopes.add("https://www.googleapis.com/auth/cloud-platform");
            scopes.add("https://www.googleapis.com/auth/compute");
            credential = credential.createScoped(scopes);
        }
        if (credential.getClientAuthentication() != null) {
            throw new GceServiceException("Not using a service account");
        }
        return new Compute.Builder((HttpTransport)httpTransport, (JsonFactory)jsonFactory, (HttpRequestInitializer)credential).setApplicationName("DruidAutoscaler").build();
    }

    private synchronized Compute createComputeService() throws IOException, GeneralSecurityException, InterruptedException, GceServiceException {
        int maxRetries = 5;
        for (int retries = 0; this.cachedComputeService == null && retries < 5; ++retries) {
            if (retries > 0) {
                Thread.sleep(5000L);
            }
            log.info("Creating new ComputeService [%d/%d]", new Object[]{retries + 1, 5});
            try {
                this.cachedComputeService = this.createComputeServiceImpl();
                continue;
            }
            catch (Throwable e) {
                log.error(e, "Got Exception in creating the ComputeService", new Object[0]);
                throw e;
            }
        }
        return this.cachedComputeService;
    }

    @Nullable
    private Operation.Error waitForOperationEnd(Compute compute, Operation operation) throws Exception {
        String status = operation.getStatus();
        String opId = operation.getName();
        for (int i = 0; i < 10; ++i) {
            if (operation == null || "DONE".equals(status)) {
                return operation == null ? null : operation.getError();
            }
            log.info("Waiting for operation %s to end", new Object[]{opId});
            Thread.sleep(5000L);
            Compute.ZoneOperations.Get get = compute.zoneOperations().get(this.envConfig.getProjectId(), this.envConfig.getZoneName(), opId);
            operation = (Operation)get.execute();
            if (operation == null) continue;
            status = operation.getStatus();
        }
        throw new InterruptedException(StringUtils.format((String)"Timed out waiting for operation %s to complete", (Object[])new Object[]{opId}));
    }

    public AutoScalingData provision() {
        String project = this.envConfig.getProjectId();
        String zone = this.envConfig.getZoneName();
        int numInstances = this.envConfig.getNumInstances();
        String managedInstanceGroupName = this.envConfig.getManagedInstanceGroupName();
        try {
            List<String> before = this.getRunningInstances();
            log.debug("Existing instances [%s]", new Object[]{String.join((CharSequence)",", before)});
            int toSize = Math.min(before.size() + numInstances, this.getMaxNumWorkers());
            if (before.size() >= toSize) {
                return new AutoScalingData(new ArrayList());
            }
            log.info("Asked to provision instances, will resize to %d", new Object[]{toSize});
            Compute computeService = this.createComputeService();
            Compute.InstanceGroupManagers.Resize request = computeService.instanceGroupManagers().resize(project, zone, managedInstanceGroupName, Integer.valueOf(toSize));
            Operation response = (Operation)request.execute();
            Operation.Error err = this.waitForOperationEnd(computeService, response);
            if (err == null || err.isEmpty()) {
                List<String> after = null;
                for (int i = 0; i < 10 && (after = this.getRunningInstances()).size() != toSize; ++i) {
                    log.info("Machines not up yet, waiting", new Object[0]);
                    Thread.sleep(5000L);
                }
                after.removeAll(before);
                log.info("Added instances [%s]", new Object[]{String.join((CharSequence)",", after)});
                return new AutoScalingData(after);
            }
            log.error("Unable to provision instances: %s", new Object[]{err.toPrettyString()});
        }
        catch (Exception e) {
            log.error((Throwable)e, "Unable to provision any gce instances.", new Object[0]);
        }
        return new AutoScalingData(new ArrayList());
    }

    public AutoScalingData terminate(List<String> ips) {
        log.info("Asked to terminate: [%s]", new Object[]{String.join((CharSequence)",", ips)});
        if (ips.isEmpty()) {
            return new AutoScalingData(new ArrayList());
        }
        ArrayList nodeIds = this.ipToIdLookup(ips);
        try {
            return this.terminateWithIds(nodeIds != null ? nodeIds : new ArrayList());
        }
        catch (Exception e) {
            log.error((Throwable)e, "Unable to terminate any instances.", new Object[0]);
            return new AutoScalingData(new ArrayList());
        }
    }

    private List<String> namesToInstances(List<String> names) {
        ArrayList<String> instances = new ArrayList<String>();
        for (String name : names) {
            instances.add(StringUtils.format((String)"zones/%s/instances/%s", (Object[])new Object[]{this.envConfig.getZoneName(), name}));
        }
        return instances;
    }

    public AutoScalingData terminateWithIds(List<String> ids) {
        log.info("Asked to terminate IDs: [%s]", new Object[]{String.join((CharSequence)",", ids)});
        if (ids.isEmpty()) {
            return new AutoScalingData(new ArrayList());
        }
        try {
            String project = this.envConfig.getProjectId();
            String zone = this.envConfig.getZoneName();
            String managedInstanceGroupName = this.envConfig.getManagedInstanceGroupName();
            List<String> before = this.getRunningInstances();
            InstanceGroupManagersDeleteInstancesRequest requestBody = new InstanceGroupManagersDeleteInstancesRequest();
            requestBody.setInstances(this.namesToInstances(ids));
            Compute computeService = this.createComputeService();
            Compute.InstanceGroupManagers.DeleteInstances request = computeService.instanceGroupManagers().deleteInstances(project, zone, managedInstanceGroupName, requestBody);
            Operation response = (Operation)request.execute();
            Operation.Error err = this.waitForOperationEnd(computeService, response);
            if (err == null || err.isEmpty()) {
                List<String> after = null;
                for (int i = 0; i < 10 && (after = this.getRunningInstances()).size() != before.size() - ids.size(); ++i) {
                    log.info("Machines not down yet, waiting", new Object[0]);
                    Thread.sleep(5000L);
                }
                before.removeAll(after);
                return new AutoScalingData(before);
            }
            log.error("Unable to terminate instances: %s", new Object[]{err.toPrettyString()});
        }
        catch (Exception e) {
            log.error((Throwable)e, "Unable to terminate any instances.", new Object[0]);
        }
        return new AutoScalingData(new ArrayList());
    }

    private List<String> getRunningInstances() {
        long maxResults = 500L;
        ArrayList<String> ids = new ArrayList<String>();
        try {
            String project = this.envConfig.getProjectId();
            String zone = this.envConfig.getZoneName();
            String managedInstanceGroupName = this.envConfig.getManagedInstanceGroupName();
            Compute computeService = this.createComputeService();
            Compute.InstanceGroupManagers.ListManagedInstances request = computeService.instanceGroupManagers().listManagedInstances(project, zone, managedInstanceGroupName);
            request.setMaxResults(Long.valueOf(500L));
            InstanceGroupManagersListManagedInstancesResponse response = (InstanceGroupManagersListManagedInstancesResponse)request.execute();
            for (ManagedInstance mi : response.getManagedInstances()) {
                ids.add(GceUtils.extractNameFromInstance(mi.getInstance()));
            }
            log.debug("Found running instances [%s]", new Object[]{String.join((CharSequence)",", ids)});
        }
        catch (Exception e) {
            log.error((Throwable)e, "Unable to get instances.", new Object[0]);
        }
        return ids;
    }

    public List<String> ipToIdLookup(List<String> ips) {
        log.info("Asked IPs -> IDs for: [%s]", new Object[]{String.join((CharSequence)",", ips)});
        if (ips.isEmpty()) {
            return new ArrayList<String>();
        }
        if (!InetAddresses.isInetAddress((String)ips.get(0))) {
            log.debug("Not IPs, doing nothing", new Object[0]);
            return ips;
        }
        String project = this.envConfig.getProjectId();
        String zone = this.envConfig.getZoneName();
        try {
            InstanceList response;
            Compute computeService = this.createComputeService();
            Compute.Instances.List request = computeService.instances().list(project, zone);
            ArrayList<String> instanceIds = new ArrayList<String>();
            do {
                if ((response = (InstanceList)request.execute()).getItems() == null) continue;
                for (Instance instance : response.getItems()) {
                    for (NetworkInterface ni : instance.getNetworkInterfaces()) {
                        if (!ips.contains(ni.getNetworkIP())) continue;
                        instanceIds.add(instance.getName());
                    }
                }
                request.setPageToken(response.getNextPageToken());
            } while (response.getNextPageToken() != null);
            log.debug("Converted to [%s]", new Object[]{String.join((CharSequence)",", instanceIds)});
            return instanceIds;
        }
        catch (Exception e) {
            log.error((Throwable)e, "Unable to convert IPs to IDs.", new Object[0]);
            return new ArrayList<String>();
        }
    }

    public List<String> idToIpLookup(List<String> nodeIds) {
        log.info("Asked IDs -> IPs for: [%s]", new Object[]{String.join((CharSequence)",", nodeIds)});
        if (nodeIds.isEmpty()) {
            return new ArrayList<String>();
        }
        String project = this.envConfig.getProjectId();
        String zone = this.envConfig.getZoneName();
        try {
            InstanceList response;
            Compute computeService = this.createComputeService();
            Compute.Instances.List request = computeService.instances().list(project, zone);
            request.setFilter(GceUtils.buildFilter(nodeIds, "name"));
            ArrayList<String> instanceIps = new ArrayList<String>();
            do {
                if ((response = (InstanceList)request.execute()).getItems() == null) continue;
                for (Instance instance : response.getItems()) {
                    String ip = ((NetworkInterface)instance.getNetworkInterfaces().get(0)).getNetworkIP();
                    if (ip != null && !"null".equals(ip)) {
                        instanceIps.add(ip);
                        continue;
                    }
                    log.warn("Call returned null IP for %s, skipping", new Object[]{instance.getName()});
                }
                request.setPageToken(response.getNextPageToken());
            } while (response.getNextPageToken() != null);
            return instanceIps;
        }
        catch (Exception e) {
            log.error((Throwable)e, "Unable to convert IDs to IPs.", new Object[0]);
            return new ArrayList<String>();
        }
    }

    public String toString() {
        return "gceAutoScaler={envConfig=" + this.envConfig + ", maxNumWorkers=" + this.maxNumWorkers + ", minNumWorkers=" + this.minNumWorkers + "}";
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        GceAutoScaler that = (GceAutoScaler)o;
        return Objects.equals(this.envConfig, that.envConfig) && this.minNumWorkers == that.minNumWorkers && this.maxNumWorkers == that.maxNumWorkers;
    }

    public int hashCode() {
        int result = 0;
        result = 31 * result + Objects.hashCode(this.envConfig);
        result = 31 * result + this.minNumWorkers;
        result = 31 * result + this.maxNumWorkers;
        return result;
    }
}

