/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.segment.loading;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Suppliers;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import com.google.inject.Inject;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.locks.Lock;
import java.util.function.Supplier;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.NullOutputStream;
import org.apache.druid.error.DruidException;
import org.apache.druid.guice.annotations.Json;
import org.apache.druid.java.util.common.FileUtils;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.concurrent.Execs;
import org.apache.druid.java.util.common.io.Closer;
import org.apache.druid.java.util.emitter.EmittingLogger;
import org.apache.druid.segment.IndexIO;
import org.apache.druid.segment.ReferenceCountedObjectProvider;
import org.apache.druid.segment.ReferenceCountedSegmentProvider;
import org.apache.druid.segment.Segment;
import org.apache.druid.segment.SegmentLazyLoadFailCallback;
import org.apache.druid.segment.loading.AcquireSegmentAction;
import org.apache.druid.segment.loading.CacheEntry;
import org.apache.druid.segment.loading.DataSegmentPusher;
import org.apache.druid.segment.loading.LoadSpec;
import org.apache.druid.segment.loading.MMappedQueryableSegmentizerFactory;
import org.apache.druid.segment.loading.SegmentCacheEntryIdentifier;
import org.apache.druid.segment.loading.SegmentCacheManager;
import org.apache.druid.segment.loading.SegmentLoaderConfig;
import org.apache.druid.segment.loading.SegmentLoadingException;
import org.apache.druid.segment.loading.SegmentizerFactory;
import org.apache.druid.segment.loading.StorageLocation;
import org.apache.druid.segment.loading.StorageLocationSelectorStrategy;
import org.apache.druid.timeline.DataSegment;
import org.apache.druid.timeline.SegmentId;
import org.apache.druid.utils.CloseableUtils;

public class SegmentLocalCacheManager
implements SegmentCacheManager {
    @VisibleForTesting
    static final String DOWNLOAD_START_MARKER_FILE_NAME = "downloadStartMarker";
    private static final EmittingLogger log = new EmittingLogger(SegmentLocalCacheManager.class);
    private final SegmentLoaderConfig config;
    private final ObjectMapper jsonMapper;
    private final List<StorageLocation> locations;
    private final ConcurrentHashMap<DataSegment, ReferenceCountingLock> segmentLocks = new ConcurrentHashMap();
    private final StorageLocationSelectorStrategy strategy;
    private final IndexIO indexIO;
    private final ListeningExecutorService virtualStorageLoadOnDemandExec;
    private ExecutorService loadOnBootstrapExec = null;
    private ExecutorService loadOnDownloadExec = null;

    @Inject
    public SegmentLocalCacheManager(List<StorageLocation> locations, SegmentLoaderConfig config, @Nonnull StorageLocationSelectorStrategy strategy, IndexIO indexIO, @Json ObjectMapper mapper) {
        this.config = config;
        this.jsonMapper = mapper;
        this.locations = locations;
        this.strategy = strategy;
        this.indexIO = indexIO;
        log.info("Using storage location strategy[%s].", new Object[]{this.strategy.getClass().getSimpleName()});
        if (config.isVirtualStorage()) {
            log.info("Using virtual storage mode - on demand load threads: [%d].", new Object[]{config.getVirtualStorageLoadThreads()});
            if (config.getNumThreadsToLoadSegmentsIntoPageCacheOnDownload() > 0) {
                throw DruidException.defensive((String)"Invalid configuration: virtualStorage is incompatible with numThreadsToLoadSegmentsIntoPageCacheOnDownload", (Object[])new Object[0]);
            }
            if (config.getNumThreadsToLoadSegmentsIntoPageCacheOnBootstrap() > 0) {
                throw DruidException.defensive((String)"Invalid configuration: virtualStorage is incompatible with numThreadsToLoadSegmentsIntoPageCacheOnBootstrap", (Object[])new Object[0]);
            }
            this.virtualStorageLoadOnDemandExec = MoreExecutors.listeningDecorator((ExecutorService)Executors.newFixedThreadPool(config.getVirtualStorageLoadThreads(), Execs.makeThreadFactory((String)"VirtualStorageOnDemandLoadingThread-%s")));
        } else {
            log.info("Number of threads to load segments into page cache - on bootstrap: [%d], on download: [%d].", new Object[]{config.getNumThreadsToLoadSegmentsIntoPageCacheOnBootstrap(), config.getNumThreadsToLoadSegmentsIntoPageCacheOnDownload()});
            if (config.getNumThreadsToLoadSegmentsIntoPageCacheOnBootstrap() > 0) {
                this.loadOnBootstrapExec = Execs.multiThreaded((int)config.getNumThreadsToLoadSegmentsIntoPageCacheOnBootstrap(), (String)"Load-SegmentsIntoPageCacheOnBootstrap-%s");
            }
            if (config.getNumThreadsToLoadSegmentsIntoPageCacheOnDownload() > 0) {
                this.loadOnDownloadExec = Executors.newFixedThreadPool(config.getNumThreadsToLoadSegmentsIntoPageCacheOnDownload(), Execs.makeThreadFactory((String)"LoadSegmentsIntoPageCacheOnDownload-%s"));
            }
            this.virtualStorageLoadOnDemandExec = null;
        }
    }

    @Override
    public boolean canHandleSegments() {
        boolean isLocationsValid = this.locations != null && !this.locations.isEmpty();
        boolean isLocationsConfigValid = this.config.getLocations() != null && !this.config.getLocations().isEmpty();
        return isLocationsValid || isLocationsConfigValid;
    }

    @Override
    public List<DataSegment> getCachedSegments() throws IOException {
        if (!this.canHandleSegments()) {
            throw DruidException.defensive((String)"canHandleSegments() is false. getCachedSegments() must be invoked only when canHandleSegments() returns true.", (Object[])new Object[0]);
        }
        File infoDir = this.getEffectiveInfoDir();
        FileUtils.mkdirp((File)infoDir);
        ArrayList<DataSegment> cachedSegments = new ArrayList<DataSegment>();
        File[] segmentsToLoad = infoDir.listFiles();
        int ignored = 0;
        for (int i = 0; i < segmentsToLoad.length; ++i) {
            File file = segmentsToLoad[i];
            log.info("Loading segment cache file [%d/%d][%s].", new Object[]{i + 1, segmentsToLoad.length, file});
            try {
                DataSegment segment = (DataSegment)this.jsonMapper.readValue(file, DataSegment.class);
                boolean removeInfo = false;
                if (!segment.getId().toString().equals(file.getName())) {
                    log.warn("Ignoring cache file[%s] for segment[%s].", new Object[]{file.getPath(), segment.getId()});
                    ++ignored;
                } else {
                    removeInfo = true;
                    SegmentCacheEntry cacheEntry = new SegmentCacheEntry(segment);
                    for (StorageLocation location : this.locations) {
                        File legacyPath = new File(location.getPath(), DataSegmentPusher.getDefaultStorageDir((DataSegment)segment, (boolean)false));
                        if (legacyPath.exists()) {
                            File destination = cacheEntry.toPotentialLocation(location.getPath());
                            FileUtils.mkdirp((File)destination);
                            File[] oldFiles = legacyPath.listFiles();
                            File[] newFiles = destination.listFiles();
                            if (oldFiles != null && oldFiles.length > 0 && newFiles != null && newFiles.length == 0) {
                                Files.move(legacyPath.toPath(), destination.toPath(), StandardCopyOption.ATOMIC_MOVE);
                            }
                            SegmentLocalCacheManager.cleanupLegacyCacheLocation(location.getPath(), legacyPath);
                        }
                        if (!cacheEntry.checkExists(location.getPath())) continue;
                        removeInfo = false;
                        boolean reserveResult = this.config.isVirtualStorage() ? location.reserveWeak(cacheEntry) : location.reserve(cacheEntry);
                        if (!reserveResult) {
                            log.makeAlert("storage[%s:%,d] has more segments than it is allowed. Currently loading Segment[%s:%,d]. Please increase druid.segmentCache.locations maxSize param", new Object[]{location.getPath(), location.availableSizeBytes(), segment.getId(), segment.getSize()}).emit();
                        }
                        cachedSegments.add(segment);
                    }
                }
                if (!removeInfo) continue;
                SegmentId segmentId = segment.getId();
                log.warn("Unable to find cache file for segment[%s]. Deleting lookup entry.", new Object[]{segmentId});
                this.removeInfoFile(segment);
                continue;
            }
            catch (Exception e) {
                log.makeAlert((Throwable)e, "Failed to load segment from segment cache file.", new Object[0]).addData("file", (Object)file).emit();
            }
        }
        if (ignored > 0) {
            log.makeAlert("Ignored misnamed segment cache files on startup.", new Object[0]).addData("numIgnored", (Object)ignored).emit();
        }
        return cachedSegments;
    }

    @Override
    public void storeInfoFile(DataSegment segment) throws IOException {
        File segmentInfoCacheFile = new File(this.getEffectiveInfoDir(), segment.getId().toString());
        if (!segmentInfoCacheFile.exists()) {
            this.jsonMapper.writeValue(segmentInfoCacheFile, (Object)segment);
        }
    }

    @Override
    public void removeInfoFile(DataSegment segment) {
        File segmentInfoCacheFile = new File(this.getEffectiveInfoDir(), segment.getId().toString());
        if (!segmentInfoCacheFile.delete()) {
            log.warn("Unable to delete cache file[%s] for segment[%s].", new Object[]{segmentInfoCacheFile, segment.getId()});
        }
    }

    @Override
    public Optional<Segment> acquireCachedSegment(DataSegment dataSegment) {
        SegmentCacheEntryIdentifier cacheEntryIdentifier = new SegmentCacheEntryIdentifier(dataSegment.getId());
        for (StorageLocation location : this.locations) {
            SegmentCacheEntry cacheEntry = (SegmentCacheEntry)location.getStaticCacheEntry(cacheEntryIdentifier);
            if (cacheEntry != null) {
                return cacheEntry.acquireReference();
            }
            StorageLocation.ReservationHold hold = location.addWeakReservationHoldIfExists(cacheEntryIdentifier);
            try {
                Optional<Segment> segment;
                if (hold == null) continue;
                if (((SegmentCacheEntry)hold.getEntry()).isMounted() && (segment = ((SegmentCacheEntry)hold.getEntry()).acquireReference()).isPresent()) {
                    return ReferenceCountedSegmentProvider.wrapCloseable((ReferenceCountedSegmentProvider.LeafReference)((ReferenceCountedSegmentProvider.LeafReference)segment.get()), hold);
                }
                hold.close();
            }
            catch (Throwable e) {
                hold.close();
                throw e;
            }
        }
        return Optional.empty();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public AcquireSegmentAction acquireSegment(DataSegment dataSegment) throws SegmentLoadingException {
        ReferenceCountingLock lock;
        SegmentCacheEntryIdentifier identifier = new SegmentCacheEntryIdentifier(dataSegment.getId());
        AcquireSegmentAction acquireExisting = this.acquireExistingSegment(identifier);
        if (acquireExisting != null) {
            return acquireExisting;
        }
        ReferenceCountingLock referenceCountingLock = lock = this.lock(dataSegment);
        synchronized (referenceCountingLock) {
            try {
                AcquireSegmentAction retryAcquireExisting = this.acquireExistingSegment(identifier);
                if (retryAcquireExisting != null) {
                    AcquireSegmentAction acquireSegmentAction = retryAcquireExisting;
                    return acquireSegmentAction;
                }
                if (!this.config.isVirtualStorage()) {
                    AcquireSegmentAction acquireSegmentAction = AcquireSegmentAction.missingSegment();
                    return acquireSegmentAction;
                }
                Iterator<StorageLocation> iterator = this.strategy.getLocations();
                while (iterator.hasNext()) {
                    AcquireSegmentAction acquireSegmentAction;
                    StorageLocation location = iterator.next();
                    StorageLocation.ReservationHold hold = location.addWeakReservationHold(identifier, () -> new SegmentCacheEntry(dataSegment));
                    try {
                        if (hold == null) continue;
                        acquireSegmentAction = new AcquireSegmentAction(this.makeOnDemandLoadSupplier((SegmentCacheEntry)hold.getEntry(), location), hold);
                    }
                    catch (Throwable t) {
                        throw CloseableUtils.closeAndWrapInCatch((Throwable)t, hold);
                    }
                    return acquireSegmentAction;
                }
                throw DruidException.forPersona((DruidException.Persona)DruidException.Persona.USER).ofCategory(DruidException.Category.CAPACITY_EXCEEDED).build("Unable to load segment[%s] on demand, ensure enough disk space has been allocated to load all segments involved in the query", new Object[]{dataSegment.getId()});
            }
            finally {
                this.unlock(dataSegment, lock);
            }
        }
    }

    @Nullable
    private AcquireSegmentAction acquireExistingSegment(SegmentCacheEntryIdentifier identifier) {
        Closer safetyNet = Closer.create();
        for (StorageLocation location : this.locations) {
            try {
                StorageLocation.ReservationHold hold = (StorageLocation.ReservationHold)safetyNet.register(location.addWeakReservationHoldIfExists(identifier));
                if (hold == null) continue;
                if (((SegmentCacheEntry)hold.getEntry()).isMounted()) {
                    return new AcquireSegmentAction(() -> Futures.immediateFuture((Object)((SegmentCacheEntry)hold.getEntry()).referenceProvider), (Closeable)hold);
                }
                return new AcquireSegmentAction(this.makeOnDemandLoadSupplier((SegmentCacheEntry)hold.getEntry(), location), (Closeable)hold);
            }
            catch (Throwable t) {
                throw CloseableUtils.closeAndWrapInCatch((Throwable)t, (Closeable)safetyNet);
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void load(DataSegment dataSegment) throws SegmentLoadingException {
        ReferenceCountingLock lock;
        if (this.config.isVirtualStorage()) {
            return;
        }
        SegmentCacheEntry cacheEntry = new SegmentCacheEntry(dataSegment);
        ReferenceCountingLock referenceCountingLock = lock = this.lock(dataSegment);
        synchronized (referenceCountingLock) {
            try {
                SegmentCacheEntry entry = this.assignLocationAndMount(cacheEntry, SegmentLazyLoadFailCallback.NOOP);
                if (this.loadOnDownloadExec != null) {
                    this.loadOnDownloadExec.submit(entry::loadIntoPageCache);
                }
            }
            finally {
                this.unlock(dataSegment, lock);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void bootstrap(DataSegment dataSegment, SegmentLazyLoadFailCallback loadFailed) throws SegmentLoadingException {
        ReferenceCountingLock lock;
        if (this.config.isVirtualStorage()) {
            ReferenceCountingLock lock2;
            SegmentCacheEntryIdentifier id = new SegmentCacheEntryIdentifier(dataSegment.getId());
            ReferenceCountingLock referenceCountingLock = lock2 = this.lock(dataSegment);
            synchronized (referenceCountingLock) {
                try {
                    for (StorageLocation location : this.locations) {
                        SegmentCacheEntry entry = (SegmentCacheEntry)location.getCacheEntry(id);
                        if (entry == null) continue;
                        entry.lazyLoadCallback = loadFailed;
                        entry.mount(location);
                    }
                }
                finally {
                    this.unlock(dataSegment, lock2);
                }
            }
            return;
        }
        ReferenceCountingLock referenceCountingLock = lock = this.lock(dataSegment);
        synchronized (referenceCountingLock) {
            try {
                SegmentCacheEntry entry = this.assignLocationAndMount(new SegmentCacheEntry(dataSegment), loadFailed);
                if (this.loadOnBootstrapExec != null) {
                    this.loadOnBootstrapExec.submit(entry::loadIntoPageCache);
                }
            }
            finally {
                this.unlock(dataSegment, lock);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    @Nullable
    public File getSegmentFiles(DataSegment segment) {
        ReferenceCountingLock lock;
        SegmentCacheEntry cacheEntry = new SegmentCacheEntry(segment);
        ReferenceCountingLock referenceCountingLock = lock = this.lock(segment);
        synchronized (referenceCountingLock) {
            try {
                StorageLocation location;
                SegmentCacheEntry entry;
                Iterator<StorageLocation> iterator = this.locations.iterator();
                do {
                    if (!iterator.hasNext()) return null;
                } while ((entry = (SegmentCacheEntry)(location = iterator.next()).getCacheEntry(cacheEntry.id)) == null);
                File file = entry.storageDir;
                return file;
            }
            finally {
                this.unlock(segment, lock);
            }
        }
    }

    @Override
    public void drop(DataSegment segment) {
        SegmentCacheEntryIdentifier id = new SegmentCacheEntryIdentifier(segment.getId());
        for (StorageLocation location : this.locations) {
            SegmentCacheEntry entry = (SegmentCacheEntry)location.getCacheEntry(id);
            if (entry == null) continue;
            location.release(entry);
        }
    }

    @Override
    public void shutdownBootstrap() {
        if (this.loadOnBootstrapExec == null) {
            return;
        }
        this.loadOnBootstrapExec.shutdown();
    }

    @Override
    public void shutdown() {
        if (this.loadOnDownloadExec != null) {
            this.loadOnDownloadExec.shutdown();
        }
        if (this.virtualStorageLoadOnDemandExec != null) {
            this.virtualStorageLoadOnDemandExec.shutdown();
        }
    }

    @VisibleForTesting
    public ConcurrentHashMap<DataSegment, ReferenceCountingLock> getSegmentLocks() {
        return this.segmentLocks;
    }

    @VisibleForTesting
    List<StorageLocation> getLocations() {
        return this.locations;
    }

    @VisibleForTesting
    boolean isSegmentCached(DataSegment segment) {
        SegmentCacheEntry cacheEntry = new SegmentCacheEntry(segment);
        for (StorageLocation location : this.locations) {
            if (!cacheEntry.checkExists(location.getPath())) continue;
            return true;
        }
        return false;
    }

    private File getEffectiveInfoDir() {
        File infoDir;
        if (this.config.getInfoDir() != null) {
            infoDir = this.config.getInfoDir();
        } else if (!this.config.getLocations().isEmpty()) {
            infoDir = new File(this.config.getLocations().get(0).getPath(), "info_dir");
        } else if (!this.locations.isEmpty()) {
            infoDir = new File(this.locations.get(0).getPath(), "info_dir");
        } else {
            throw DruidException.forPersona((DruidException.Persona)DruidException.Persona.OPERATOR).ofCategory(DruidException.Category.NOT_FOUND).build("Could not determine infoDir. Make sure 'druid.segmentCache.infoDir' or 'druid.segmentCache.locations' is set correctly.", new Object[0]);
        }
        return infoDir;
    }

    private Supplier<ListenableFuture<ReferenceCountedObjectProvider<Segment>>> makeOnDemandLoadSupplier(SegmentCacheEntry entry, StorageLocation location) {
        return Suppliers.memoize(() -> this.virtualStorageLoadOnDemandExec.submit(() -> {
            entry.mount(location);
            return entry.referenceProvider;
        }));
    }

    private ReferenceCountingLock lock(DataSegment dataSegment) {
        return this.segmentLocks.compute(dataSegment, (segment, lock) -> {
            ReferenceCountingLock nonNullLock = lock == null ? new ReferenceCountingLock() : lock;
            nonNullLock.increment();
            return nonNullLock;
        });
    }

    private void unlock(DataSegment dataSegment, ReferenceCountingLock lock) {
        this.segmentLocks.compute(dataSegment, (segment, existingLock) -> {
            if (existingLock == null) {
                throw new ISE("Lock has already been removed", new Object[0]);
            }
            if (existingLock != lock) {
                throw new ISE("Different lock instance", new Object[0]);
            }
            if (existingLock.numReferences == 1) {
                return null;
            }
            existingLock.decrement();
            return existingLock;
        });
    }

    private SegmentCacheEntry assignLocationAndMount(SegmentCacheEntry cacheEntry, SegmentLazyLoadFailCallback segmentLoadFailCallback) throws SegmentLoadingException {
        SegmentCacheEntry entry;
        try {
            for (StorageLocation location : this.locations) {
                if (!cacheEntry.checkExists(location.getPath())) continue;
                if (location.isReserved(cacheEntry.id) || location.reserve(cacheEntry)) {
                    entry = (SegmentCacheEntry)location.getCacheEntry(cacheEntry.id);
                    if (entry == null) continue;
                    entry.lazyLoadCallback = segmentLoadFailCallback;
                    entry.mount(location);
                    return entry;
                }
                SegmentLocalCacheManager.deleteCacheEntryDirectory(cacheEntry.toPotentialLocation(location.getPath()));
            }
        }
        catch (SegmentLoadingException e) {
            log.warn((Throwable)e, "Failed to load segment[%s] in existing location, trying new location", new Object[]{cacheEntry.id});
        }
        Iterator<StorageLocation> locationsIterator = this.strategy.getLocations();
        while (locationsIterator.hasNext()) {
            StorageLocation location;
            location = locationsIterator.next();
            if (!location.reserve(cacheEntry)) continue;
            try {
                entry = (SegmentCacheEntry)location.getCacheEntry(cacheEntry.id);
                if (entry == null) continue;
                entry.lazyLoadCallback = segmentLoadFailCallback;
                entry.mount(location);
                return entry;
            }
            catch (SegmentLoadingException e) {
                log.warn((Throwable)e, "Failed to load segment[%s] in location[%s], trying next location", new Object[]{cacheEntry.id, location.getPath()});
            }
        }
        throw new SegmentLoadingException("Failed to load segment[%s] in all locations.", new Object[]{cacheEntry.id});
    }

    private static void deleteCacheEntryDirectory(File path) {
        log.info("Deleting directory[%s]", new Object[]{path});
        try {
            FileUtils.deleteDirectory((File)path);
        }
        catch (Exception e) {
            log.error((Throwable)e, "Unable to remove directory[%s]", new Object[]{path});
        }
    }

    private static void cleanupLegacyCacheLocation(File baseFile, File cacheFile) {
        File[] children;
        if (cacheFile.equals(baseFile)) {
            return;
        }
        SegmentLocalCacheManager.deleteCacheEntryDirectory(cacheFile);
        File parent = cacheFile.getParentFile();
        if (parent != null && ((children = parent.listFiles()) == null || children.length == 0)) {
            SegmentLocalCacheManager.cleanupLegacyCacheLocation(baseFile, parent);
        }
    }

    private static boolean isPossiblyCorrupted(File dir) {
        return SegmentLocalCacheManager.hasStartMarker(dir);
    }

    private static boolean hasStartMarker(File localStorageDir) {
        File downloadStartMarker = new File(localStorageDir.getPath(), DOWNLOAD_START_MARKER_FILE_NAME);
        return downloadStartMarker.exists();
    }

    private final class SegmentCacheEntry
    implements CacheEntry {
        private final SegmentCacheEntryIdentifier id;
        private final DataSegment dataSegment;
        private final String relativePathString;
        private SegmentLazyLoadFailCallback lazyLoadCallback = SegmentLazyLoadFailCallback.NOOP;
        private StorageLocation location;
        private File storageDir;
        private ReferenceCountedSegmentProvider referenceProvider;

        private SegmentCacheEntry(DataSegment dataSegment) {
            this.dataSegment = dataSegment;
            this.id = new SegmentCacheEntryIdentifier(dataSegment.getId());
            this.relativePathString = dataSegment.getId().toString();
        }

        @Override
        public SegmentCacheEntryIdentifier getId() {
            return this.id;
        }

        @Override
        public long getSize() {
            return this.dataSegment.getSize();
        }

        @Override
        public synchronized boolean isMounted() {
            return this.referenceProvider != null;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void mount(StorageLocation mountLocation) throws SegmentLoadingException {
            if (!mountLocation.isReserved(this.id) && !mountLocation.isWeakReserved(this.id)) {
                log.debug("aborting mount in location[%s] since entry[%s] is no longer reserved", new Object[]{mountLocation.getPath(), this.id});
                return;
            }
            try {
                SegmentCacheEntry segmentCacheEntry = this;
                synchronized (segmentCacheEntry) {
                    if (this.location != null) {
                        log.debug("already mounted [%s] in location[%s], but asked to load in [%s], unmounting old location", new Object[]{this.id, this.location.getPath(), mountLocation.getPath()});
                        if (!this.location.equals(mountLocation)) {
                            throw DruidException.defensive((String)"already mounted[%s] in location[%s] which is different from requested[%s]", (Object[])new Object[]{this.id, this.location.getPath(), mountLocation.getPath()});
                        }
                        log.debug("already mounted [%s] in location[%s]", new Object[]{this.id, mountLocation.getPath()});
                        return;
                    }
                    this.location = mountLocation;
                    this.storageDir = new File(this.location.getPath(), this.relativePathString);
                    boolean needsLoad = true;
                    if (this.storageDir.exists()) {
                        if (SegmentLocalCacheManager.isPossiblyCorrupted(this.storageDir)) {
                            log.warn("[%s] may be damaged. Delete all the segment files and pull from DeepStorage again.", new Object[]{this.storageDir.getAbsolutePath()});
                            SegmentLocalCacheManager.deleteCacheEntryDirectory(this.storageDir);
                        } else {
                            needsLoad = false;
                        }
                    }
                    if (needsLoad) {
                        this.loadInLocationWithStartMarker(this.dataSegment, this.storageDir);
                    }
                    SegmentizerFactory factory = this.getSegmentFactory(this.storageDir);
                    boolean lazy = SegmentLocalCacheManager.this.config.isLazyLoadOnStart() && this.lazyLoadCallback != SegmentLazyLoadFailCallback.NOOP;
                    Segment segment = factory.factorize(this.dataSegment, this.storageDir, lazy, this.lazyLoadCallback);
                    this.lazyLoadCallback = SegmentLazyLoadFailCallback.NOOP;
                    this.referenceProvider = ReferenceCountedSegmentProvider.of((Segment)segment);
                }
                if (!mountLocation.isReserved(this.id) && !mountLocation.isWeakReserved(this.id)) {
                    log.debug("aborting mount in location[%s] since entry[%s] is no longer reserved", new Object[]{mountLocation.getPath(), this.id});
                    this.unmount();
                }
            }
            catch (SegmentLoadingException e) {
                log.makeAlert((Throwable)e, "Failed to load segment in current location [%s], try next location if any", new Object[]{this.location.getPath().getAbsolutePath()}).addData("location", (Object)this.location.getPath().getAbsolutePath()).emit();
                throw new SegmentLoadingException("Failed to load segment[%s] in reserved location[%s]", new Object[]{this.dataSegment.getId(), this.location.getPath().getAbsolutePath()});
            }
            finally {
                this.unmount();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void unmount() {
            Lock lock;
            SegmentCacheEntry segmentCacheEntry = this;
            synchronized (segmentCacheEntry) {
                if (this.location == null) {
                    return;
                }
                lock = this.location.getLock().readLock();
            }
            lock.lock();
            try {
                segmentCacheEntry = this;
                synchronized (segmentCacheEntry) {
                    block14: {
                        if (this.referenceProvider != null) {
                            this.referenceProvider.close();
                            this.referenceProvider = null;
                        }
                        if (SegmentLocalCacheManager.this.config.isDeleteOnRemove()) break block14;
                        return;
                    }
                    if (this.storageDir != null) {
                        SegmentLocalCacheManager.deleteCacheEntryDirectory(this.storageDir);
                        this.storageDir = null;
                        this.location = null;
                    }
                }
            }
            finally {
                lock.unlock();
            }
        }

        public synchronized Optional<Segment> acquireReference() {
            if (this.referenceProvider == null) {
                return Optional.empty();
            }
            return this.referenceProvider.acquireReference();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void loadIntoPageCache() {
            if (!this.isMounted()) {
                return;
            }
            SegmentCacheEntry segmentCacheEntry = this;
            synchronized (segmentCacheEntry) {
                File[] children = this.storageDir.listFiles();
                if (children != null) {
                    for (File child : children) {
                        try (InputStream in = Files.newInputStream(child.toPath(), new OpenOption[0]);){
                            IOUtils.copy((InputStream)in, (OutputStream)NullOutputStream.NULL_OUTPUT_STREAM);
                            log.info("Loaded [%s] into page cache.", new Object[]{child.getAbsolutePath()});
                        }
                        catch (Exception e) {
                            log.error((Throwable)e, "Failed to load [%s] into page cache", new Object[]{child.getAbsolutePath()});
                        }
                    }
                }
            }
        }

        public boolean checkExists(File location) {
            return this.toPotentialLocation(location).exists();
        }

        public File toPotentialLocation(File location) {
            return new File(location, this.relativePathString);
        }

        @GuardedBy(value="this")
        private void loadInLocationWithStartMarker(DataSegment segment, File storageDir) throws SegmentLoadingException {
            File downloadStartMarker = new File(storageDir, SegmentLocalCacheManager.DOWNLOAD_START_MARKER_FILE_NAME);
            try {
                FileUtils.mkdirp((File)storageDir);
                if (!downloadStartMarker.createNewFile()) {
                    throw new SegmentLoadingException("Was not able to create new download marker for [%s]", new Object[]{storageDir});
                }
                this.loadInLocation(segment, storageDir);
                if (!downloadStartMarker.delete()) {
                    throw new SegmentLoadingException("Unable to remove marker file for [%s]", new Object[]{storageDir});
                }
            }
            catch (IOException e) {
                throw new SegmentLoadingException((Throwable)e, "Unable to create marker file for [%s]", new Object[]{storageDir});
            }
        }

        @GuardedBy(value="this")
        private void loadInLocation(DataSegment segment, File storageDir) throws SegmentLoadingException {
            LoadSpec loadSpec = (LoadSpec)SegmentLocalCacheManager.this.jsonMapper.convertValue((Object)segment.getLoadSpec(), LoadSpec.class);
            LoadSpec.LoadSpecResult result = loadSpec.loadSegment(storageDir);
            if (result.getSize() != segment.getSize()) {
                log.warn("Segment [%s] is different than expected size. Expected [%d] found [%d]", new Object[]{segment.getId(), segment.getSize(), result.getSize()});
            }
        }

        @GuardedBy(value="this")
        private SegmentizerFactory getSegmentFactory(File segmentFiles) throws SegmentLoadingException {
            MMappedQueryableSegmentizerFactory factory;
            File factoryJson = new File(segmentFiles, "factory.json");
            if (factoryJson.exists()) {
                try {
                    factory = (SegmentizerFactory)SegmentLocalCacheManager.this.jsonMapper.readValue(factoryJson, SegmentizerFactory.class);
                }
                catch (IOException e) {
                    throw new SegmentLoadingException((Throwable)e, "Failed to get segment factory for %s", new Object[]{e.getMessage()});
                }
            } else {
                factory = new MMappedQueryableSegmentizerFactory(SegmentLocalCacheManager.this.indexIO);
            }
            return factory;
        }

        public boolean equals(Object o) {
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            SegmentCacheEntry that = (SegmentCacheEntry)o;
            return Objects.equals(this.dataSegment, that.dataSegment);
        }

        public int hashCode() {
            return Objects.hashCode(this.dataSegment);
        }
    }

    private static final class ReferenceCountingLock {
        private int numReferences;

        private ReferenceCountingLock() {
        }

        private void increment() {
            ++this.numReferences;
        }

        private void decrement() {
            --this.numReferences;
        }
    }
}

