/*
 * Decompiled with CFR 0.152.
 */
package org.apache.curator;

import com.google.common.io.Closeables;
import java.io.Closeable;
import java.io.IOException;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.curator.HandleHolder;
import org.apache.curator.SessionFailRetryLoop;
import org.apache.curator.TimeTrace;
import org.apache.curator.drivers.TracerDriver;
import org.apache.curator.ensemble.EnsembleProvider;
import org.apache.curator.utils.ZookeeperFactory;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ConnectionState
implements Watcher,
Closeable {
    private volatile long connectionStartMs = 0L;
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final HandleHolder zooKeeper;
    private final AtomicBoolean isConnected = new AtomicBoolean(false);
    private final AtomicBoolean lost = new AtomicBoolean(false);
    private final EnsembleProvider ensembleProvider;
    private final int connectionTimeoutMs;
    private final AtomicReference<TracerDriver> tracer;
    private final Queue<Exception> backgroundExceptions = new ConcurrentLinkedQueue<Exception>();
    private final Queue<Watcher> parentWatchers = new ConcurrentLinkedQueue<Watcher>();
    private static final int MAX_BACKGROUND_EXCEPTIONS = 10;
    private static final boolean LOG_EVENTS = Boolean.getBoolean("curator-log-events");

    ConnectionState(ZookeeperFactory zookeeperFactory, EnsembleProvider ensembleProvider, int sessionTimeoutMs, int connectionTimeoutMs, Watcher parentWatcher, AtomicReference<TracerDriver> tracer, boolean canBeReadOnly) {
        this.ensembleProvider = ensembleProvider;
        this.connectionTimeoutMs = connectionTimeoutMs;
        this.tracer = tracer;
        if (parentWatcher != null) {
            this.parentWatchers.offer(parentWatcher);
        }
        this.zooKeeper = new HandleHolder(zookeeperFactory, this, ensembleProvider, sessionTimeoutMs, canBeReadOnly);
    }

    ZooKeeper getZooKeeper() throws Exception {
        long elapsed;
        Exception exception;
        if (SessionFailRetryLoop.sessionForThreadHasFailed()) {
            throw new SessionFailRetryLoop.SessionFailedException();
        }
        if (this.lost.compareAndSet(true, false)) {
            this.log.info("resetting after loss");
            this.reset();
        }
        if ((exception = this.backgroundExceptions.poll()) != null) {
            this.log.error("Background exception caught", (Throwable)exception);
            this.tracer.get().addCount("background-exceptions", 1);
            throw exception;
        }
        boolean localIsConnected = this.isConnected.get();
        if (!localIsConnected && (elapsed = System.currentTimeMillis() - this.connectionStartMs) >= (long)this.connectionTimeoutMs) {
            if (this.zooKeeper.hasNewConnectionString()) {
                this.handleNewConnectionString();
            } else {
                KeeperException.ConnectionLossException connectionLossException = new KeeperException.ConnectionLossException();
                if (!Boolean.getBoolean("curator-dont-log-connection-problems")) {
                    this.log.error(String.format("Connection timed out for connection string (%s) and timeout (%d) / elapsed (%d)", this.zooKeeper.getConnectionString(), this.connectionTimeoutMs, elapsed), (Throwable)connectionLossException);
                }
                this.tracer.get().addCount("connections-timed-out", 1);
                throw connectionLossException;
            }
        }
        return this.zooKeeper.getZooKeeper();
    }

    boolean isConnected() {
        return this.isConnected.get();
    }

    void start() throws Exception {
        this.log.debug("Starting");
        this.ensembleProvider.start();
        this.reset();
    }

    @Override
    public void close() throws IOException {
        this.log.debug("Closing");
        Closeables.closeQuietly((Closeable)this.ensembleProvider);
        try {
            this.zooKeeper.closeAndClear();
        }
        catch (Exception e) {
            throw new IOException(e);
        }
        finally {
            this.isConnected.set(false);
            this.lost.set(false);
        }
    }

    void addParentWatcher(Watcher watcher) {
        this.parentWatchers.offer(watcher);
    }

    void removeParentWatcher(Watcher watcher) {
        this.parentWatchers.remove(watcher);
    }

    void markLost() {
        this.log.info("lost marked");
        this.lost.set(true);
    }

    public void process(WatchedEvent event) {
        boolean wasConnected;
        if (LOG_EVENTS) {
            this.log.debug("ConnectState watcher: " + event);
        }
        for (Watcher parentWatcher : this.parentWatchers) {
            TimeTrace timeTrace = new TimeTrace("connection-state-parent-process", this.tracer.get());
            parentWatcher.process(event);
            timeTrace.commit();
        }
        boolean newIsConnected = wasConnected = this.isConnected.get();
        if (event.getType() == Watcher.Event.EventType.None) {
            newIsConnected = this.checkState(event.getState(), wasConnected);
        }
        if (newIsConnected != wasConnected) {
            this.isConnected.set(newIsConnected);
            if (newIsConnected) {
                this.lost.set(false);
            }
            this.connectionStartMs = System.currentTimeMillis();
        }
    }

    EnsembleProvider getEnsembleProvider() {
        return this.ensembleProvider;
    }

    private synchronized void reset() throws Exception {
        this.log.debug("reset");
        this.isConnected.set(false);
        this.connectionStartMs = System.currentTimeMillis();
        this.zooKeeper.closeAndReset();
        this.zooKeeper.getZooKeeper();
    }

    private boolean checkState(Watcher.Event.KeeperState state, boolean wasConnected) {
        boolean isConnected = wasConnected;
        boolean checkNewConnectionString = true;
        switch (state) {
            default: {
                isConnected = false;
                break;
            }
            case SyncConnected: 
            case ConnectedReadOnly: {
                isConnected = true;
                break;
            }
            case AuthFailed: {
                isConnected = false;
                this.log.error("Authentication failed");
                break;
            }
            case Expired: {
                isConnected = false;
                checkNewConnectionString = false;
                this.handleExpiredSession();
            }
            case SaslAuthenticated: 
        }
        if (checkNewConnectionString && this.zooKeeper.hasNewConnectionString()) {
            this.handleNewConnectionString();
        }
        return isConnected;
    }

    private void handleNewConnectionString() {
        this.log.info("Connection string changed");
        this.tracer.get().addCount("connection-string-changed", 1);
        try {
            this.reset();
        }
        catch (Exception e) {
            this.queueBackgroundException(e);
        }
    }

    private void handleExpiredSession() {
        this.log.warn("Session expired event received");
        this.tracer.get().addCount("session-expired", 1);
        try {
            this.reset();
        }
        catch (Exception e) {
            this.queueBackgroundException(e);
        }
    }

    private void queueBackgroundException(Exception e) {
        while (this.backgroundExceptions.size() >= 10) {
            this.backgroundExceptions.poll();
        }
        this.backgroundExceptions.offer(e);
    }
}

