/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.rng.examples.stress;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Formatter;
import java.util.List;
import java.util.concurrent.Callable;
import org.apache.commons.rng.UniformRandomProvider;
import org.apache.commons.rng.core.source64.RandomLongSource;
import org.apache.commons.rng.examples.stress.ApplicationException;
import org.apache.commons.rng.examples.stress.Hex;
import org.apache.commons.rng.examples.stress.LogUtils;
import org.apache.commons.rng.examples.stress.RNGUtils;
import org.apache.commons.rng.examples.stress.RngDataOutput;
import org.apache.commons.rng.examples.stress.StandardOptions;
import org.apache.commons.rng.simple.RandomSource;
import picocli.CommandLine;

@CommandLine.Command(name="output", description={"Output data from a named random data generator."})
class OutputCommand
implements Callable<Void> {
    private static final String NEW_LINE = System.lineSeparator();
    private static final char LEFT_SQUARE_BRACKET = '[';
    private static final char RIGHT_SQUARE_BRACKET = ']';
    private static final String[] BIT_REP = new String[]{"0000", "0001", "0010", "0011", "0100", "0101", "0110", "0111", "1000", "1001", "1010", "1011", "1100", "1101", "1110", "1111"};
    @CommandLine.Mixin
    private StandardOptions reusableOptions;
    @CommandLine.Parameters(index="0", description={"The random source."})
    private RandomSource randomSource;
    @CommandLine.Parameters(index="1..*", description={"The arguments to pass to the constructor."}, paramLabel="<argument>")
    private List<String> arguments = new ArrayList<String>();
    @CommandLine.Option(names={"-o", "--out"}, description={"The output file (default: stdout)."})
    private File fileOutput;
    @CommandLine.Option(names={"-f", "--format"}, description={"Output format (default: ${DEFAULT-VALUE}).", "Valid values: ${COMPLETION-CANDIDATES}."})
    private OutputFormat outputFormat = OutputFormat.DIEHARDER;
    @CommandLine.Option(names={"-s", "--seed"}, description={"The 64-bit number random seed (default: auto)."})
    private Long seed;
    @CommandLine.Option(names={"-x", "--hex-seed"}, description={"The hex-encoded random seed.", "Seed conversion for multi-byte primitives use little-endian format.", "Over-rides the --seed parameter."})
    private String byteSeed;
    @CommandLine.Option(names={"-n", "--count"}, description={"The count of numbers to output.", "Use negative for an unlimited stream."})
    private long count = 10L;
    @CommandLine.Option(names={"--buffer-size"}, description={"Byte-buffer size for binary data (default: ${DEFAULT-VALUE}).", "When outputing binary data the count parameter controls the number of buffers written."})
    private int bufferSize = 8192;
    @CommandLine.Option(names={"-b", "--byte-order"}, description={"Byte-order of the output data (default: ${DEFAULT-VALUE}).", "Uses the Java default of big-endian. This may not match the platform byte-order.", "Valid values: BIG_ENDIAN, LITTLE_ENDIAN."})
    private ByteOrder byteOrder = ByteOrder.BIG_ENDIAN;
    @CommandLine.Option(names={"-r", "--reverse-bits"}, description={"Reverse the bits in the data (default: ${DEFAULT-VALUE})."})
    private boolean reverseBits;
    @CommandLine.Option(names={"--high-bits"}, description={"Use the upper 32-bits from the 64-bit long output.", "Takes precedent over --low-bits."})
    private boolean longHighBits;
    @CommandLine.Option(names={"--low-bits"}, description={"Use the lower 32-bits from the 64-bit long output."})
    private boolean longLowBits;
    @CommandLine.Option(names={"--raw64"}, description={"Use 64-bit output (default is 32-bit).", "This is ignored if not a native 64-bit generator.", "In 32-bit mode the output uses the upper then lower bits of 64-bit generators sequentially."})
    private boolean raw64;

    OutputCommand() {
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public Void call() {
        LogUtils.setLogLevel(this.reusableOptions.logLevel);
        Object objectSeed = this.createSeed();
        UniformRandomProvider rng = this.createRNG(objectSeed);
        if (this.longHighBits) {
            rng = RNGUtils.createLongUpperBitsIntProvider(rng);
        } else if (this.longLowBits) {
            rng = RNGUtils.createLongLowerBitsIntProvider(rng);
        }
        if (this.reverseBits) {
            rng = RNGUtils.createReverseBitsProvider(rng);
        }
        if (this.outputFormat != OutputFormat.BINARY) {
            rng = this.toOutputFormat(rng);
        }
        try (OutputStream out = this.createOutputStream();){
            switch (this.outputFormat) {
                case BINARY: {
                    this.writeBinaryData(rng, out);
                    return null;
                }
                case DIEHARDER: {
                    this.writeDieharder(rng, out);
                    return null;
                }
                case BITS: {
                    this.writeBitData(rng, out);
                    return null;
                }
                default: {
                    throw new ApplicationException("Unknown output format: " + (Object)((Object)this.outputFormat));
                }
            }
        }
        catch (IOException ex) {
            throw new ApplicationException("IO error: " + ex.getMessage(), ex);
        }
    }

    private Object createSeed() {
        if (this.byteSeed != null) {
            try {
                return Hex.decodeHex(this.byteSeed);
            }
            catch (IllegalArgumentException ex) {
                throw new ApplicationException("Invalid hex seed: " + ex.getMessage(), ex);
            }
        }
        if (this.seed != null) {
            return this.seed;
        }
        return null;
    }

    private String createSeedString() {
        if (this.byteSeed != null) {
            return this.byteSeed;
        }
        if (this.seed != null) {
            return this.seed.toString();
        }
        return "auto";
    }

    private UniformRandomProvider createRNG(Object objectSeed) {
        if (this.randomSource == null) {
            throw new ApplicationException("Random source is null");
        }
        ArrayList<Object> data = new ArrayList<Object>();
        OutputCommand.stripArrayFormatting(this.arguments);
        for (String argument : this.arguments) {
            data.add(RNGUtils.parseArgument(argument));
        }
        try {
            return RandomSource.create((RandomSource)this.randomSource, (Object)objectSeed, (Object[])data.toArray());
        }
        catch (IllegalArgumentException | IllegalStateException ex) {
            throw new ApplicationException("Failed to create RNG: " + this.randomSource + ". " + ex.getMessage(), ex);
        }
    }

    private static void stripArrayFormatting(List<String> arguments) {
        int size = arguments.size();
        if (size > 1) {
            String last;
            String first = arguments.get(0);
            if (first.charAt(0) == '[') {
                arguments.set(0, first.substring(1));
            }
            if ((last = arguments.get(size - 1)).charAt(last.length() - 1) == ']') {
                arguments.set(size - 1, last.substring(0, last.length() - 1));
            }
        }
        for (int i = 0; i < size; ++i) {
            String argument = arguments.get(i);
            if (!argument.endsWith(",")) continue;
            arguments.set(i, argument.substring(0, argument.length() - 1));
        }
    }

    private UniformRandomProvider toOutputFormat(UniformRandomProvider rng) {
        UniformRandomProvider convertedRng = rng;
        if (rng instanceof RandomLongSource && !this.raw64) {
            convertedRng = RNGUtils.createIntProvider(rng);
        }
        if (this.byteOrder == ByteOrder.LITTLE_ENDIAN) {
            convertedRng = RNGUtils.createReverseBytesProvider(convertedRng);
        }
        return convertedRng;
    }

    private OutputStream createOutputStream() {
        if (this.fileOutput != null) {
            try {
                Files.newOutputStream(this.fileOutput.toPath(), new OpenOption[0]);
            }
            catch (IOException ex) {
                throw new ApplicationException("Failed to create output: " + this.fileOutput, ex);
            }
        }
        return new FilterOutputStream(System.out){

            @Override
            public void close() {
            }
        };
    }

    private static void checkCount(long count, OutputFormat format) {
        if (count <= 0L) {
            throw new ApplicationException((Object)((Object)format) + " format requires a positive count: " + count);
        }
    }

    private void writeDieharder(UniformRandomProvider rng, OutputStream out) throws IOException {
        OutputCommand.checkCount(this.count, OutputFormat.DIEHARDER);
        try (BufferedWriter output = new BufferedWriter(new OutputStreamWriter(out, StandardCharsets.UTF_8));){
            OutputCommand.writeHeaderLine(output);
            output.write("# generator ");
            output.write(rng.toString());
            output.write("  seed = ");
            output.write(this.createSeedString());
            output.write(NEW_LINE);
            OutputCommand.writeHeaderLine(output);
            output.write("type: d");
            output.write(NEW_LINE);
            output.write("count: ");
            output.write(Long.toString(this.count));
            output.write(NEW_LINE);
            output.write("numbit: 32");
            output.write(NEW_LINE);
            for (long c = 0L; c < this.count; ++c) {
                String text = Long.toString((long)rng.nextInt() & 0xFFFFFFFFL);
                for (int i = 10 - text.length(); i > 0; --i) {
                    output.write(32);
                }
                output.write(text);
                output.write(NEW_LINE);
            }
        }
    }

    private static void writeHeaderLine(Writer output) throws IOException {
        output.write("#==================================================================");
        output.write(NEW_LINE);
    }

    private void writeBinaryData(UniformRandomProvider rng, OutputStream out) throws IOException {
        long limit = this.count < 1L ? Long.MAX_VALUE : this.count;
        try (RngDataOutput data = RNGUtils.createDataOutput(rng, this.raw64, out, this.bufferSize, this.byteOrder);){
            for (long c = 0L; c < limit; ++c) {
                data.write(rng);
            }
        }
    }

    private void writeBitData(UniformRandomProvider rng, OutputStream out) throws IOException {
        OutputCommand.checkCount(this.count, OutputFormat.BITS);
        boolean asLong = rng instanceof RandomLongSource;
        try (BufferedWriter output = new BufferedWriter(new OutputStreamWriter(out, StandardCharsets.UTF_8));){
            for (long c = 0L; c < this.count; ++c) {
                if (asLong) {
                    OutputCommand.writeLong(output, rng.nextLong());
                    continue;
                }
                OutputCommand.writeInt(output, rng.nextInt());
            }
        }
    }

    static void writeLong(Writer out, long value) throws IOException {
        OutputCommand.writeByte(out, (int)(value >>> 56) & 0xFF);
        out.write(32);
        OutputCommand.writeByte(out, (int)(value >>> 48) & 0xFF);
        out.write(32);
        OutputCommand.writeByte(out, (int)(value >>> 40) & 0xFF);
        out.write(32);
        OutputCommand.writeByte(out, (int)(value >>> 32) & 0xFF);
        out.write(32);
        OutputCommand.writeByte(out, (int)(value >>> 24) & 0xFF);
        out.write(32);
        OutputCommand.writeByte(out, (int)(value >>> 16) & 0xFF);
        out.write(32);
        OutputCommand.writeByte(out, (int)(value >>> 8) & 0xFF);
        out.write(32);
        OutputCommand.writeByte(out, (int)(value >>> 0) & 0xFF);
        new Formatter(out).format("  %20s %20d%n", Long.toUnsignedString(value), value);
    }

    static void writeInt(Writer out, int value) throws IOException {
        OutputCommand.writeByte(out, value >>> 24 & 0xFF);
        out.write(32);
        OutputCommand.writeByte(out, value >>> 16 & 0xFF);
        out.write(32);
        OutputCommand.writeByte(out, value >>> 8 & 0xFF);
        out.write(32);
        OutputCommand.writeByte(out, value >>> 0 & 0xFF);
        new Formatter(out).format("  %10d %11d%n", (long)value & 0xFFFFFFFFL, value);
    }

    private static void writeByte(Writer out, int value) throws IOException {
        out.write(BIT_REP[value >>> 4]);
        out.write(BIT_REP[value & 0xF]);
    }

    static enum OutputFormat {
        BINARY,
        DIEHARDER,
        BITS;

    }
}

