/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.csv;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.util.zip.GZIPInputStream;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.csv.ExtendedBufferedReader;
import org.apache.commons.csv.Lexer;
import org.apache.commons.csv.Token;
import org.apache.commons.io.IOUtils;

public class PerformanceTest {
    private static final String[] PROPS = new String[]{"java.version", "java.vendor", "java.vm.version", "java.vm.name", "os.name", "os.arch", "os.version"};
    private static int max = 11;
    private static int num = 0;
    private static long[] elapsedTimes = new long[max];
    private static final CSVFormat format = CSVFormat.EXCEL;
    private static final File BIG_FILE = new File("src/test/resources/perf/worldcitiespop.txt");

    public static void main(String[] args) throws Exception {
        String[] tests;
        if (BIG_FILE.exists()) {
            System.out.printf("Found test fixture %s: %,d bytes.%n", BIG_FILE, BIG_FILE.length());
        } else {
            File compressedFile = new File(BIG_FILE.getParentFile(), BIG_FILE.getName() + ".gz");
            System.out.printf("Decompressing test fixture %s...%n", compressedFile);
            long bytesOut = 0L;
            try (GZIPInputStream input = new GZIPInputStream(new FileInputStream(compressedFile));
                 FileOutputStream output = new FileOutputStream(BIG_FILE);){
                bytesOut = IOUtils.copy((InputStream)input, (OutputStream)output);
            }
            System.out.printf("Decompressed test fixture %s: %,d bytes to: %s: %,d bytes.%n", compressedFile, compressedFile.length(), BIG_FILE, bytesOut);
        }
        int argc = args.length;
        if (argc > 0) {
            max = Integer.parseInt(args[0]);
        }
        if (argc > 1) {
            tests = new String[argc - 1];
            for (int i = 1; i < argc; ++i) {
                tests[i - 1] = args[i];
            }
        } else {
            tests = new String[]{"file", "split", "extb", "exts", "csv", "csv-path", "csv-path-db", "csv-url", "lexreset", "lexnew"};
        }
        for (String p : PROPS) {
            System.out.printf("%s=%s%n", p, System.getProperty(p));
        }
        System.out.printf("Max count: %d%n%n", max);
        for (String test : tests) {
            if ("file".equals(test)) {
                PerformanceTest.testReadBigFile(false);
                continue;
            }
            if ("split".equals(test)) {
                PerformanceTest.testReadBigFile(true);
                continue;
            }
            if ("csv".equals(test)) {
                PerformanceTest.testParseCommonsCSV();
                continue;
            }
            if ("csv-path".equals(test)) {
                PerformanceTest.testParsePath();
                continue;
            }
            if ("csv-path-db".equals(test)) {
                PerformanceTest.testParsePathDoubleBuffering();
                continue;
            }
            if ("csv-url".equals(test)) {
                PerformanceTest.testParseURL();
                continue;
            }
            if ("lexreset".equals(test)) {
                PerformanceTest.testCSVLexer(false, test);
                continue;
            }
            if ("lexnew".equals(test)) {
                PerformanceTest.testCSVLexer(true, test);
                continue;
            }
            if (test.startsWith("CSVLexer")) {
                PerformanceTest.testCSVLexer(false, test);
                continue;
            }
            if ("extb".equals(test)) {
                PerformanceTest.testExtendedBuffer(false);
                continue;
            }
            if ("exts".equals(test)) {
                PerformanceTest.testExtendedBuffer(true);
                continue;
            }
            System.out.printf("Invalid test name: %s%n", test);
        }
    }

    private static Reader createReader() throws IOException {
        return new InputStreamReader((InputStream)new FileInputStream(BIG_FILE), StandardCharsets.ISO_8859_1);
    }

    private static void show(String msg, Stats s, long start) {
        long elapsed = System.currentTimeMillis() - start;
        System.out.printf("%-20s: %5dms %d lines %d fields%n", msg, elapsed, s.count, s.fields);
        PerformanceTest.elapsedTimes[PerformanceTest.num] = elapsed;
        ++num;
    }

    private static void show() {
        if (num > 1) {
            long tot = 0L;
            for (int i = 1; i < num; ++i) {
                tot += elapsedTimes[i];
            }
            System.out.printf("%-20s: %5dms%n%n", "Average(not first)", tot / (long)(num - 1));
        }
        num = 0;
    }

    private static void testReadBigFile(boolean split) throws Exception {
        for (int i = 0; i < max; ++i) {
            Stats stats;
            long startMillis;
            try (BufferedReader in = new BufferedReader(PerformanceTest.createReader());){
                startMillis = System.currentTimeMillis();
                stats = PerformanceTest.readAll(in, split);
            }
            PerformanceTest.show(split ? "file+split" : "file", stats, startMillis);
        }
        PerformanceTest.show();
    }

    private static Stats readAll(BufferedReader in, boolean split) throws IOException {
        String record;
        int count = 0;
        int fields = 0;
        while ((record = in.readLine()) != null) {
            ++count;
            fields += split ? record.split(",").length : 1;
        }
        return new Stats(count, fields);
    }

    private static void testExtendedBuffer(boolean makeString) throws Exception {
        for (int i = 0; i < max; ++i) {
            long startMillis;
            int fields = 0;
            int lines = 0;
            try (ExtendedBufferedReader in = new ExtendedBufferedReader(PerformanceTest.createReader());){
                int read;
                startMillis = System.currentTimeMillis();
                if (makeString) {
                    StringBuilder sb = new StringBuilder();
                    while ((read = in.read()) != -1) {
                        sb.append((char)read);
                        if (read == 44) {
                            sb.toString();
                            sb = new StringBuilder();
                            ++fields;
                            continue;
                        }
                        if (read != 10) continue;
                        sb.toString();
                        sb = new StringBuilder();
                        ++lines;
                    }
                } else {
                    while ((read = in.read()) != -1) {
                        if (read == 44) {
                            ++fields;
                            continue;
                        }
                        if (read != 10) continue;
                        ++lines;
                    }
                }
            }
            PerformanceTest.show("Extended" + (makeString ? " toString" : ""), new Stats(lines, fields += lines), startMillis);
        }
        PerformanceTest.show();
    }

    private static void testParser(String msg, CSVParserFactory fac) throws Exception {
        for (int i = 0; i < max; ++i) {
            Stats stats;
            long startMillis;
            try (CSVParser parser = fac.createParser();){
                startMillis = System.currentTimeMillis();
                stats = PerformanceTest.iterate((Iterable<CSVRecord>)parser);
            }
            PerformanceTest.show(msg, stats, startMillis);
        }
        PerformanceTest.show();
    }

    private static void testParseCommonsCSV() throws Exception {
        PerformanceTest.testParser("CSV", () -> new CSVParser(PerformanceTest.createReader(), format));
    }

    private static void testParsePath() throws Exception {
        PerformanceTest.testParser("CSV-PATH", () -> CSVParser.parse((InputStream)Files.newInputStream(Paths.get(BIG_FILE.toURI()), new OpenOption[0]), (Charset)StandardCharsets.ISO_8859_1, (CSVFormat)format));
    }

    private static void testParsePathDoubleBuffering() throws Exception {
        PerformanceTest.testParser("CSV-PATH-DB", () -> CSVParser.parse((Reader)Files.newBufferedReader(Paths.get(BIG_FILE.toURI()), StandardCharsets.ISO_8859_1), (CSVFormat)format));
    }

    private static void testParseURL() throws Exception {
        PerformanceTest.testParser("CSV-URL", () -> CSVParser.parse((URL)BIG_FILE.toURI().toURL(), (Charset)StandardCharsets.ISO_8859_1, (CSVFormat)format));
    }

    private static Constructor<Lexer> getLexerCtor(String clazz) throws Exception {
        Class<?> lexer = Class.forName("org.apache.commons.csv." + clazz);
        return lexer.getConstructor(CSVFormat.class, ExtendedBufferedReader.class);
    }

    private static void testCSVLexer(boolean newToken, String test) throws Exception {
        Token token = new Token();
        String dynamic = "";
        for (int i = 0; i < max; ++i) {
            Stats stats;
            long startMillis;
            String simpleName;
            try (ExtendedBufferedReader input = new ExtendedBufferedReader(PerformanceTest.createReader());
                 Lexer lexer = PerformanceTest.createTestCSVLexer(test, input);){
                if (test.startsWith("CSVLexer")) {
                    dynamic = "!";
                }
                simpleName = lexer.getClass().getSimpleName();
                int count = 0;
                int fields = 0;
                startMillis = System.currentTimeMillis();
                do {
                    if (newToken) {
                        token = new Token();
                    } else {
                        token.reset();
                    }
                    lexer.nextToken(token);
                    switch (token.type) {
                        case EOF: {
                            break;
                        }
                        case EORECORD: {
                            ++fields;
                            ++count;
                            break;
                        }
                        case INVALID: {
                            throw new IOException("invalid parse sequence <" + token.content.toString() + ">");
                        }
                        case TOKEN: {
                            ++fields;
                            break;
                        }
                        case COMMENT: {
                            break;
                        }
                        default: {
                            throw new IllegalStateException("Unexpected Token type: " + token.type);
                        }
                    }
                } while (!token.type.equals((Object)Token.Type.EOF));
                stats = new Stats(count, fields);
            }
            PerformanceTest.show(simpleName + dynamic + " " + (newToken ? "new" : "reset"), stats, startMillis);
        }
        PerformanceTest.show();
    }

    private static Lexer createTestCSVLexer(String test, ExtendedBufferedReader input) throws InstantiationException, IllegalAccessException, InvocationTargetException, Exception {
        return test.startsWith("CSVLexer") ? PerformanceTest.getLexerCtor(test).newInstance(format, input) : new Lexer(format, input);
    }

    private static Stats iterate(Iterable<CSVRecord> it) {
        int count = 0;
        int fields = 0;
        for (CSVRecord record : it) {
            ++count;
            fields += record.size();
        }
        return new Stats(count, fields);
    }

    @FunctionalInterface
    private static interface CSVParserFactory {
        public CSVParser createParser() throws IOException;
    }

    private static class Stats {
        final int count;
        final int fields;

        Stats(int c, int f) {
            this.count = c;
            this.fields = f;
        }
    }
}

