/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.csv;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PipedReader;
import java.io.PipedWriter;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.csv.CSVException;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVPrinter;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.csv.Utils;
import org.apache.commons.io.input.BOMInputStream;
import org.apache.commons.io.input.BrokenInputStream;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.EnumSource;

public class CSVParserTest {
    private static final Charset UTF_8 = StandardCharsets.UTF_8;
    private static final String UTF_8_NAME = UTF_8.name();
    private static final String CSV_INPUT = "a,b,c,d\n a , b , 1 2 \n\"foo baar\", b,\n   \"foo\n,,\n\"\",,\n\"\"\",d,e\n";
    private static final String CSV_INPUT_1 = "a,b,c,d";
    private static final String CSV_INPUT_2 = "a,b,1 2";
    private static final String[][] RESULT = new String[][]{{"a", "b", "c", "d"}, {"a", "b", "1 2"}, {"foo baar", "b", ""}, {"foo\n,,\n\",,\n\"", "d", "e"}};
    private static final String CSV_INPUT_NO_COMMENT = "A,B\r\n1,2\r\n";
    private static final String CSV_INPUT_HEADER_COMMENT = "# header comment\r\nA,B\r\n1,2\r\n";
    private static final String CSV_INPUT_HEADER_TRAILER_COMMENT = "# header comment\r\nA,B\r\n1,2\r\n# comment";
    private static final String CSV_INPUT_MULTILINE_HEADER_TRAILER_COMMENT = "# multi-line\r\n# header comment\r\nA,B\r\n1,2\r\n# multi-line\r\n# comment";
    private static final CSVFormat FORMAT_AUTO_HEADER = CSVFormat.Builder.create((CSVFormat)CSVFormat.DEFAULT).setCommentMarker('#').setHeader(new String[0]).build();
    private static final CSVFormat FORMAT_EXPLICIT_HEADER = CSVFormat.Builder.create((CSVFormat)CSVFormat.DEFAULT).setSkipHeaderRecord(true).setCommentMarker('#').setHeader(new String[]{"A", "B"}).build();
    CSVFormat FORMAT_EXPLICIT_HEADER_NOSKIP = CSVFormat.Builder.create((CSVFormat)CSVFormat.DEFAULT).setCommentMarker('#').setHeader(new String[]{"A", "B"}).build();

    private BOMInputStream createBOMInputStream(String resource) throws IOException {
        return new BOMInputStream(ClassLoader.getSystemClassLoader().getResource(resource).openStream());
    }

    CSVRecord parse(CSVParser parser, int failParseRecordNo) throws IOException {
        if (parser.getRecordNumber() + 1L == (long)failParseRecordNo) {
            Assertions.assertThrows(IOException.class, () -> parser.nextRecord());
            return null;
        }
        return parser.nextRecord();
    }

    private void parseFully(CSVParser parser) {
        parser.forEach(Assertions::assertNotNull);
    }

    @Test
    public void testBackslashEscaping() throws IOException {
        String code = "one,two,three\n'',''\n/',/'\n'/'','/''\n'''',''''\n/,,/,\n//,//\n'//','//'\n   8   ,   \"quoted \"\" /\" // string\"   \n9,   /\n   \n";
        String[][] res = new String[][]{{"one", "two", "three"}, {"", ""}, {"'", "'"}, {"'", "'"}, {"'", "'"}, {",", ","}, {"/", "/"}, {"/", "/"}, {"   8   ", "   \"quoted \"\" /\" / string\"   "}, {"9", "   \n   "}};
        CSVFormat format = CSVFormat.newFormat((char)',').withQuote('\'').withRecordSeparator("\r\n").withEscape('/').withIgnoreEmptyLines();
        try (CSVParser parser = CSVParser.parse((String)"one,two,three\n'',''\n/',/'\n'/'','/''\n'''',''''\n/,,/,\n//,//\n'//','//'\n   8   ,   \"quoted \"\" /\" // string\"   \n9,   /\n   \n", (CSVFormat)format);){
            List records = parser.getRecords();
            Assertions.assertFalse((boolean)records.isEmpty());
            Utils.compare("Records do not match expected result", res, records);
        }
    }

    @Test
    public void testBackslashEscaping2() throws IOException {
        String code = " , , \n \t ,  , \n // , /, , /,\n";
        String[][] res = new String[][]{{" ", " ", " "}, {" \t ", "  ", " "}, {" / ", " , ", " ,"}};
        CSVFormat format = CSVFormat.newFormat((char)',').withRecordSeparator("\r\n").withEscape('/').withIgnoreEmptyLines();
        try (CSVParser parser = CSVParser.parse((String)" , , \n \t ,  , \n // , /, , /,\n", (CSVFormat)format);){
            List records = parser.getRecords();
            Assertions.assertFalse((boolean)records.isEmpty());
            Utils.compare("", res, records);
        }
    }

    @Test
    @Disabled
    public void testBackslashEscapingOld() throws IOException {
        String code = "one,two,three\non\\\"e,two\non\"e,two\none,\"tw\\\"o\"\none,\"t\\,wo\"\none,two,\"th,ree\"\n\"a\\\\\"\na\\,b\n\"a\\\\,b\"";
        String[][] res = new String[][]{{"one", "two", "three"}, {"on\\\"e", "two"}, {"on\"e", "two"}, {"one", "tw\"o"}, {"one", "t\\,wo"}, {"one", "two", "th,ree"}, {"a\\\\"}, {"a\\", "b"}, {"a\\\\,b"}};
        try (CSVParser parser = CSVParser.parse((String)"one,two,three\non\\\"e,two\non\"e,two\none,\"tw\\\"o\"\none,\"t\\,wo\"\none,two,\"th,ree\"\n\"a\\\\\"\na\\,b\n\"a\\\\,b\"", (CSVFormat)CSVFormat.DEFAULT);){
            List records = parser.getRecords();
            Assertions.assertEquals((int)res.length, (int)records.size());
            Assertions.assertFalse((boolean)records.isEmpty());
            for (int i = 0; i < res.length; ++i) {
                Assertions.assertArrayEquals((Object[])res[i], (Object[])((CSVRecord)records.get(i)).values());
            }
        }
    }

    @Test
    @Disabled(value="CSV-107")
    public void testBOM() throws IOException {
        URL url = ClassLoader.getSystemClassLoader().getResource("org/apache/commons/csv/CSVFileParser/bom.csv");
        try (CSVParser parser = CSVParser.parse((URL)url, (Charset)StandardCharsets.UTF_8, (CSVFormat)CSVFormat.EXCEL.withHeader(new String[0]));){
            parser.forEach(record -> Assertions.assertNotNull((Object)record.get("Date")));
        }
    }

    @Test
    public void testBOMInputStreamParserWithInputStream() throws IOException {
        try (BOMInputStream inputStream = this.createBOMInputStream("org/apache/commons/csv/CSVFileParser/bom.csv");
             CSVParser parser = CSVParser.parse((InputStream)inputStream, (Charset)UTF_8, (CSVFormat)CSVFormat.EXCEL.withHeader(new String[0]));){
            parser.forEach(record -> Assertions.assertNotNull((Object)record.get("Date")));
        }
    }

    @Test
    public void testBOMInputStreamParserWithReader() throws IOException {
        try (InputStreamReader reader = new InputStreamReader((InputStream)this.createBOMInputStream("org/apache/commons/csv/CSVFileParser/bom.csv"), UTF_8_NAME);
             CSVParser parser = new CSVParser((Reader)reader, CSVFormat.EXCEL.withHeader(new String[0]));){
            parser.forEach(record -> Assertions.assertNotNull((Object)record.get("Date")));
        }
    }

    @Test
    public void testBOMInputStreamParseWithReader() throws IOException {
        try (InputStreamReader reader = new InputStreamReader((InputStream)this.createBOMInputStream("org/apache/commons/csv/CSVFileParser/bom.csv"), UTF_8_NAME);
             CSVParser parser = CSVParser.parse((Reader)reader, (CSVFormat)CSVFormat.EXCEL.withHeader(new String[0]));){
            parser.forEach(record -> Assertions.assertNotNull((Object)record.get("Date")));
        }
    }

    @Test
    public void testCarriageReturnEndings() throws IOException {
        String code = "foo\rbaar,\rhello,world\r,kanu";
        try (CSVParser parser = CSVParser.parse((String)"foo\rbaar,\rhello,world\r,kanu", (CSVFormat)CSVFormat.DEFAULT);){
            List records = parser.getRecords();
            Assertions.assertEquals((int)4, (int)records.size());
        }
    }

    @Test
    public void testCarriageReturnLineFeedEndings() throws IOException {
        String code = "foo\r\nbaar,\r\nhello,world\r\n,kanu";
        try (CSVParser parser = CSVParser.parse((String)"foo\r\nbaar,\r\nhello,world\r\n,kanu", (CSVFormat)CSVFormat.DEFAULT);){
            List records = parser.getRecords();
            Assertions.assertEquals((int)4, (int)records.size());
        }
    }

    @Test
    public void testClose() throws Exception {
        Iterator records;
        StringReader in = new StringReader("# comment\na,b,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withCommentMarker('#').withHeader(new String[0]).parse((Reader)in);){
            records = parser.iterator();
            Assertions.assertTrue((boolean)records.hasNext());
        }
        Assertions.assertFalse((boolean)records.hasNext());
        Assertions.assertThrows(NoSuchElementException.class, records::next);
    }

    @Test
    public void testCSV141CSVFormat_DEFAULT() throws Exception {
        this.testCSV141Failure(CSVFormat.DEFAULT, 3);
    }

    @Test
    public void testCSV141CSVFormat_INFORMIX_UNLOAD() throws Exception {
        this.testCSV141Failure(CSVFormat.INFORMIX_UNLOAD, 1);
    }

    @Test
    public void testCSV141CSVFormat_INFORMIX_UNLOAD_CSV() throws Exception {
        this.testCSV141Failure(CSVFormat.INFORMIX_UNLOAD_CSV, 3);
    }

    @Test
    public void testCSV141CSVFormat_ORACLE() throws Exception {
        this.testCSV141Failure(CSVFormat.ORACLE, 2);
    }

    @Test
    public void testCSV141CSVFormat_POSTGRESQL_CSV() throws Exception {
        this.testCSV141Failure(CSVFormat.POSTGRESQL_CSV, 3);
    }

    @Test
    public void testCSV141Excel() throws Exception {
        this.testCSV141Ok(CSVFormat.EXCEL);
    }

    private void testCSV141Failure(CSVFormat format, int failParseRecordNo) throws IOException {
        Path path = Paths.get("src/test/resources/org/apache/commons/csv/CSV-141/csv-141.csv", new String[0]);
        try (CSVParser parser = CSVParser.parse((Path)path, (Charset)StandardCharsets.UTF_8, (CSVFormat)format);){
            CSVRecord record = this.parse(parser, failParseRecordNo);
            if (record == null) {
                return;
            }
            Assertions.assertEquals((Object)"1414770317901", (Object)record.get(0));
            Assertions.assertEquals((Object)"android.widget.EditText", (Object)record.get(1));
            Assertions.assertEquals((Object)"pass sem1 _84*|*", (Object)record.get(2));
            Assertions.assertEquals((Object)"0", (Object)record.get(3));
            Assertions.assertEquals((Object)"pass sem1 _8", (Object)record.get(4));
            Assertions.assertEquals((int)5, (int)record.size());
            record = this.parse(parser, failParseRecordNo);
            if (record == null) {
                return;
            }
            Assertions.assertEquals((Object)"1414770318470", (Object)record.get(0));
            Assertions.assertEquals((Object)"android.widget.EditText", (Object)record.get(1));
            Assertions.assertEquals((Object)"pass sem1 _84:|", (Object)record.get(2));
            Assertions.assertEquals((Object)"0", (Object)record.get(3));
            Assertions.assertEquals((Object)"pass sem1 _84:\\", (Object)record.get(4));
            Assertions.assertEquals((int)5, (int)record.size());
            Assertions.assertThrows(IOException.class, () -> parser.nextRecord());
        }
    }

    private void testCSV141Ok(CSVFormat format) throws IOException {
        Path path = Paths.get("src/test/resources/org/apache/commons/csv/CSV-141/csv-141.csv", new String[0]);
        try (CSVParser parser = CSVParser.parse((Path)path, (Charset)StandardCharsets.UTF_8, (CSVFormat)format);){
            CSVRecord record = parser.nextRecord();
            Assertions.assertEquals((Object)"1414770317901", (Object)record.get(0));
            Assertions.assertEquals((Object)"android.widget.EditText", (Object)record.get(1));
            Assertions.assertEquals((Object)"pass sem1 _84*|*", (Object)record.get(2));
            Assertions.assertEquals((Object)"0", (Object)record.get(3));
            Assertions.assertEquals((Object)"pass sem1 _8", (Object)record.get(4));
            Assertions.assertEquals((int)5, (int)record.size());
            record = parser.nextRecord();
            Assertions.assertEquals((Object)"1414770318470", (Object)record.get(0));
            Assertions.assertEquals((Object)"android.widget.EditText", (Object)record.get(1));
            Assertions.assertEquals((Object)"pass sem1 _84:|", (Object)record.get(2));
            Assertions.assertEquals((Object)"0", (Object)record.get(3));
            Assertions.assertEquals((Object)"pass sem1 _84:\\", (Object)record.get(4));
            Assertions.assertEquals((int)5, (int)record.size());
            record = parser.nextRecord();
            Assertions.assertEquals((Object)"1414770318327", (Object)record.get(0));
            Assertions.assertEquals((Object)"android.widget.EditText", (Object)record.get(1));
            Assertions.assertEquals((Object)"pass sem1\n1414770318628\"", (Object)record.get(2));
            Assertions.assertEquals((Object)"android.widget.EditText", (Object)record.get(3));
            Assertions.assertEquals((Object)"pass sem1 _84*|*", (Object)record.get(4));
            Assertions.assertEquals((Object)"0", (Object)record.get(5));
            Assertions.assertEquals((Object)"pass sem1\n", (Object)record.get(6));
            Assertions.assertEquals((int)7, (int)record.size());
            record = parser.nextRecord();
            Assertions.assertNull((Object)record);
        }
    }

    @Test
    public void testCSV141RFC4180() throws Exception {
        this.testCSV141Failure(CSVFormat.RFC4180, 3);
    }

    @Test
    public void testCSV235() throws IOException {
        String dqString = "\"aaa\",\"b\"\"bb\",\"ccc\"";
        try (CSVParser parser = CSVFormat.RFC4180.parse((Reader)new StringReader("\"aaa\",\"b\"\"bb\",\"ccc\""));){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            Assertions.assertFalse((boolean)records.hasNext());
            Assertions.assertEquals((int)3, (int)record.size());
            Assertions.assertEquals((Object)"aaa", (Object)record.get(0));
            Assertions.assertEquals((Object)"b\"bb", (Object)record.get(1));
            Assertions.assertEquals((Object)"ccc", (Object)record.get(2));
        }
    }

    @Test
    public void testCSV57() throws Exception {
        try (CSVParser parser = CSVParser.parse((String)"", (CSVFormat)CSVFormat.DEFAULT);){
            List list = parser.getRecords();
            Assertions.assertNotNull((Object)list);
            Assertions.assertEquals((int)0, (int)list.size());
        }
    }

    @Test
    public void testDefaultFormat() throws IOException {
        List records;
        String code = "a,b#\n\"\n\",\" \",#\n#,\"\"\n# Final comment\n";
        String[][] res = new String[][]{{"a", "b#"}, {"\n", " ", "#"}, {"#", ""}, {"# Final comment"}};
        CSVFormat format = CSVFormat.DEFAULT;
        Assertions.assertFalse((boolean)format.isCommentMarkerSet());
        String[][] res_comments = new String[][]{{"a", "b#"}, {"\n", " ", "#"}};
        try (CSVParser parser = CSVParser.parse((String)"a,b#\n\"\n\",\" \",#\n#,\"\"\n# Final comment\n", (CSVFormat)format);){
            records = parser.getRecords();
            Assertions.assertFalse((boolean)records.isEmpty());
            Utils.compare("Failed to parse without comments", res, records);
            format = CSVFormat.DEFAULT.withCommentMarker('#');
        }
        parser = CSVParser.parse((String)"a,b#\n\"\n\",\" \",#\n#,\"\"\n# Final comment\n", (CSVFormat)format);
        try {
            records = parser.getRecords();
            Utils.compare("Failed to parse with comments", res_comments, records);
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
    }

    @Test
    public void testDuplicateHeadersAllowedByDefault() throws Exception {
        CSVParser parser = CSVParser.parse((String)"a,b,a\n1,2,3\nx,y,z", (CSVFormat)CSVFormat.DEFAULT.withHeader(new String[0]));
        if (parser != null) {
            parser.close();
        }
    }

    @Test
    public void testDuplicateHeadersNotAllowed() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> CSVParser.parse((String)"a,b,a\n1,2,3\nx,y,z", (CSVFormat)CSVFormat.DEFAULT.withHeader(new String[0]).withAllowDuplicateHeaderNames(false)));
    }

    @Test
    public void testEmptyFile() throws Exception {
        try (CSVParser parser = CSVParser.parse((Path)Paths.get("src/test/resources/org/apache/commons/csv/empty.txt", new String[0]), (Charset)StandardCharsets.UTF_8, (CSVFormat)CSVFormat.DEFAULT);){
            Assertions.assertNull((Object)parser.nextRecord());
        }
    }

    @Test
    public void testEmptyFileHeaderParsing() throws Exception {
        try (CSVParser parser = CSVParser.parse((String)"", (CSVFormat)CSVFormat.DEFAULT.withFirstRecordAsHeader());){
            Assertions.assertNull((Object)parser.nextRecord());
            Assertions.assertTrue((boolean)parser.getHeaderNames().isEmpty());
        }
    }

    @Test
    public void testEmptyLineBehaviorCSV() throws Exception {
        String[] codes = new String[]{"hello,\r\n\r\n\r\n", "hello,\n\n\n", "hello,\"\"\r\n\r\n\r\n", "hello,\"\"\n\n\n"};
        String[][] res = new String[][]{{"hello", ""}};
        for (String code : codes) {
            try (CSVParser parser = CSVParser.parse((String)code, (CSVFormat)CSVFormat.DEFAULT);){
                List records = parser.getRecords();
                Assertions.assertEquals((int)res.length, (int)records.size());
                Assertions.assertFalse((boolean)records.isEmpty());
                for (int i = 0; i < res.length; ++i) {
                    Assertions.assertArrayEquals((Object[])res[i], (Object[])((CSVRecord)records.get(i)).values());
                }
            }
        }
    }

    @Test
    public void testEmptyLineBehaviorExcel() throws Exception {
        String[] codes = new String[]{"hello,\r\n\r\n\r\n", "hello,\n\n\n", "hello,\"\"\r\n\r\n\r\n", "hello,\"\"\n\n\n"};
        String[][] res = new String[][]{{"hello", ""}, {""}, {""}};
        for (String code : codes) {
            try (CSVParser parser = CSVParser.parse((String)code, (CSVFormat)CSVFormat.EXCEL);){
                List records = parser.getRecords();
                Assertions.assertEquals((int)res.length, (int)records.size());
                Assertions.assertFalse((boolean)records.isEmpty());
                for (int i = 0; i < res.length; ++i) {
                    Assertions.assertArrayEquals((Object[])res[i], (Object[])((CSVRecord)records.get(i)).values());
                }
            }
        }
    }

    @Test
    public void testEmptyString() throws Exception {
        try (CSVParser parser = CSVParser.parse((String)"", (CSVFormat)CSVFormat.DEFAULT);){
            Assertions.assertNull((Object)parser.nextRecord());
        }
    }

    @Test
    public void testEndOfFileBehaviorCSV() throws Exception {
        String[] codes = new String[]{"hello,\r\n\r\nworld,\r\n", "hello,\r\n\r\nworld,", "hello,\r\n\r\nworld,\"\"\r\n", "hello,\r\n\r\nworld,\"\"", "hello,\r\n\r\nworld,\n", "hello,\r\n\r\nworld,", "hello,\r\n\r\nworld,\"\"\n", "hello,\r\n\r\nworld,\"\""};
        String[][] res = new String[][]{{"hello", ""}, {"world", ""}};
        for (String code : codes) {
            try (CSVParser parser = CSVParser.parse((String)code, (CSVFormat)CSVFormat.DEFAULT);){
                List records = parser.getRecords();
                Assertions.assertEquals((int)res.length, (int)records.size());
                Assertions.assertFalse((boolean)records.isEmpty());
                for (int i = 0; i < res.length; ++i) {
                    Assertions.assertArrayEquals((Object[])res[i], (Object[])((CSVRecord)records.get(i)).values());
                }
            }
        }
    }

    @Test
    public void testEndOfFileBehaviorExcel() throws Exception {
        String[] codes = new String[]{"hello,\r\n\r\nworld,\r\n", "hello,\r\n\r\nworld,", "hello,\r\n\r\nworld,\"\"\r\n", "hello,\r\n\r\nworld,\"\"", "hello,\r\n\r\nworld,\n", "hello,\r\n\r\nworld,", "hello,\r\n\r\nworld,\"\"\n", "hello,\r\n\r\nworld,\"\""};
        String[][] res = new String[][]{{"hello", ""}, {""}, {"world", ""}};
        for (String code : codes) {
            try (CSVParser parser = CSVParser.parse((String)code, (CSVFormat)CSVFormat.EXCEL);){
                List records = parser.getRecords();
                Assertions.assertEquals((int)res.length, (int)records.size());
                Assertions.assertFalse((boolean)records.isEmpty());
                for (int i = 0; i < res.length; ++i) {
                    Assertions.assertArrayEquals((Object[])res[i], (Object[])((CSVRecord)records.get(i)).values());
                }
            }
        }
    }

    @Test
    public void testExcelFormat1() throws IOException {
        String code = "value1,value2,value3,value4\r\na,b,c,d\r\n  x,,,\r\n\r\n\"\"\"hello\"\"\",\"  \"\"world\"\"\",\"abc\ndef\",\r\n";
        String[][] res = new String[][]{{"value1", "value2", "value3", "value4"}, {"a", "b", "c", "d"}, {"  x", "", "", ""}, {""}, {"\"hello\"", "  \"world\"", "abc\ndef", ""}};
        try (CSVParser parser = CSVParser.parse((String)"value1,value2,value3,value4\r\na,b,c,d\r\n  x,,,\r\n\r\n\"\"\"hello\"\"\",\"  \"\"world\"\"\",\"abc\ndef\",\r\n", (CSVFormat)CSVFormat.EXCEL);){
            List records = parser.getRecords();
            Assertions.assertEquals((int)res.length, (int)records.size());
            Assertions.assertFalse((boolean)records.isEmpty());
            for (int i = 0; i < res.length; ++i) {
                Assertions.assertArrayEquals((Object[])res[i], (Object[])((CSVRecord)records.get(i)).values());
            }
        }
    }

    @Test
    public void testExcelFormat2() throws Exception {
        String code = "foo,baar\r\n\r\nhello,\r\n\r\nworld,\r\n";
        String[][] res = new String[][]{{"foo", "baar"}, {""}, {"hello", ""}, {""}, {"world", ""}};
        try (CSVParser parser = CSVParser.parse((String)"foo,baar\r\n\r\nhello,\r\n\r\nworld,\r\n", (CSVFormat)CSVFormat.EXCEL);){
            List records = parser.getRecords();
            Assertions.assertEquals((int)res.length, (int)records.size());
            Assertions.assertFalse((boolean)records.isEmpty());
            for (int i = 0; i < res.length; ++i) {
                Assertions.assertArrayEquals((Object[])res[i], (Object[])((CSVRecord)records.get(i)).values());
            }
        }
    }

    @Test
    public void testExcelHeaderCountLessThanData() throws Exception {
        String code = "A,B,C,,\r\na,b,c,d,e\r\n";
        try (CSVParser parser = CSVParser.parse((String)"A,B,C,,\r\na,b,c,d,e\r\n", (CSVFormat)CSVFormat.EXCEL.withHeader(new String[0]));){
            parser.getRecords().forEach(record -> {
                Assertions.assertEquals((Object)"a", (Object)record.get("A"));
                Assertions.assertEquals((Object)"b", (Object)record.get("B"));
                Assertions.assertEquals((Object)"c", (Object)record.get("C"));
            });
        }
    }

    @Test
    public void testFirstEndOfLineCr() throws IOException {
        String data = "foo\rbaar,\rhello,world\r,kanu";
        try (CSVParser parser = CSVParser.parse((String)"foo\rbaar,\rhello,world\r,kanu", (CSVFormat)CSVFormat.DEFAULT);){
            List records = parser.getRecords();
            Assertions.assertEquals((int)4, (int)records.size());
            Assertions.assertEquals((Object)"\r", (Object)parser.getFirstEndOfLine());
        }
    }

    @Test
    public void testFirstEndOfLineCrLf() throws IOException {
        String data = "foo\r\nbaar,\r\nhello,world\r\n,kanu";
        try (CSVParser parser = CSVParser.parse((String)"foo\r\nbaar,\r\nhello,world\r\n,kanu", (CSVFormat)CSVFormat.DEFAULT);){
            List records = parser.getRecords();
            Assertions.assertEquals((int)4, (int)records.size());
            Assertions.assertEquals((Object)"\r\n", (Object)parser.getFirstEndOfLine());
        }
    }

    @Test
    public void testFirstEndOfLineLf() throws IOException {
        String data = "foo\nbaar,\nhello,world\n,kanu";
        try (CSVParser parser = CSVParser.parse((String)"foo\nbaar,\nhello,world\n,kanu", (CSVFormat)CSVFormat.DEFAULT);){
            List records = parser.getRecords();
            Assertions.assertEquals((int)4, (int)records.size());
            Assertions.assertEquals((Object)"\n", (Object)parser.getFirstEndOfLine());
        }
    }

    @Test
    public void testForEach() throws Exception {
        try (StringReader in = new StringReader("a,b,c\n1,2,3\nx,y,z");
             CSVParser parser = CSVFormat.DEFAULT.parse((Reader)in);){
            ArrayList<CSVRecord> records = new ArrayList<CSVRecord>();
            for (CSVRecord record : parser) {
                records.add(record);
            }
            Assertions.assertEquals((int)3, (int)records.size());
            Assertions.assertArrayEquals((Object[])new String[]{"a", "b", "c"}, (Object[])((CSVRecord)records.get(0)).values());
            Assertions.assertArrayEquals((Object[])new String[]{"1", "2", "3"}, (Object[])((CSVRecord)records.get(1)).values());
            Assertions.assertArrayEquals((Object[])new String[]{"x", "y", "z"}, (Object[])((CSVRecord)records.get(2)).values());
        }
    }

    @Test
    public void testGetHeaderComment_HeaderComment1() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_HEADER_COMMENT, (CSVFormat)FORMAT_AUTO_HEADER);){
            parser.getRecords();
            Assertions.assertTrue((boolean)parser.hasHeaderComment());
            Assertions.assertEquals((Object)"header comment", (Object)parser.getHeaderComment());
        }
    }

    @Test
    public void testGetHeaderComment_HeaderComment2() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_HEADER_COMMENT, (CSVFormat)FORMAT_EXPLICIT_HEADER);){
            parser.getRecords();
            Assertions.assertTrue((boolean)parser.hasHeaderComment());
            Assertions.assertEquals((Object)"header comment", (Object)parser.getHeaderComment());
        }
    }

    @Test
    public void testGetHeaderComment_HeaderComment3() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_HEADER_COMMENT, (CSVFormat)this.FORMAT_EXPLICIT_HEADER_NOSKIP);){
            parser.getRecords();
            Assertions.assertFalse((boolean)parser.hasHeaderComment());
            Assertions.assertNull((Object)parser.getHeaderComment());
        }
    }

    @Test
    public void testGetHeaderComment_HeaderTrailerComment() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_MULTILINE_HEADER_TRAILER_COMMENT, (CSVFormat)FORMAT_AUTO_HEADER);){
            parser.getRecords();
            Assertions.assertTrue((boolean)parser.hasHeaderComment());
            Assertions.assertEquals((Object)"multi-line\nheader comment", (Object)parser.getHeaderComment());
        }
    }

    @Test
    public void testGetHeaderComment_NoComment1() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_NO_COMMENT, (CSVFormat)FORMAT_AUTO_HEADER);){
            parser.getRecords();
            Assertions.assertFalse((boolean)parser.hasHeaderComment());
            Assertions.assertNull((Object)parser.getHeaderComment());
        }
    }

    @Test
    public void testGetHeaderComment_NoComment2() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_NO_COMMENT, (CSVFormat)FORMAT_EXPLICIT_HEADER);){
            parser.getRecords();
            Assertions.assertFalse((boolean)parser.hasHeaderComment());
            Assertions.assertNull((Object)parser.getHeaderComment());
        }
    }

    @Test
    public void testGetHeaderComment_NoComment3() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_NO_COMMENT, (CSVFormat)this.FORMAT_EXPLICIT_HEADER_NOSKIP);){
            parser.getRecords();
            Assertions.assertFalse((boolean)parser.hasHeaderComment());
            Assertions.assertNull((Object)parser.getHeaderComment());
        }
    }

    @Test
    public void testGetHeaderMap() throws Exception {
        try (CSVParser parser = CSVParser.parse((String)"a,b,c\n1,2,3\nx,y,z", (CSVFormat)CSVFormat.DEFAULT.withHeader(new String[]{"A", "B", "C"}));){
            Map headerMap = parser.getHeaderMap();
            Iterator columnNames = headerMap.keySet().iterator();
            Assertions.assertEquals((Object)"A", columnNames.next());
            Assertions.assertEquals((Object)"B", columnNames.next());
            Assertions.assertEquals((Object)"C", columnNames.next());
            Iterator records = parser.iterator();
            for (int i = 0; i < 3; ++i) {
                Assertions.assertTrue((boolean)records.hasNext());
                CSVRecord record = (CSVRecord)records.next();
                Assertions.assertEquals((Object)record.get(0), (Object)record.get("A"));
                Assertions.assertEquals((Object)record.get(1), (Object)record.get("B"));
                Assertions.assertEquals((Object)record.get(2), (Object)record.get("C"));
            }
            Assertions.assertFalse((boolean)records.hasNext());
        }
    }

    @Test
    public void testGetHeaderNames() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)"a,b,c\n1,2,3\nx,y,z", (CSVFormat)CSVFormat.DEFAULT.withHeader(new String[]{"A", "B", "C"}));){
            Map nameIndexMap = parser.getHeaderMap();
            List headerNames = parser.getHeaderNames();
            Assertions.assertNotNull((Object)headerNames);
            Assertions.assertEquals((int)nameIndexMap.size(), (int)headerNames.size());
            for (int i = 0; i < headerNames.size(); ++i) {
                String name = (String)headerNames.get(i);
                Assertions.assertEquals((int)i, (int)((Integer)nameIndexMap.get(name)));
            }
        }
    }

    @Test
    public void testGetHeaderNamesReadOnly() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)"a,b,c\n1,2,3\nx,y,z", (CSVFormat)CSVFormat.DEFAULT.withHeader(new String[]{"A", "B", "C"}));){
            List headerNames = parser.getHeaderNames();
            Assertions.assertNotNull((Object)headerNames);
            Assertions.assertThrows(UnsupportedOperationException.class, () -> headerNames.add("This is a read-only list."));
        }
    }

    @Test
    public void testGetLine() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT, (CSVFormat)CSVFormat.DEFAULT.withIgnoreSurroundingSpaces());){
            for (Object[] objectArray : RESULT) {
                Assertions.assertArrayEquals((Object[])objectArray, (Object[])parser.nextRecord().values());
            }
            Assertions.assertNull((Object)parser.nextRecord());
        }
    }

    @Test
    public void testGetLineNumberWithCR() throws Exception {
        this.validateLineNumbers(String.valueOf('\r'));
    }

    @Test
    public void testGetLineNumberWithCRLF() throws Exception {
        this.validateLineNumbers("\r\n");
    }

    @Test
    public void testGetLineNumberWithLF() throws Exception {
        this.validateLineNumbers(String.valueOf('\n'));
    }

    @Test
    public void testGetOneLine() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_1, (CSVFormat)CSVFormat.DEFAULT);){
            CSVRecord record = (CSVRecord)parser.getRecords().get(0);
            Assertions.assertArrayEquals((Object[])RESULT[0], (Object[])record.values());
        }
    }

    @Test
    public void testGetOneLineOneParser() throws IOException {
        CSVFormat format = CSVFormat.DEFAULT;
        try (PipedWriter writer = new PipedWriter();
             CSVParser parser = new CSVParser((Reader)new PipedReader(writer), format);){
            writer.append(CSV_INPUT_1);
            writer.append(format.getRecordSeparator());
            CSVRecord record1 = parser.nextRecord();
            Assertions.assertArrayEquals((Object[])RESULT[0], (Object[])record1.values());
            writer.append(CSV_INPUT_2);
            writer.append(format.getRecordSeparator());
            CSVRecord record2 = parser.nextRecord();
            Assertions.assertArrayEquals((Object[])RESULT[1], (Object[])record2.values());
        }
    }

    @Test
    public void testGetRecordNumberWithCR() throws Exception {
        this.validateRecordNumbers(String.valueOf('\r'));
    }

    @Test
    public void testGetRecordNumberWithCRLF() throws Exception {
        this.validateRecordNumbers("\r\n");
    }

    @Test
    public void testGetRecordNumberWithLF() throws Exception {
        this.validateRecordNumbers(String.valueOf('\n'));
    }

    @Test
    public void testGetRecordPositionWithCRLF() throws Exception {
        this.validateRecordPosition("\r\n");
    }

    @Test
    public void testGetRecordPositionWithLF() throws Exception {
        this.validateRecordPosition(String.valueOf('\n'));
    }

    @Test
    public void testGetRecords() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT, (CSVFormat)CSVFormat.DEFAULT.withIgnoreSurroundingSpaces());){
            List records = parser.getRecords();
            Assertions.assertEquals((int)RESULT.length, (int)records.size());
            Assertions.assertFalse((boolean)records.isEmpty());
            for (int i = 0; i < RESULT.length; ++i) {
                Assertions.assertArrayEquals((Object[])RESULT[i], (Object[])((CSVRecord)records.get(i)).values());
            }
        }
    }

    @Test
    public void testGetRecordsFromBrokenInputStream() throws IOException {
        CSVParser parser = CSVParser.parse((InputStream)new BrokenInputStream(), (Charset)UTF_8, (CSVFormat)CSVFormat.DEFAULT);
        Assertions.assertThrows(UncheckedIOException.class, () -> ((CSVParser)parser).getRecords());
    }

    @Test
    public void testGetRecordWithMultiLineValues() throws Exception {
        try (CSVParser parser = CSVParser.parse((String)"\"a\r\n1\",\"a\r\n2\"\r\n\"b\r\n1\",\"b\r\n2\"\r\n\"c\r\n1\",\"c\r\n2\"", (CSVFormat)CSVFormat.DEFAULT.withRecordSeparator("\r\n"));){
            Assertions.assertEquals((long)0L, (long)parser.getRecordNumber());
            Assertions.assertEquals((long)0L, (long)parser.getCurrentLineNumber());
            CSVRecord record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)3L, (long)parser.getCurrentLineNumber());
            Assertions.assertEquals((long)1L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)1L, (long)parser.getRecordNumber());
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)6L, (long)parser.getCurrentLineNumber());
            Assertions.assertEquals((long)2L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)2L, (long)parser.getRecordNumber());
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)9L, (long)parser.getCurrentLineNumber());
            Assertions.assertEquals((long)3L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)3L, (long)parser.getRecordNumber());
            record = parser.nextRecord();
            Assertions.assertNull((Object)record);
            Assertions.assertEquals((long)9L, (long)parser.getCurrentLineNumber());
            Assertions.assertEquals((long)3L, (long)parser.getRecordNumber());
        }
    }

    @Test
    public void testGetTrailerComment_HeaderComment1() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_HEADER_COMMENT, (CSVFormat)FORMAT_AUTO_HEADER);){
            parser.getRecords();
            Assertions.assertFalse((boolean)parser.hasTrailerComment());
            Assertions.assertNull((Object)parser.getTrailerComment());
        }
    }

    @Test
    public void testGetTrailerComment_HeaderComment2() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_HEADER_COMMENT, (CSVFormat)FORMAT_EXPLICIT_HEADER);){
            parser.getRecords();
            Assertions.assertFalse((boolean)parser.hasTrailerComment());
            Assertions.assertNull((Object)parser.getTrailerComment());
        }
    }

    @Test
    public void testGetTrailerComment_HeaderComment3() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_HEADER_COMMENT, (CSVFormat)this.FORMAT_EXPLICIT_HEADER_NOSKIP);){
            parser.getRecords();
            Assertions.assertFalse((boolean)parser.hasTrailerComment());
            Assertions.assertNull((Object)parser.getTrailerComment());
        }
    }

    @Test
    public void testGetTrailerComment_HeaderTrailerComment1() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_HEADER_TRAILER_COMMENT, (CSVFormat)FORMAT_AUTO_HEADER);){
            parser.getRecords();
            Assertions.assertTrue((boolean)parser.hasTrailerComment());
            Assertions.assertEquals((Object)"comment", (Object)parser.getTrailerComment());
        }
    }

    @Test
    public void testGetTrailerComment_HeaderTrailerComment2() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_HEADER_TRAILER_COMMENT, (CSVFormat)FORMAT_EXPLICIT_HEADER);){
            parser.getRecords();
            Assertions.assertTrue((boolean)parser.hasTrailerComment());
            Assertions.assertEquals((Object)"comment", (Object)parser.getTrailerComment());
        }
    }

    @Test
    public void testGetTrailerComment_HeaderTrailerComment3() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_HEADER_TRAILER_COMMENT, (CSVFormat)this.FORMAT_EXPLICIT_HEADER_NOSKIP);){
            parser.getRecords();
            Assertions.assertTrue((boolean)parser.hasTrailerComment());
            Assertions.assertEquals((Object)"comment", (Object)parser.getTrailerComment());
        }
    }

    @Test
    public void testGetTrailerComment_MultilineComment() throws IOException {
        try (CSVParser parser = CSVParser.parse((String)CSV_INPUT_MULTILINE_HEADER_TRAILER_COMMENT, (CSVFormat)FORMAT_AUTO_HEADER);){
            parser.getRecords();
            Assertions.assertTrue((boolean)parser.hasTrailerComment());
            Assertions.assertEquals((Object)"multi-line\ncomment", (Object)parser.getTrailerComment());
        }
    }

    @Test
    public void testHeader() throws Exception {
        StringReader in = new StringReader("a,b,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[0]).parse((Reader)in);){
            Iterator records = parser.iterator();
            for (int i = 0; i < 2; ++i) {
                Assertions.assertTrue((boolean)records.hasNext());
                CSVRecord record = (CSVRecord)records.next();
                Assertions.assertEquals((Object)record.get(0), (Object)record.get("a"));
                Assertions.assertEquals((Object)record.get(1), (Object)record.get("b"));
                Assertions.assertEquals((Object)record.get(2), (Object)record.get("c"));
            }
            Assertions.assertFalse((boolean)records.hasNext());
        }
    }

    @Test
    public void testHeaderComment() throws Exception {
        StringReader in = new StringReader("# comment\na,b,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withCommentMarker('#').withHeader(new String[0]).parse((Reader)in);){
            Iterator records = parser.iterator();
            for (int i = 0; i < 2; ++i) {
                Assertions.assertTrue((boolean)records.hasNext());
                CSVRecord record = (CSVRecord)records.next();
                Assertions.assertEquals((Object)record.get(0), (Object)record.get("a"));
                Assertions.assertEquals((Object)record.get(1), (Object)record.get("b"));
                Assertions.assertEquals((Object)record.get(2), (Object)record.get("c"));
            }
            Assertions.assertFalse((boolean)records.hasNext());
        }
    }

    @Test
    public void testHeaderMissing() throws Exception {
        StringReader in = new StringReader("a,,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[0]).withAllowMissingColumnNames().parse((Reader)in);){
            Iterator records = parser.iterator();
            for (int i = 0; i < 2; ++i) {
                Assertions.assertTrue((boolean)records.hasNext());
                CSVRecord record = (CSVRecord)records.next();
                Assertions.assertEquals((Object)record.get(0), (Object)record.get("a"));
                Assertions.assertEquals((Object)record.get(2), (Object)record.get("c"));
            }
            Assertions.assertFalse((boolean)records.hasNext());
        }
    }

    @Test
    public void testHeaderMissingWithNull() throws Exception {
        StringReader in = new StringReader("a,,c,,e\n1,2,3,4,5\nv,w,x,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[0]).withNullString("").withAllowMissingColumnNames().parse((Reader)in);){
            parser.iterator();
        }
    }

    @Test
    public void testHeadersMissing() throws Exception {
        try (StringReader in = new StringReader("a,,c,,e\n1,2,3,4,5\nv,w,x,y,z");
             CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[0]).withAllowMissingColumnNames().parse((Reader)in);){
            parser.iterator();
        }
    }

    @Test
    public void testHeadersMissingException() {
        StringReader in = new StringReader("a,,c,,e\n1,2,3,4,5\nv,w,x,y,z");
        Assertions.assertThrows(IllegalArgumentException.class, () -> CSVFormat.DEFAULT.withHeader(new String[0]).parse(in).iterator());
    }

    @Test
    public void testHeadersMissingOneColumnException() {
        StringReader in = new StringReader("a,,c,d,e\n1,2,3,4,5\nv,w,x,y,z");
        Assertions.assertThrows(IllegalArgumentException.class, () -> CSVFormat.DEFAULT.withHeader(new String[0]).parse(in).iterator());
    }

    @Test
    public void testHeadersWithNullColumnName() throws IOException {
        StringReader in = new StringReader("header1,null,header3\n1,2,3\n4,5,6");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[0]).withNullString("null").withAllowMissingColumnNames().parse((Reader)in);){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            CSVParser recordParser = record.getParser();
            Assertions.assertEquals(Arrays.asList("header1", "header3"), (Object)recordParser.getHeaderNames());
            Assertions.assertEquals((int)2, (int)recordParser.getHeaderMap().size());
        }
    }

    @Test
    public void testIgnoreCaseHeaderMapping() throws Exception {
        StringReader reader = new StringReader("1,2,3");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[]{"One", "TWO", "three"}).withIgnoreHeaderCase().parse((Reader)reader);){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            Assertions.assertEquals((Object)"1", (Object)record.get("one"));
            Assertions.assertEquals((Object)"2", (Object)record.get("two"));
            Assertions.assertEquals((Object)"3", (Object)record.get("THREE"));
        }
    }

    @Test
    public void testIgnoreEmptyLines() throws IOException {
        String code = "\nfoo,baar\n\r\n,\n\n,world\r\n\n";
        try (CSVParser parser = CSVParser.parse((String)"\nfoo,baar\n\r\n,\n\n,world\r\n\n", (CSVFormat)CSVFormat.DEFAULT);){
            List records = parser.getRecords();
            Assertions.assertEquals((int)3, (int)records.size());
        }
    }

    @Test
    public void testInvalidFormat() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> CSVFormat.DEFAULT.withDelimiter('\r'));
    }

    @Test
    public void testIterator() throws Exception {
        StringReader in = new StringReader("a,b,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.parse((Reader)in);){
            Iterator iterator = parser.iterator();
            Assertions.assertTrue((boolean)iterator.hasNext());
            Assertions.assertThrows(UnsupportedOperationException.class, iterator::remove);
            Assertions.assertArrayEquals((Object[])new String[]{"a", "b", "c"}, (Object[])((CSVRecord)iterator.next()).values());
            Assertions.assertArrayEquals((Object[])new String[]{"1", "2", "3"}, (Object[])((CSVRecord)iterator.next()).values());
            Assertions.assertTrue((boolean)iterator.hasNext());
            Assertions.assertTrue((boolean)iterator.hasNext());
            Assertions.assertTrue((boolean)iterator.hasNext());
            Assertions.assertArrayEquals((Object[])new String[]{"x", "y", "z"}, (Object[])((CSVRecord)iterator.next()).values());
            Assertions.assertFalse((boolean)iterator.hasNext());
            Assertions.assertThrows(NoSuchElementException.class, iterator::next);
        }
    }

    @Test
    public void testIteratorSequenceBreaking() throws IOException {
        int recordNumber;
        String fiveRows = "1\n2\n3\n4\n5\n";
        try (CSVParser parser = CSVFormat.DEFAULT.parse((Reader)new StringReader("1\n2\n3\n4\n5\n"));){
            CSVRecord record;
            Iterator iter = parser.iterator();
            int recordNumber2 = 0;
            while (iter.hasNext()) {
                record = (CSVRecord)iter.next();
                Assertions.assertEquals((Object)String.valueOf(++recordNumber2), (Object)record.get(0));
                if (recordNumber2 < 2) continue;
                break;
            }
            iter.hasNext();
            while (iter.hasNext()) {
                record = (CSVRecord)iter.next();
                Assertions.assertEquals((Object)String.valueOf(++recordNumber2), (Object)record.get(0));
            }
        }
        parser = CSVFormat.DEFAULT.parse((Reader)new StringReader("1\n2\n3\n4\n5\n"));
        try {
            recordNumber = 0;
            for (CSVRecord record : parser) {
                Assertions.assertEquals((Object)String.valueOf(++recordNumber), (Object)record.get(0));
                if (recordNumber < 2) continue;
                break;
            }
            for (CSVRecord record : parser) {
                Assertions.assertEquals((Object)String.valueOf(++recordNumber), (Object)record.get(0));
            }
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
        parser = CSVFormat.DEFAULT.parse((Reader)new StringReader("1\n2\n3\n4\n5\n"));
        try {
            recordNumber = 0;
            for (CSVRecord record : parser) {
                Assertions.assertEquals((Object)String.valueOf(++recordNumber), (Object)record.get(0));
                if (recordNumber < 2) continue;
                break;
            }
            parser.iterator().hasNext();
            for (CSVRecord record : parser) {
                Assertions.assertEquals((Object)String.valueOf(++recordNumber), (Object)record.get(0));
            }
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
    }

    @Test
    public void testLineFeedEndings() throws IOException {
        String code = "foo\nbaar,\nhello,world\n,kanu";
        try (CSVParser parser = CSVParser.parse((String)"foo\nbaar,\nhello,world\n,kanu", (CSVFormat)CSVFormat.DEFAULT);){
            List records = parser.getRecords();
            Assertions.assertEquals((int)4, (int)records.size());
        }
    }

    @Test
    public void testMappedButNotSetAsOutlook2007ContactExport() throws Exception {
        StringReader in = new StringReader("a,b,c\n1,2\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[]{"A", "B", "C"}).withSkipHeaderRecord().parse((Reader)in);){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            Assertions.assertTrue((boolean)record.isMapped("A"));
            Assertions.assertTrue((boolean)record.isMapped("B"));
            Assertions.assertTrue((boolean)record.isMapped("C"));
            Assertions.assertTrue((boolean)record.isSet("A"));
            Assertions.assertTrue((boolean)record.isSet("B"));
            Assertions.assertFalse((boolean)record.isSet("C"));
            Assertions.assertEquals((Object)"1", (Object)record.get("A"));
            Assertions.assertEquals((Object)"2", (Object)record.get("B"));
            Assertions.assertFalse((boolean)record.isConsistent());
            record = (CSVRecord)records.next();
            Assertions.assertTrue((boolean)record.isMapped("A"));
            Assertions.assertTrue((boolean)record.isMapped("B"));
            Assertions.assertTrue((boolean)record.isMapped("C"));
            Assertions.assertTrue((boolean)record.isSet("A"));
            Assertions.assertTrue((boolean)record.isSet("B"));
            Assertions.assertTrue((boolean)record.isSet("C"));
            Assertions.assertEquals((Object)"x", (Object)record.get("A"));
            Assertions.assertEquals((Object)"y", (Object)record.get("B"));
            Assertions.assertEquals((Object)"z", (Object)record.get("C"));
            Assertions.assertTrue((boolean)record.isConsistent());
            Assertions.assertFalse((boolean)records.hasNext());
        }
    }

    @Test
    @Disabled
    public void testMongoDbCsv() throws Exception {
        try (CSVParser parser = CSVParser.parse((String)"\"a a\",b,c\nd,e,f", (CSVFormat)CSVFormat.MONGODB_CSV);){
            Iterator itr1 = parser.iterator();
            Iterator itr2 = parser.iterator();
            CSVRecord first = (CSVRecord)itr1.next();
            Assertions.assertEquals((Object)"a a", (Object)first.get(0));
            Assertions.assertEquals((Object)"b", (Object)first.get(1));
            Assertions.assertEquals((Object)"c", (Object)first.get(2));
            CSVRecord second = (CSVRecord)itr2.next();
            Assertions.assertEquals((Object)"d", (Object)second.get(0));
            Assertions.assertEquals((Object)"e", (Object)second.get(1));
            Assertions.assertEquals((Object)"f", (Object)second.get(2));
        }
    }

    @Test
    public void testMultipleIterators() throws Exception {
        try (CSVParser parser = CSVParser.parse((String)"a,b,c\r\nd,e,f", (CSVFormat)CSVFormat.DEFAULT);){
            Iterator itr1 = parser.iterator();
            CSVRecord first = (CSVRecord)itr1.next();
            Assertions.assertEquals((Object)"a", (Object)first.get(0));
            Assertions.assertEquals((Object)"b", (Object)first.get(1));
            Assertions.assertEquals((Object)"c", (Object)first.get(2));
            CSVRecord second = (CSVRecord)itr1.next();
            Assertions.assertEquals((Object)"d", (Object)second.get(0));
            Assertions.assertEquals((Object)"e", (Object)second.get(1));
            Assertions.assertEquals((Object)"f", (Object)second.get(2));
        }
    }

    @Test
    public void testNewCSVParserNullReaderFormat() {
        Assertions.assertThrows(NullPointerException.class, () -> new CSVParser(null, CSVFormat.DEFAULT));
    }

    @Test
    public void testNewCSVParserReaderNullFormat() {
        Assertions.assertThrows(NullPointerException.class, () -> new CSVParser((Reader)new StringReader(""), null));
    }

    @Test
    public void testNoHeaderMap() throws Exception {
        try (CSVParser parser = CSVParser.parse((String)"a,b,c\n1,2,3\nx,y,z", (CSVFormat)CSVFormat.DEFAULT);){
            Assertions.assertNull((Object)parser.getHeaderMap());
        }
    }

    @Test
    public void testNotValueCSV() throws IOException {
        String source = "#";
        CSVFormat csvFormat = CSVFormat.DEFAULT.withCommentMarker('#');
        try (CSVParser csvParser = csvFormat.parse((Reader)new StringReader("#"));){
            CSVRecord csvRecord = csvParser.nextRecord();
            Assertions.assertNull((Object)csvRecord);
        }
    }

    @Test
    public void testParse() throws Exception {
        ClassLoader loader = ClassLoader.getSystemClassLoader();
        URL url = loader.getResource("org/apache/commons/csv/CSVFileParser/test.csv");
        CSVFormat format = CSVFormat.DEFAULT.withHeader(new String[]{"A", "B", "C", "D"});
        Charset charset = StandardCharsets.UTF_8;
        try (CSVParser parser = CSVParser.parse((Reader)new InputStreamReader(url.openStream(), charset), (CSVFormat)format);){
            this.parseFully(parser);
        }
        parser = CSVParser.parse((String)new String(Files.readAllBytes(Paths.get(url.toURI())), charset), (CSVFormat)format);
        try {
            this.parseFully(parser);
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
        parser = CSVParser.parse((File)new File(url.toURI()), (Charset)charset, (CSVFormat)format);
        try {
            this.parseFully(parser);
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
        parser = CSVParser.parse((InputStream)url.openStream(), (Charset)charset, (CSVFormat)format);
        try {
            this.parseFully(parser);
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
        parser = CSVParser.parse((Path)Paths.get(url.toURI()), (Charset)charset, (CSVFormat)format);
        try {
            this.parseFully(parser);
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
        parser = CSVParser.parse((URL)url, (Charset)charset, (CSVFormat)format);
        try {
            this.parseFully(parser);
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
        parser = new CSVParser((Reader)new InputStreamReader(url.openStream(), charset), format);
        try {
            this.parseFully(parser);
        }
        finally {
            parser.close();
        }
        parser = new CSVParser((Reader)new InputStreamReader(url.openStream(), charset), format, 0L, 1L);
        try {
            this.parseFully(parser);
        }
        finally {
            parser.close();
        }
    }

    @Test
    public void testParseFileNullFormat() {
        Assertions.assertThrows(NullPointerException.class, () -> CSVParser.parse((File)new File("CSVFileParser/test.csv"), (Charset)Charset.defaultCharset(), null));
    }

    @Test
    public void testParseNullFileFormat() {
        Assertions.assertThrows(NullPointerException.class, () -> CSVParser.parse((File)null, (Charset)Charset.defaultCharset(), (CSVFormat)CSVFormat.DEFAULT));
    }

    @Test
    public void testParseNullPathFormat() {
        Assertions.assertThrows(NullPointerException.class, () -> CSVParser.parse((Path)null, (Charset)Charset.defaultCharset(), (CSVFormat)CSVFormat.DEFAULT));
    }

    @Test
    public void testParseNullStringFormat() {
        Assertions.assertThrows(NullPointerException.class, () -> CSVParser.parse((String)null, (CSVFormat)CSVFormat.DEFAULT));
    }

    @Test
    public void testParseNullUrlCharsetFormat() {
        Assertions.assertThrows(NullPointerException.class, () -> CSVParser.parse((URL)null, (Charset)Charset.defaultCharset(), (CSVFormat)CSVFormat.DEFAULT));
    }

    @Test
    public void testParserUrlNullCharsetFormat() {
        Assertions.assertThrows(NullPointerException.class, () -> CSVParser.parse((URL)new URL("https://commons.apache.org"), null, (CSVFormat)CSVFormat.DEFAULT));
    }

    @Test
    public void testParseStringNullFormat() {
        Assertions.assertThrows(NullPointerException.class, () -> CSVParser.parse((String)"csv data", (CSVFormat)null));
    }

    @Test
    public void testParseUrlCharsetNullFormat() {
        Assertions.assertThrows(NullPointerException.class, () -> CSVParser.parse((URL)new URL("https://commons.apache.org"), (Charset)Charset.defaultCharset(), null));
    }

    @Test
    public void testParseWithDelimiterStringWithEscape() throws IOException {
        String source = "a![!|!]b![|]c[|]xyz\r\nabc[abc][|]xyz";
        CSVFormat csvFormat = CSVFormat.DEFAULT.builder().setDelimiter("[|]").setEscape('!').build();
        try (CSVParser csvParser = csvFormat.parse((Reader)new StringReader("a![!|!]b![|]c[|]xyz\r\nabc[abc][|]xyz"));){
            CSVRecord csvRecord = csvParser.nextRecord();
            Assertions.assertEquals((Object)"a[|]b![|]c", (Object)csvRecord.get(0));
            Assertions.assertEquals((Object)"xyz", (Object)csvRecord.get(1));
            csvRecord = csvParser.nextRecord();
            Assertions.assertEquals((Object)"abc[abc]", (Object)csvRecord.get(0));
            Assertions.assertEquals((Object)"xyz", (Object)csvRecord.get(1));
        }
    }

    @Test
    public void testParseWithDelimiterStringWithQuote() throws IOException {
        String source = "'a[|]b[|]c'[|]xyz\r\nabc[abc][|]xyz";
        CSVFormat csvFormat = CSVFormat.DEFAULT.builder().setDelimiter("[|]").setQuote('\'').build();
        try (CSVParser csvParser = csvFormat.parse((Reader)new StringReader("'a[|]b[|]c'[|]xyz\r\nabc[abc][|]xyz"));){
            CSVRecord csvRecord = csvParser.nextRecord();
            Assertions.assertEquals((Object)"a[|]b[|]c", (Object)csvRecord.get(0));
            Assertions.assertEquals((Object)"xyz", (Object)csvRecord.get(1));
            csvRecord = csvParser.nextRecord();
            Assertions.assertEquals((Object)"abc[abc]", (Object)csvRecord.get(0));
            Assertions.assertEquals((Object)"xyz", (Object)csvRecord.get(1));
        }
    }

    @Test
    public void testParseWithDelimiterWithEscape() throws IOException {
        String source = "a!,b!,c,xyz";
        CSVFormat csvFormat = CSVFormat.DEFAULT.withEscape('!');
        try (CSVParser csvParser = csvFormat.parse((Reader)new StringReader("a!,b!,c,xyz"));){
            CSVRecord csvRecord = csvParser.nextRecord();
            Assertions.assertEquals((Object)"a,b,c", (Object)csvRecord.get(0));
            Assertions.assertEquals((Object)"xyz", (Object)csvRecord.get(1));
        }
    }

    @Test
    public void testParseWithDelimiterWithQuote() throws IOException {
        String source = "'a,b,c',xyz";
        CSVFormat csvFormat = CSVFormat.DEFAULT.withQuote('\'');
        try (CSVParser csvParser = csvFormat.parse((Reader)new StringReader("'a,b,c',xyz"));){
            CSVRecord csvRecord = csvParser.nextRecord();
            Assertions.assertEquals((Object)"a,b,c", (Object)csvRecord.get(0));
            Assertions.assertEquals((Object)"xyz", (Object)csvRecord.get(1));
        }
    }

    @Test
    public void testParseWithQuoteThrowsException() {
        CSVFormat csvFormat = CSVFormat.DEFAULT.withQuote('\'');
        Assertions.assertThrows(IOException.class, () -> csvFormat.parse((Reader)new StringReader("'a,b,c','")).nextRecord());
        Assertions.assertThrows(IOException.class, () -> csvFormat.parse((Reader)new StringReader("'a,b,c'abc,xyz")).nextRecord());
        Assertions.assertThrows(IOException.class, () -> csvFormat.parse((Reader)new StringReader("'abc'a,b,c',xyz")).nextRecord());
    }

    @Test
    public void testParseWithQuoteWithEscape() throws IOException {
        String source = "'a?,b?,c?d',xyz";
        CSVFormat csvFormat = CSVFormat.DEFAULT.withQuote('\'').withEscape('?');
        try (CSVParser csvParser = csvFormat.parse((Reader)new StringReader("'a?,b?,c?d',xyz"));){
            CSVRecord csvRecord = csvParser.nextRecord();
            Assertions.assertEquals((Object)"a,b,c?d", (Object)csvRecord.get(0));
            Assertions.assertEquals((Object)"xyz", (Object)csvRecord.get(1));
        }
    }

    @ParameterizedTest
    @EnumSource(value=CSVFormat.Predefined.class)
    public void testParsingPrintedEmptyFirstColumn(CSVFormat.Predefined format) throws Exception {
        String[][] lines = new String[][]{{"a", "b"}, {"", "x"}};
        StringWriter buf = new StringWriter();
        try (CSVPrinter printer = new CSVPrinter((Appendable)buf, format.getFormat());){
            printer.printRecords(Stream.of(lines));
        }
        try (CSVParser csvRecords = new CSVParser((Reader)new StringReader(buf.toString()), format.getFormat());){
            for (Object[] objectArray : lines) {
                Assertions.assertArrayEquals((Object[])objectArray, (Object[])csvRecords.nextRecord().values());
            }
            Assertions.assertNull((Object)csvRecords.nextRecord());
        }
    }

    @Test
    public void testProvidedHeader() throws Exception {
        StringReader in = new StringReader("a,b,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[]{"A", "B", "C"}).parse((Reader)in);){
            Iterator records = parser.iterator();
            for (int i = 0; i < 3; ++i) {
                Assertions.assertTrue((boolean)records.hasNext());
                CSVRecord record = (CSVRecord)records.next();
                Assertions.assertTrue((boolean)record.isMapped("A"));
                Assertions.assertTrue((boolean)record.isMapped("B"));
                Assertions.assertTrue((boolean)record.isMapped("C"));
                Assertions.assertFalse((boolean)record.isMapped("NOT MAPPED"));
                Assertions.assertEquals((Object)record.get(0), (Object)record.get("A"));
                Assertions.assertEquals((Object)record.get(1), (Object)record.get("B"));
                Assertions.assertEquals((Object)record.get(2), (Object)record.get("C"));
            }
            Assertions.assertFalse((boolean)records.hasNext());
        }
    }

    @Test
    public void testProvidedHeaderAuto() throws Exception {
        StringReader in = new StringReader("a,b,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[0]).parse((Reader)in);){
            Iterator records = parser.iterator();
            for (int i = 0; i < 2; ++i) {
                Assertions.assertTrue((boolean)records.hasNext());
                CSVRecord record = (CSVRecord)records.next();
                Assertions.assertTrue((boolean)record.isMapped("a"));
                Assertions.assertTrue((boolean)record.isMapped("b"));
                Assertions.assertTrue((boolean)record.isMapped("c"));
                Assertions.assertFalse((boolean)record.isMapped("NOT MAPPED"));
                Assertions.assertEquals((Object)record.get(0), (Object)record.get("a"));
                Assertions.assertEquals((Object)record.get(1), (Object)record.get("b"));
                Assertions.assertEquals((Object)record.get(2), (Object)record.get("c"));
            }
            Assertions.assertFalse((boolean)records.hasNext());
        }
    }

    @Test
    public void testRepeatedHeadersAreReturnedInCSVRecordHeaderNames() throws IOException {
        StringReader in = new StringReader("header1,header2,header1\n1,2,3\n4,5,6");
        try (CSVParser parser = CSVFormat.DEFAULT.withFirstRecordAsHeader().withTrim().parse((Reader)in);){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            CSVParser recordParser = record.getParser();
            Assertions.assertEquals(Arrays.asList("header1", "header2", "header1"), (Object)recordParser.getHeaderNames());
        }
    }

    @Test
    public void testRoundtrip() throws Exception {
        StringWriter out = new StringWriter();
        String data = "a,b,c\r\n1,2,3\r\nx,y,z\r\n";
        try (CSVPrinter printer = new CSVPrinter((Appendable)out, CSVFormat.DEFAULT);
             CSVParser parse = CSVParser.parse((String)"a,b,c\r\n1,2,3\r\nx,y,z\r\n", (CSVFormat)CSVFormat.DEFAULT);){
            for (CSVRecord record : parse) {
                printer.printRecord((Iterable)record);
            }
            Assertions.assertEquals((Object)"a,b,c\r\n1,2,3\r\nx,y,z\r\n", (Object)out.toString());
        }
    }

    @Test
    public void testSkipAutoHeader() throws Exception {
        StringReader in = new StringReader("a,b,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[0]).parse((Reader)in);){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            Assertions.assertEquals((Object)"1", (Object)record.get("a"));
            Assertions.assertEquals((Object)"2", (Object)record.get("b"));
            Assertions.assertEquals((Object)"3", (Object)record.get("c"));
        }
    }

    @Test
    public void testSkipHeaderOverrideDuplicateHeaders() throws Exception {
        StringReader in = new StringReader("a,a,a\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[]{"X", "Y", "Z"}).withSkipHeaderRecord().parse((Reader)in);){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            Assertions.assertEquals((Object)"1", (Object)record.get("X"));
            Assertions.assertEquals((Object)"2", (Object)record.get("Y"));
            Assertions.assertEquals((Object)"3", (Object)record.get("Z"));
        }
    }

    @Test
    public void testSkipSetAltHeaders() throws Exception {
        StringReader in = new StringReader("a,b,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[]{"X", "Y", "Z"}).withSkipHeaderRecord().parse((Reader)in);){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            Assertions.assertEquals((Object)"1", (Object)record.get("X"));
            Assertions.assertEquals((Object)"2", (Object)record.get("Y"));
            Assertions.assertEquals((Object)"3", (Object)record.get("Z"));
        }
    }

    @Test
    public void testSkipSetHeader() throws Exception {
        StringReader in = new StringReader("a,b,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[]{"a", "b", "c"}).withSkipHeaderRecord().parse((Reader)in);){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            Assertions.assertEquals((Object)"1", (Object)record.get("a"));
            Assertions.assertEquals((Object)"2", (Object)record.get("b"));
            Assertions.assertEquals((Object)"3", (Object)record.get("c"));
        }
    }

    @Test
    @Disabled
    public void testStartWithEmptyLinesThenHeaders() throws Exception {
        String[] codes = new String[]{"\r\n\r\n\r\nhello,\r\n\r\n\r\n", "hello,\n\n\n", "hello,\"\"\r\n\r\n\r\n", "hello,\"\"\n\n\n"};
        String[][] res = new String[][]{{"hello", ""}, {""}, {""}};
        for (String code : codes) {
            try (CSVParser parser = CSVParser.parse((String)code, (CSVFormat)CSVFormat.EXCEL);){
                List records = parser.getRecords();
                Assertions.assertEquals((int)res.length, (int)records.size());
                Assertions.assertFalse((boolean)records.isEmpty());
                for (int i = 0; i < res.length; ++i) {
                    Assertions.assertArrayEquals((Object[])res[i], (Object[])((CSVRecord)records.get(i)).values());
                }
            }
        }
    }

    @Test
    public void testStream() throws Exception {
        StringReader in = new StringReader("a,b,c\n1,2,3\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.parse((Reader)in);){
            List list = parser.stream().collect(Collectors.toList());
            Assertions.assertFalse((boolean)list.isEmpty());
            Assertions.assertArrayEquals((Object[])new String[]{"a", "b", "c"}, (Object[])((CSVRecord)list.get(0)).values());
            Assertions.assertArrayEquals((Object[])new String[]{"1", "2", "3"}, (Object[])((CSVRecord)list.get(1)).values());
            Assertions.assertArrayEquals((Object[])new String[]{"x", "y", "z"}, (Object[])((CSVRecord)list.get(2)).values());
        }
    }

    @Test
    public void testThrowExceptionWithLineAndPosition() throws IOException {
        String csvContent = "col1,col2,col3,col4,col5,col6,col7,col8,col9,col10\nrec1,rec2,rec3,rec4,rec5,rec6,rec7,rec8,\"\"rec9\"\",rec10";
        StringReader stringReader = new StringReader("col1,col2,col3,col4,col5,col6,col7,col8,col9,col10\nrec1,rec2,rec3,rec4,rec5,rec6,rec7,rec8,\"\"rec9\"\",rec10");
        CSVFormat csvFormat = CSVFormat.DEFAULT.builder().setHeader(new String[0]).setSkipHeaderRecord(true).build();
        try (CSVParser csvParser = csvFormat.parse((Reader)stringReader);){
            UncheckedIOException exception = (UncheckedIOException)Assertions.assertThrows(UncheckedIOException.class, () -> ((CSVParser)csvParser).getRecords());
            Assertions.assertInstanceOf(CSVException.class, (Object)exception.getCause());
            Assertions.assertTrue((boolean)exception.getMessage().contains("Invalid character between encapsulated token and delimiter at line: 2, position: 94"), exception::getMessage);
        }
    }

    @Test
    public void testTrailingDelimiter() throws Exception {
        StringReader in = new StringReader("a,a,a,\n\"1\",\"2\",\"3\",\nx,y,z,");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[]{"X", "Y", "Z"}).withSkipHeaderRecord().withTrailingDelimiter().parse((Reader)in);){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            Assertions.assertEquals((Object)"1", (Object)record.get("X"));
            Assertions.assertEquals((Object)"2", (Object)record.get("Y"));
            Assertions.assertEquals((Object)"3", (Object)record.get("Z"));
            Assertions.assertEquals((int)3, (int)record.size());
        }
    }

    @Test
    public void testTrim() throws Exception {
        StringReader in = new StringReader("a,a,a\n\" 1 \",\" 2 \",\" 3 \"\nx,y,z");
        try (CSVParser parser = CSVFormat.DEFAULT.withHeader(new String[]{"X", "Y", "Z"}).withSkipHeaderRecord().withTrim().parse((Reader)in);){
            Iterator records = parser.iterator();
            CSVRecord record = (CSVRecord)records.next();
            Assertions.assertEquals((Object)"1", (Object)record.get("X"));
            Assertions.assertEquals((Object)"2", (Object)record.get("Y"));
            Assertions.assertEquals((Object)"3", (Object)record.get("Z"));
            Assertions.assertEquals((int)3, (int)record.size());
        }
    }

    private void validateLineNumbers(String lineSeparator) throws IOException {
        try (CSVParser parser = CSVParser.parse((String)("a" + lineSeparator + "b" + lineSeparator + "c"), (CSVFormat)CSVFormat.DEFAULT.withRecordSeparator(lineSeparator));){
            Assertions.assertEquals((long)0L, (long)parser.getCurrentLineNumber());
            Assertions.assertNotNull((Object)parser.nextRecord());
            Assertions.assertEquals((long)1L, (long)parser.getCurrentLineNumber());
            Assertions.assertNotNull((Object)parser.nextRecord());
            Assertions.assertEquals((long)2L, (long)parser.getCurrentLineNumber());
            Assertions.assertNotNull((Object)parser.nextRecord());
            Assertions.assertEquals((long)3L, (long)parser.getCurrentLineNumber());
            Assertions.assertNull((Object)parser.nextRecord());
            Assertions.assertEquals((long)3L, (long)parser.getCurrentLineNumber());
        }
    }

    private void validateRecordNumbers(String lineSeparator) throws IOException {
        try (CSVParser parser = CSVParser.parse((String)("a" + lineSeparator + "b" + lineSeparator + "c"), (CSVFormat)CSVFormat.DEFAULT.withRecordSeparator(lineSeparator));){
            Assertions.assertEquals((long)0L, (long)parser.getRecordNumber());
            CSVRecord record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)1L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)1L, (long)parser.getRecordNumber());
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)2L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)2L, (long)parser.getRecordNumber());
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)3L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)3L, (long)parser.getRecordNumber());
            record = parser.nextRecord();
            Assertions.assertNull((Object)record);
            Assertions.assertEquals((long)3L, (long)parser.getRecordNumber());
        }
    }

    private void validateRecordPosition(String lineSeparator) throws IOException {
        long positionRecord3;
        CSVRecord record;
        String nl = lineSeparator;
        String code = "a,b,c" + lineSeparator + "1,2,3" + lineSeparator + "'A" + nl + "A','B" + nl + "B',CC" + lineSeparator + "\u00c4,\u00d6,\u00dc" + lineSeparator + "EOF,EOF,EOF";
        CSVFormat format = CSVFormat.newFormat((char)',').withQuote('\'').withRecordSeparator(lineSeparator);
        try (CSVParser parser = CSVParser.parse((String)code, (CSVFormat)format);){
            Assertions.assertEquals((long)0L, (long)parser.getRecordNumber());
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)1L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)code.indexOf(97), (long)record.getCharacterPosition());
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)2L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)code.indexOf(49), (long)record.getCharacterPosition());
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            positionRecord3 = record.getCharacterPosition();
            Assertions.assertEquals((long)3L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)code.indexOf("'A"), (long)record.getCharacterPosition());
            Assertions.assertEquals((Object)("A" + lineSeparator + "A"), (Object)record.get(0));
            Assertions.assertEquals((Object)("B" + lineSeparator + "B"), (Object)record.get(1));
            Assertions.assertEquals((Object)"CC", (Object)record.get(2));
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)4L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)code.indexOf(196), (long)record.getCharacterPosition());
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)5L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)code.indexOf("EOF"), (long)record.getCharacterPosition());
        }
        parser = new CSVParser((Reader)new StringReader(code.substring((int)positionRecord3)), format, positionRecord3, 3L);
        try {
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)3L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)code.indexOf("'A"), (long)record.getCharacterPosition());
            Assertions.assertEquals((Object)("A" + lineSeparator + "A"), (Object)record.get(0));
            Assertions.assertEquals((Object)("B" + lineSeparator + "B"), (Object)record.get(1));
            Assertions.assertEquals((Object)"CC", (Object)record.get(2));
            record = parser.nextRecord();
            Assertions.assertNotNull((Object)record);
            Assertions.assertEquals((long)4L, (long)record.getRecordNumber());
            Assertions.assertEquals((long)code.indexOf(196), (long)record.getCharacterPosition());
            Assertions.assertEquals((Object)"\u00c4", (Object)record.get(0));
        }
        finally {
            parser.close();
        }
    }
}

