/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.ByteStreams;
import com.google.common.io.CountingOutputStream;
import java.io.DataInput;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import javax.crypto.Cipher;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.ParameterizedClass;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.hints.ChecksumMismatchException;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.io.util.FileInputStreamPlus;
import org.apache.cassandra.schema.CompressionParams;
import org.apache.cassandra.security.EncryptionContext;
import org.apache.cassandra.serializers.MarshalException;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Hex;
import org.apache.cassandra.utils.JsonUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class HintsDescriptor {
    private static final Logger logger = LoggerFactory.getLogger(HintsDescriptor.class);
    static final int VERSION_30 = 1;
    static final int VERSION_40 = 2;
    static final int VERSION_50 = 3;
    static final int CURRENT_VERSION = DatabaseDescriptor.getStorageCompatibilityMode().isBefore(5) ? 2 : 3;
    static final String COMPRESSION = "compression";
    static final String ENCRYPTION = "encryption";
    static final Pattern pattern = Pattern.compile("^[a-fA-F0-9]{8}\\-[a-fA-F0-9]{4}\\-[a-fA-F0-9]{4}\\-[a-fA-F0-9]{4}\\-[a-fA-F0-9]{12}\\-(\\d+)\\-(\\d+)\\.hints$");
    final UUID hostId;
    final int version;
    final long timestamp;
    final String hintsFileName;
    final String crc32FileName;
    final ImmutableMap<String, Object> parameters;
    final ParameterizedClass compressionConfig;
    private final Cipher cipher;
    private final ICompressor compressor;
    private volatile transient long hintsFileSize = -1L;

    HintsDescriptor(UUID hostId, int version, long timestamp, ImmutableMap<String, Object> parameters) {
        this.hostId = hostId;
        this.version = version;
        this.timestamp = timestamp;
        this.hintsFileName = hostId + "-" + timestamp + "-" + version + ".hints";
        this.crc32FileName = hostId + "-" + timestamp + "-" + version + ".crc32";
        this.compressionConfig = HintsDescriptor.createCompressionConfig(parameters);
        EncryptionData encryption = HintsDescriptor.createEncryption(parameters);
        if (encryption == null) {
            this.cipher = null;
            this.compressor = null;
        } else {
            if (this.compressionConfig != null) {
                throw new IllegalStateException("a hints file cannot be configured for both compression and encryption");
            }
            this.cipher = encryption.cipher;
            this.compressor = encryption.compressor;
            parameters = encryption.params;
        }
        this.parameters = parameters;
    }

    HintsDescriptor(UUID hostId, long timestamp, ImmutableMap<String, Object> parameters) {
        this(hostId, CURRENT_VERSION, timestamp, parameters);
    }

    HintsDescriptor(UUID hostId, long timestamp) {
        this(hostId, CURRENT_VERSION, timestamp, (ImmutableMap<String, Object>)ImmutableMap.of());
    }

    static ParameterizedClass createCompressionConfig(Map<String, Object> params) {
        if (params.containsKey(COMPRESSION)) {
            Map compressorConfig = (Map)params.get(COMPRESSION);
            return new ParameterizedClass((String)compressorConfig.get("class_name"), (Map)compressorConfig.get("parameters"));
        }
        return null;
    }

    static EncryptionData createEncryption(ImmutableMap<String, Object> params) {
        if (params.containsKey((Object)ENCRYPTION)) {
            Map encryptionConfig = (Map)params.get((Object)ENCRYPTION);
            EncryptionContext encryptionContext = EncryptionContext.createFromMap(encryptionConfig, DatabaseDescriptor.getEncryptionContext());
            try {
                Cipher cipher;
                if (encryptionConfig.containsKey("encIV")) {
                    cipher = encryptionContext.getDecryptor();
                } else {
                    cipher = encryptionContext.getEncryptor();
                    ImmutableMap encParams = ImmutableMap.builder().putAll(encryptionContext.toHeaderParameters()).put((Object)"encIV", (Object)Hex.bytesToHex(cipher.getIV())).build();
                    HashMap<String, Object> map = new HashMap<String, Object>((Map<String, Object>)params);
                    map.put(ENCRYPTION, encParams);
                    params = ImmutableMap.builder().putAll(map).build();
                }
                return new EncryptionData(cipher, encryptionContext.getCompressor(), params);
            }
            catch (IOException ioe) {
                logger.warn("failed to create encyption context for hints file. ignoring encryption for hints.", (Throwable)ioe);
                return null;
            }
        }
        return null;
    }

    String fileName() {
        return this.hintsFileName;
    }

    String checksumFileName() {
        return this.crc32FileName;
    }

    File file(File hintsDirectory) {
        return new File(hintsDirectory, this.fileName());
    }

    File checksumFile(File hintsDirectory) {
        return new File(hintsDirectory, this.checksumFileName());
    }

    long hintsFileSize(File hintsDirectory) {
        long size = this.hintsFileSize;
        if (size == -1L) {
            this.hintsFileSize = size = this.file(hintsDirectory).length();
        }
        return size;
    }

    void hintsFileSize(long value) {
        this.hintsFileSize = value;
    }

    int messagingVersion() {
        return HintsDescriptor.messagingVersion(this.version);
    }

    static int messagingVersion(int hintsVersion) {
        switch (hintsVersion) {
            case 1: {
                return 10;
            }
            case 2: {
                return 12;
            }
            case 3: {
                return 13;
            }
        }
        throw new AssertionError();
    }

    static boolean isHintFileName(Path path) {
        return pattern.matcher(path.getFileName().toString()).matches();
    }

    static Optional<HintsDescriptor> readFromFileQuietly(Path path) {
        Optional<HintsDescriptor> optional;
        FileInputStreamPlus raf = new FileInputStreamPlus(path);
        try {
            optional = Optional.of(HintsDescriptor.deserialize(raf));
        }
        catch (Throwable throwable) {
            try {
                try {
                    raf.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (ChecksumMismatchException e) {
                throw new FSReadError((Throwable)e, path);
            }
            catch (IOException e) {
                HintsDescriptor.handleDescriptorIOE(e, path);
                return Optional.empty();
            }
        }
        raf.close();
        return optional;
    }

    @VisibleForTesting
    static void handleDescriptorIOE(IOException e, Path path) {
        try {
            if (Files.size(path) > 0L) {
                String newFileName = path.getFileName().toString().replace(".hints", ".corrupt.hints");
                Path target = path.getParent().resolve(newFileName);
                logger.error("Failed to deserialize hints descriptor {} - saving file as {}", new Object[]{path.toString(), target, e});
                Files.move(path, target, new CopyOption[0]);
            } else {
                logger.warn("Found empty hints file {} on startup, removing", (Object)path.toString());
                Files.delete(path);
            }
        }
        catch (IOException ex) {
            logger.error("Error handling corrupt hints file {}", (Object)path.toString(), (Object)ex);
        }
    }

    static HintsDescriptor readFromFile(File path) {
        HintsDescriptor hintsDescriptor;
        FileInputStreamPlus raf = new FileInputStreamPlus(path);
        try {
            hintsDescriptor = HintsDescriptor.deserialize(raf);
        }
        catch (Throwable throwable) {
            try {
                try {
                    raf.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new FSReadError((Throwable)e, path);
            }
        }
        raf.close();
        return hintsDescriptor;
    }

    public boolean isCompressed() {
        return this.compressionConfig != null;
    }

    public boolean isEncrypted() {
        return this.cipher != null;
    }

    public ICompressor createCompressor() {
        if (this.isCompressed()) {
            return CompressionParams.createCompressor(this.compressionConfig);
        }
        if (this.isEncrypted()) {
            return this.compressor;
        }
        return null;
    }

    public Cipher getCipher() {
        return this.isEncrypted() ? this.cipher : null;
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("hostId", (Object)this.hostId).add("version", this.version).add("timestamp", this.timestamp).add("parameters", this.parameters).toString();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof HintsDescriptor)) {
            return false;
        }
        HintsDescriptor hd = (HintsDescriptor)o;
        return Objects.equal((Object)this.hostId, (Object)hd.hostId) && Objects.equal((Object)this.version, (Object)hd.version) && Objects.equal((Object)this.timestamp, (Object)hd.timestamp) && Objects.equal(this.parameters, hd.parameters);
    }

    public int hashCode() {
        return Objects.hashCode((Object[])new Object[]{this.hostId, this.version, this.timestamp, this.parameters});
    }

    void serialize(DataOutputPlus out) throws IOException {
        CRC32 crc = new CRC32();
        out.writeInt(this.version);
        FBUtilities.updateChecksumInt(crc, this.version);
        out.writeLong(this.timestamp);
        HintsDescriptor.updateChecksumLong(crc, this.timestamp);
        out.writeLong(this.hostId.getMostSignificantBits());
        HintsDescriptor.updateChecksumLong(crc, this.hostId.getMostSignificantBits());
        out.writeLong(this.hostId.getLeastSignificantBits());
        HintsDescriptor.updateChecksumLong(crc, this.hostId.getLeastSignificantBits());
        byte[] paramsBytes = JsonUtils.writeAsJsonBytes(this.parameters);
        out.writeInt(paramsBytes.length);
        FBUtilities.updateChecksumInt(crc, paramsBytes.length);
        out.writeInt((int)crc.getValue());
        out.write(paramsBytes);
        crc.update(paramsBytes, 0, paramsBytes.length);
        out.writeInt((int)crc.getValue());
    }

    int serializedSize() {
        int serializedParamsLength;
        int size = TypeSizes.sizeof(this.version);
        size += TypeSizes.sizeof(this.timestamp);
        size += TypeSizes.sizeof(this.hostId.getMostSignificantBits());
        size += TypeSizes.sizeof(this.hostId.getLeastSignificantBits());
        try (CountingOutputStream out = new CountingOutputStream(ByteStreams.nullOutputStream());){
            JsonUtils.JSON_OBJECT_MAPPER.writeValue((OutputStream)out, this.parameters);
            serializedParamsLength = (int)out.getCount();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        size += TypeSizes.sizeof(serializedParamsLength);
        size += 4;
        size += serializedParamsLength;
        return size += 4;
    }

    static HintsDescriptor deserialize(DataInput in) throws IOException {
        CRC32 crc = new CRC32();
        int version = in.readInt();
        FBUtilities.updateChecksumInt(crc, version);
        long timestamp = in.readLong();
        HintsDescriptor.updateChecksumLong(crc, timestamp);
        long msb = in.readLong();
        HintsDescriptor.updateChecksumLong(crc, msb);
        long lsb = in.readLong();
        HintsDescriptor.updateChecksumLong(crc, lsb);
        UUID hostId = new UUID(msb, lsb);
        int paramsLength = in.readInt();
        FBUtilities.updateChecksumInt(crc, paramsLength);
        HintsDescriptor.validateCRC(in.readInt(), (int)crc.getValue());
        byte[] paramsBytes = new byte[paramsLength];
        in.readFully(paramsBytes, 0, paramsLength);
        crc.update(paramsBytes, 0, paramsLength);
        HintsDescriptor.validateCRC(in.readInt(), (int)crc.getValue());
        return new HintsDescriptor(hostId, version, timestamp, HintsDescriptor.decodeJSONBytes(paramsBytes));
    }

    private static ImmutableMap<String, Object> decodeJSONBytes(byte[] bytes) {
        try {
            return ImmutableMap.copyOf(JsonUtils.fromJsonMap(bytes));
        }
        catch (MarshalException e) {
            throw new MarshalException("Corrupt HintsDescriptor serialization, problem: " + e.getMessage(), e);
        }
    }

    private static void updateChecksumLong(CRC32 crc, long value) {
        FBUtilities.updateChecksumInt(crc, (int)(value & 0xFFFFFFFFL));
        FBUtilities.updateChecksumInt(crc, (int)(value >>> 32));
    }

    private static void validateCRC(int expected, int actual) throws IOException {
        if (expected != actual) {
            throw new ChecksumMismatchException("Hints Descriptor CRC Mismatch");
        }
    }

    private static final class EncryptionData {
        final Cipher cipher;
        final ICompressor compressor;
        final ImmutableMap<String, Object> params;

        private EncryptionData(Cipher cipher, ICompressor compressor, ImmutableMap<String, Object> params) {
            this.cipher = cipher;
            this.compressor = compressor;
            this.params = params;
        }
    }
}

