/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.util.zip.CRC32;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.hints.CompressedHintsWriter;
import org.apache.cassandra.hints.EncryptedHintsWriter;
import org.apache.cassandra.hints.Hint;
import org.apache.cassandra.hints.HintsDescriptor;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.util.DataOutputBuffer;
import org.apache.cassandra.io.util.DataOutputBufferFixed;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.NativeLibrary;
import org.apache.cassandra.utils.SyncUtil;
import org.apache.cassandra.utils.Throwables;

class HintsWriter
implements AutoCloseable {
    static final int PAGE_SIZE = 4096;
    private final File directory;
    private final HintsDescriptor descriptor;
    private final File file;
    protected final FileChannel channel;
    private final int fd;
    protected final CRC32 globalCRC;
    private volatile long lastSyncPosition = 0L;

    protected HintsWriter(File directory, HintsDescriptor descriptor, File file, FileChannel channel, int fd, CRC32 globalCRC) {
        this.directory = directory;
        this.descriptor = descriptor;
        this.file = file;
        this.channel = channel;
        this.fd = fd;
        this.globalCRC = globalCRC;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    static HintsWriter create(File directory, HintsDescriptor descriptor) throws IOException {
        File file = descriptor.file(directory);
        FileChannel channel = FileChannel.open(file.toPath(), StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW);
        int fd = NativeLibrary.getfd(channel);
        CRC32 crc = new CRC32();
        try (DataOutputBuffer dob = (DataOutputBuffer)DataOutputBuffer.scratchBuffer.get();){
            descriptor.serialize(dob);
            ByteBuffer descriptorBytes = dob.unsafeGetBufferAndFlip();
            FBUtilities.updateChecksum(crc, descriptorBytes);
            channel.write(descriptorBytes);
            descriptor.hintsFileSize(channel.position());
            if (descriptor.isEncrypted()) {
                EncryptedHintsWriter encryptedHintsWriter = new EncryptedHintsWriter(directory, descriptor, file, channel, fd, crc);
                return encryptedHintsWriter;
            }
            if (descriptor.isCompressed()) {
                CompressedHintsWriter compressedHintsWriter = new CompressedHintsWriter(directory, descriptor, file, channel, fd, crc);
                return compressedHintsWriter;
            }
            HintsWriter hintsWriter = new HintsWriter(directory, descriptor, file, channel, fd, crc);
            return hintsWriter;
        }
        catch (Throwable e) {
            channel.close();
            throw e;
        }
    }

    HintsDescriptor descriptor() {
        return this.descriptor;
    }

    private void writeChecksum() {
        File checksumFile = this.descriptor.checksumFile(this.directory);
        try (OutputStream out = Files.newOutputStream(checksumFile.toPath(), new OpenOption[0]);){
            out.write(Integer.toHexString((int)this.globalCRC.getValue()).getBytes(StandardCharsets.UTF_8));
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, checksumFile);
        }
    }

    @Override
    public void close() {
        Throwables.DiscreteAction[] discreteActionArray = new Throwables.DiscreteAction[2];
        discreteActionArray[0] = this::doFsync;
        discreteActionArray[1] = this.channel::close;
        Throwables.perform(this.file, Throwables.FileOpType.WRITE, discreteActionArray);
        this.writeChecksum();
    }

    public void fsync() {
        Throwables.perform(this.file, Throwables.FileOpType.WRITE, this::doFsync);
    }

    private void doFsync() throws IOException {
        SyncUtil.force(this.channel, true);
        this.lastSyncPosition = this.channel.position();
    }

    Session newSession(ByteBuffer buffer) {
        try {
            return new Session(buffer, this.channel.size());
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.file);
        }
    }

    @VisibleForTesting
    File getFile() {
        return this.file;
    }

    protected void writeBuffer(ByteBuffer bb) throws IOException {
        FBUtilities.updateChecksum(this.globalCRC, bb);
        this.channel.write(bb);
    }

    final class Session
    implements AutoCloseable {
        private final ByteBuffer buffer;
        private final long initialSize;
        private long bytesWritten;

        Session(ByteBuffer buffer, long initialSize) {
            buffer.clear();
            this.bytesWritten = 0L;
            this.buffer = buffer;
            this.initialSize = initialSize;
        }

        @VisibleForTesting
        long getBytesWritten() {
            return this.bytesWritten;
        }

        long position() {
            return this.initialSize + this.bytesWritten;
        }

        void append(ByteBuffer hint) throws IOException {
            this.bytesWritten += (long)hint.remaining();
            if (hint.remaining() > this.buffer.remaining()) {
                this.buffer.flip();
                HintsWriter.this.writeBuffer(this.buffer);
                this.buffer.clear();
            }
            if (hint.remaining() <= this.buffer.remaining()) {
                this.buffer.put(hint);
            } else {
                HintsWriter.this.writeBuffer(hint);
            }
        }

        void append(Hint hint) throws IOException {
            int hintSize = (int)Hint.serializer.serializedSize(hint, HintsWriter.this.descriptor.messagingVersion());
            int totalSize = hintSize + 12;
            if (totalSize > this.buffer.remaining()) {
                this.flushBuffer();
            }
            ByteBuffer hintBuffer = totalSize <= this.buffer.remaining() ? this.buffer : ByteBuffer.allocate(totalSize);
            CRC32 crc = new CRC32();
            try (DataOutputBufferFixed out = new DataOutputBufferFixed(hintBuffer);){
                out.writeInt(hintSize);
                FBUtilities.updateChecksumInt(crc, hintSize);
                out.writeInt((int)crc.getValue());
                long startPosition = out.position();
                Hint.serializer.serialize(hint, (DataOutputPlus)out, HintsWriter.this.descriptor.messagingVersion());
                long actualSize = out.position() - startPosition;
                Preconditions.checkState((actualSize == (long)hintSize ? 1 : 0) != 0, (Object)"Serialized hint size doesn't match calculated hint size");
                FBUtilities.updateChecksum(crc, hintBuffer, hintBuffer.position() - hintSize, hintSize);
                out.writeInt((int)crc.getValue());
            }
            if (hintBuffer == this.buffer) {
                this.bytesWritten += (long)totalSize;
            } else {
                this.append(hintBuffer.flip());
            }
        }

        @Override
        public void close() throws IOException {
            this.flushBuffer();
            this.maybeFsync();
            this.maybeSkipCache();
            HintsWriter.this.descriptor.hintsFileSize(this.position());
        }

        private void flushBuffer() throws IOException {
            this.buffer.flip();
            if (this.buffer.remaining() > 0) {
                HintsWriter.this.writeBuffer(this.buffer);
            }
            this.buffer.clear();
        }

        private void maybeFsync() {
            if (this.position() >= HintsWriter.this.lastSyncPosition + (long)DatabaseDescriptor.getTrickleFsyncIntervalInKiB() * 1024L) {
                HintsWriter.this.fsync();
            }
        }

        private void maybeSkipCache() {
            long position = this.position();
            if (position >= (long)DatabaseDescriptor.getTrickleFsyncIntervalInKiB() * 1024L) {
                NativeLibrary.trySkipCache(HintsWriter.this.fd, 0L, position - position % 4096L, HintsWriter.this.file.path());
            }
        }
    }
}

