/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.batchlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.RateLimiter;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.cassandra.batchlog.Batch;
import org.apache.cassandra.batchlog.BatchlogManagerMBean;
import org.apache.cassandra.concurrent.DebuggableScheduledThreadPoolExecutor;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.WriteType;
import org.apache.cassandra.db.marshal.BytesType;
import org.apache.cassandra.db.marshal.UUIDType;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.WriteFailureException;
import org.apache.cassandra.exceptions.WriteTimeoutException;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.hints.Hint;
import org.apache.cassandra.hints.HintsService;
import org.apache.cassandra.io.util.DataInputBuffer;
import org.apache.cassandra.io.util.DataOutputBuffer;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.locator.EndpointsForToken;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.locator.Replica;
import org.apache.cassandra.locator.ReplicaLayout;
import org.apache.cassandra.locator.ReplicaPlan;
import org.apache.cassandra.locator.Replicas;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessageFlag;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.service.WriteResponseHandler;
import org.apache.cassandra.utils.ExecutorUtils;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.MBeanWrapper;
import org.apache.cassandra.utils.UUIDGen;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BatchlogManager
implements BatchlogManagerMBean {
    public static final String MBEAN_NAME = "org.apache.cassandra.db:type=BatchlogManager";
    private static final long REPLAY_INTERVAL = 10000L;
    static final int DEFAULT_PAGE_SIZE = 128;
    private static final Logger logger = LoggerFactory.getLogger(BatchlogManager.class);
    public static final BatchlogManager instance = new BatchlogManager();
    public static final long BATCHLOG_REPLAY_TIMEOUT = Long.getLong("cassandra.batchlog.replay_timeout_in_ms", DatabaseDescriptor.getWriteRpcTimeout(TimeUnit.MILLISECONDS) * 2L);
    private volatile long totalBatchesReplayed = 0L;
    private volatile UUID lastReplayedUuid = UUIDGen.minTimeUUID(0L);
    private final ScheduledExecutorService batchlogTasks;
    private final RateLimiter rateLimiter = RateLimiter.create((double)Double.MAX_VALUE);

    public BatchlogManager() {
        DebuggableScheduledThreadPoolExecutor executor = new DebuggableScheduledThreadPoolExecutor("BatchlogTasks");
        executor.setExecuteExistingDelayedTasksAfterShutdownPolicy(false);
        this.batchlogTasks = executor;
    }

    public void start() {
        MBeanWrapper.instance.registerMBean((Object)this, MBEAN_NAME);
        this.batchlogTasks.scheduleWithFixedDelay(this::replayFailedBatches, StorageService.RING_DELAY, 10000L, TimeUnit.MILLISECONDS);
    }

    public void shutdownAndWait(long timeout, TimeUnit unit) throws InterruptedException, TimeoutException {
        ExecutorUtils.shutdownAndWait(timeout, unit, this.batchlogTasks);
    }

    public static void remove(UUID id) {
        new Mutation(PartitionUpdate.fullPartitionDelete(SystemKeyspace.Batches, UUIDType.instance.decompose(id), FBUtilities.timestampMicros(), FBUtilities.nowInSeconds())).apply();
    }

    public static void store(Batch batch) {
        BatchlogManager.store(batch, true);
    }

    public static void store(Batch batch, boolean durableWrites) {
        ArrayList<ByteBuffer> mutations = new ArrayList<ByteBuffer>(batch.encodedMutations.size() + batch.decodedMutations.size());
        mutations.addAll(batch.encodedMutations);
        for (Mutation mutation : batch.decodedMutations) {
            try {
                DataOutputBuffer buffer = new DataOutputBuffer();
                Throwable throwable = null;
                try {
                    Mutation.serializer.serialize(mutation, (DataOutputPlus)buffer, 12);
                    mutations.add(buffer.buffer());
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (buffer == null) continue;
                    if (throwable != null) {
                        try {
                            buffer.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    buffer.close();
                }
            }
            catch (IOException e) {
                throw new AssertionError((Object)e);
            }
        }
        PartitionUpdate.SimpleBuilder builder = PartitionUpdate.simpleBuilder(SystemKeyspace.Batches, batch.id);
        builder.row(new Object[0]).timestamp(batch.creationTime).add("version", 12).appendAll("mutations", mutations);
        builder.buildAsMutation().apply(durableWrites);
    }

    @Override
    @VisibleForTesting
    public int countAllBatches() {
        String query = String.format("SELECT count(*) FROM %s.%s", "system", "batches");
        UntypedResultSet results = QueryProcessor.executeInternal(query, new Object[0]);
        if (results == null || results.isEmpty()) {
            return 0;
        }
        return (int)results.one().getLong("count");
    }

    @Override
    public long getTotalBatchesReplayed() {
        return this.totalBatchesReplayed;
    }

    @Override
    public void forceBatchlogReplay() throws Exception {
        this.startBatchlogReplay().get();
    }

    public Future<?> startBatchlogReplay() {
        return this.batchlogTasks.submit(this::replayFailedBatches);
    }

    void performInitialReplay() throws InterruptedException, ExecutionException {
        this.batchlogTasks.submit(this::replayFailedBatches).get();
    }

    private void replayFailedBatches() {
        logger.trace("Started replayFailedBatches");
        int endpointsCount = StorageService.instance.getTokenMetadata().getSizeOfAllEndpoints();
        if (endpointsCount <= 0) {
            logger.trace("Replay cancelled as there are no peers in the ring.");
            return;
        }
        this.setRate(DatabaseDescriptor.getBatchlogReplayThrottleInKB());
        UUID limitUuid = UUIDGen.maxTimeUUID(System.currentTimeMillis() - BatchlogManager.getBatchlogTimeout());
        ColumnFamilyStore store = Keyspace.open("system").getColumnFamilyStore("batches");
        int pageSize = BatchlogManager.calculatePageSize(store);
        String query = String.format("SELECT id, mutations, version FROM %s.%s WHERE token(id) > token(?) AND token(id) <= token(?)", "system", "batches");
        UntypedResultSet batches = QueryProcessor.executeInternalWithPaging(query, pageSize, this.lastReplayedUuid, limitUuid);
        this.processBatchlogEntries(batches, pageSize, this.rateLimiter);
        this.lastReplayedUuid = limitUuid;
        logger.trace("Finished replayFailedBatches");
    }

    public void setRate(int throttleInKB) {
        int endpointsCount = StorageService.instance.getTokenMetadata().getSizeOfAllEndpoints();
        if (endpointsCount > 0) {
            double throughput;
            int endpointThrottleInKB = throttleInKB / endpointsCount;
            double d = throughput = endpointThrottleInKB == 0 ? Double.MAX_VALUE : (double)endpointThrottleInKB * 1024.0;
            if (this.rateLimiter.getRate() != throughput) {
                logger.debug("Updating batchlog replay throttle to {} KB/s, {} KB/s per endpoint", (Object)throttleInKB, (Object)endpointThrottleInKB);
                this.rateLimiter.setRate(throughput);
            }
        }
    }

    static int calculatePageSize(ColumnFamilyStore store) {
        double averageRowSize = store.getMeanPartitionSize();
        if (averageRowSize <= 0.0) {
            return 128;
        }
        return (int)Math.max(1.0, Math.min(128.0, 4194304.0 / averageRowSize));
    }

    private void processBatchlogEntries(UntypedResultSet batches, int pageSize, RateLimiter rateLimiter) {
        int positionInPage = 0;
        ArrayList<ReplayingBatch> unfinishedBatches = new ArrayList<ReplayingBatch>(pageSize);
        HashSet<InetAddressAndPort> hintedNodes = new HashSet<InetAddressAndPort>();
        HashSet<UUID> replayedBatches = new HashSet<UUID>();
        IOException caughtException = null;
        int skipped = 0;
        for (UntypedResultSet.Row row : batches) {
            UUID id = row.getUUID("id");
            int version = row.getInt("version");
            try {
                ReplayingBatch batch = new ReplayingBatch(id, version, row.getList("mutations", BytesType.instance));
                if (batch.replay(rateLimiter, hintedNodes) > 0) {
                    unfinishedBatches.add(batch);
                } else {
                    BatchlogManager.remove(id);
                    ++this.totalBatchesReplayed;
                }
            }
            catch (IOException e) {
                logger.warn("Skipped batch replay of {} due to {}", (Object)id, (Object)e.getMessage());
                caughtException = e;
                BatchlogManager.remove(id);
                ++skipped;
            }
            if (++positionInPage != pageSize) continue;
            this.finishAndClearBatches(unfinishedBatches, hintedNodes, replayedBatches);
            positionInPage = 0;
        }
        this.finishAndClearBatches(unfinishedBatches, hintedNodes, replayedBatches);
        if (caughtException != null) {
            logger.warn(String.format("Encountered %d unexpected exceptions while sending out batches", skipped), caughtException);
        }
        HintsService.instance.flushAndFsyncBlockingly(Iterables.transform(hintedNodes, StorageService.instance::getHostIdForEndpoint));
        replayedBatches.forEach(BatchlogManager::remove);
    }

    private void finishAndClearBatches(ArrayList<ReplayingBatch> batches, Set<InetAddressAndPort> hintedNodes, Set<UUID> replayedBatches) {
        for (ReplayingBatch batch : batches) {
            batch.finish(hintedNodes);
            replayedBatches.add(batch.id);
        }
        this.totalBatchesReplayed += (long)batches.size();
        batches.clear();
    }

    public static long getBatchlogTimeout() {
        return BATCHLOG_REPLAY_TIMEOUT;
    }

    private static class ReplayingBatch {
        private final UUID id;
        private final long writtenAt;
        private final List<Mutation> mutations;
        private final int replayedBytes;
        private List<ReplayWriteResponseHandler<Mutation>> replayHandlers;

        ReplayingBatch(UUID id, int version, List<ByteBuffer> serializedMutations) throws IOException {
            this.id = id;
            this.writtenAt = UUIDGen.unixTimestamp(id);
            this.mutations = new ArrayList<Mutation>(serializedMutations.size());
            this.replayedBytes = this.addMutations(version, serializedMutations);
        }

        public int replay(RateLimiter rateLimiter, Set<InetAddressAndPort> hintedNodes) throws IOException {
            logger.trace("Replaying batch {}", (Object)this.id);
            if (this.mutations.isEmpty()) {
                return 0;
            }
            int gcgs = ReplayingBatch.gcgs(this.mutations);
            if (TimeUnit.MILLISECONDS.toSeconds(this.writtenAt) + (long)gcgs <= (long)FBUtilities.nowInSeconds()) {
                return 0;
            }
            this.replayHandlers = ReplayingBatch.sendReplays(this.mutations, this.writtenAt, hintedNodes);
            rateLimiter.acquire(this.replayedBytes);
            return this.replayHandlers.size();
        }

        public void finish(Set<InetAddressAndPort> hintedNodes) {
            for (int i = 0; i < this.replayHandlers.size(); ++i) {
                ReplayWriteResponseHandler<Mutation> handler = this.replayHandlers.get(i);
                try {
                    handler.get();
                    continue;
                }
                catch (WriteFailureException | WriteTimeoutException e) {
                    logger.trace("Failed replaying a batched mutation to a node, will write a hint");
                    logger.trace("Failure was : {}", (Object)e.getMessage());
                    this.writeHintsForUndeliveredEndpoints(i, hintedNodes);
                    return;
                }
            }
        }

        private int addMutations(int version, List<ByteBuffer> serializedMutations) throws IOException {
            int ret = 0;
            for (ByteBuffer serializedMutation : serializedMutations) {
                ret += serializedMutation.remaining();
                DataInputBuffer in = new DataInputBuffer(serializedMutation, true);
                Throwable throwable = null;
                try {
                    this.addMutation(Mutation.serializer.deserialize(in, version));
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (in == null) continue;
                    if (throwable != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    in.close();
                }
            }
            return ret;
        }

        private void addMutation(Mutation mutation) {
            for (TableId tableId : mutation.getTableIds()) {
                if (this.writtenAt > SystemKeyspace.getTruncatedAt(tableId)) continue;
                mutation = mutation.without(tableId);
            }
            if (!mutation.isEmpty()) {
                this.mutations.add(mutation);
            }
        }

        private void writeHintsForUndeliveredEndpoints(int startFrom, Set<InetAddressAndPort> hintedNodes) {
            int gcgs = ReplayingBatch.gcgs(this.mutations);
            if (TimeUnit.MILLISECONDS.toSeconds(this.writtenAt) + (long)gcgs <= (long)FBUtilities.nowInSeconds()) {
                return;
            }
            for (int i = startFrom; i < this.replayHandlers.size(); ++i) {
                ReplayWriteResponseHandler<Mutation> handler = this.replayHandlers.get(i);
                Mutation undeliveredMutation = this.mutations.get(i);
                if (handler == null) continue;
                hintedNodes.addAll(((ReplayWriteResponseHandler)handler).undelivered);
                HintsService.instance.write(Collections2.transform((Collection)((ReplayWriteResponseHandler)handler).undelivered, StorageService.instance::getHostIdForEndpoint), Hint.create(undeliveredMutation, this.writtenAt));
            }
        }

        private static List<ReplayWriteResponseHandler<Mutation>> sendReplays(List<Mutation> mutations, long writtenAt, Set<InetAddressAndPort> hintedNodes) {
            ArrayList<ReplayWriteResponseHandler<Mutation>> handlers = new ArrayList<ReplayWriteResponseHandler<Mutation>>(mutations.size());
            for (Mutation mutation : mutations) {
                ReplayWriteResponseHandler<Mutation> handler = ReplayingBatch.sendSingleReplayMutation(mutation, writtenAt, hintedNodes);
                handlers.add(handler);
            }
            return handlers;
        }

        private static ReplayWriteResponseHandler<Mutation> sendSingleReplayMutation(Mutation mutation, long writtenAt, Set<InetAddressAndPort> hintedNodes) {
            String ks = mutation.getKeyspaceName();
            Keyspace keyspace = Keyspace.open(ks);
            Token tk = mutation.key().getToken();
            ReplicaLayout.ForTokenWrite liveAndDown = ReplicaLayout.forTokenWriteLiveAndDown(keyspace, tk);
            Replicas.temporaryAssertFull(liveAndDown.all());
            Replica selfReplica = ((EndpointsForToken)liveAndDown.all()).selfIfPresent();
            if (selfReplica != null) {
                mutation.apply();
            }
            ReplicaLayout.ForTokenWrite liveRemoteOnly = liveAndDown.filter(r -> FailureDetector.isReplicaAlive.test((Replica)r) && r != selfReplica);
            for (Replica replica : (EndpointsForToken)liveAndDown.all()) {
                if (replica == selfReplica || ((EndpointsForToken)liveRemoteOnly.all()).contains(replica)) continue;
                hintedNodes.add(replica.endpoint());
                HintsService.instance.write(StorageService.instance.getHostIdForEndpoint(replica.endpoint()), Hint.create(mutation, writtenAt));
            }
            ReplicaPlan.ForTokenWrite replicaPlan = new ReplicaPlan.ForTokenWrite(keyspace, liveAndDown.replicationStrategy(), ConsistencyLevel.ONE, (EndpointsForToken)liveRemoteOnly.pending(), (EndpointsForToken)liveRemoteOnly.all(), (EndpointsForToken)liveRemoteOnly.all(), (EndpointsForToken)liveRemoteOnly.all());
            ReplayWriteResponseHandler<Mutation> handler = new ReplayWriteResponseHandler<Mutation>(replicaPlan, System.nanoTime());
            Message<Mutation> message = Message.outWithFlag(Verb.MUTATION_REQ, mutation, MessageFlag.CALL_BACK_ON_FAILURE);
            for (Replica replica : (EndpointsForToken)liveRemoteOnly.all()) {
                MessagingService.instance().sendWriteWithCallback(message, replica, handler, false);
            }
            return handler;
        }

        private static int gcgs(Collection<Mutation> mutations) {
            int gcgs = Integer.MAX_VALUE;
            for (Mutation mutation : mutations) {
                gcgs = Math.min(gcgs, mutation.smallestGCGS());
            }
            return gcgs;
        }

        private static class ReplayWriteResponseHandler<T>
        extends WriteResponseHandler<T> {
            private final Set<InetAddressAndPort> undelivered = Collections.newSetFromMap(new ConcurrentHashMap());

            ReplayWriteResponseHandler(ReplicaPlan.ForTokenWrite replicaPlan, long queryStartNanoTime) {
                super(replicaPlan, null, WriteType.UNLOGGED_BATCH, queryStartNanoTime);
                Iterables.addAll(this.undelivered, ((EndpointsForToken)replicaPlan.contacts()).endpoints());
            }

            @Override
            protected int blockFor() {
                return ((EndpointsForToken)this.replicaPlan.contacts()).size();
            }

            @Override
            public void onResponse(Message<T> m) {
                boolean removed = this.undelivered.remove(m == null ? FBUtilities.getBroadcastAddressAndPort() : m.from());
                assert (removed);
                super.onResponse(m);
            }
        }
    }
}

