/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.transport;

import com.google.common.annotations.VisibleForTesting;
import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.ByteToMessageDecoder;
import io.netty.handler.codec.MessageToMessageDecoder;
import io.netty.handler.codec.MessageToMessageEncoder;
import io.netty.util.Attribute;
import java.io.IOException;
import java.util.EnumSet;
import java.util.List;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.metrics.ClientRequestSizeMetrics;
import org.apache.cassandra.transport.CBUtil;
import org.apache.cassandra.transport.Connection;
import org.apache.cassandra.transport.Message;
import org.apache.cassandra.transport.ProtocolException;
import org.apache.cassandra.transport.ProtocolVersion;
import org.apache.cassandra.transport.frame.FrameBodyTransformer;
import org.apache.cassandra.transport.messages.ErrorMessage;

public class Frame {
    public static final byte PROTOCOL_VERSION_MASK = 127;
    public final Header header;
    public final ByteBuf body;

    private Frame(Header header, ByteBuf body) {
        this.header = header;
        this.body = body;
    }

    public void retain() {
        this.body.retain();
    }

    public boolean release() {
        return this.body.release();
    }

    public static Frame create(Message.Type type, int streamId, ProtocolVersion version, EnumSet<Header.Flag> flags, ByteBuf body) {
        Header header = new Header(version, flags, streamId, type, body.readableBytes());
        return new Frame(header, body);
    }

    public Frame with(ByteBuf newBody) {
        return new Frame(this.header, newBody);
    }

    private static long discard(ByteBuf buffer, long remainingToDiscard) {
        int availableToDiscard = (int)Math.min(remainingToDiscard, (long)buffer.readableBytes());
        buffer.skipBytes(availableToDiscard);
        return remainingToDiscard - (long)availableToDiscard;
    }

    @ChannelHandler.Sharable
    public static class OutboundBodyTransformer
    extends MessageToMessageEncoder<Frame> {
        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void encode(ChannelHandlerContext ctx, Frame frame, List<Object> results) throws IOException {
            Connection connection = (Connection)ctx.channel().attr(Connection.attributeKey).get();
            if (frame.header.type == Message.Type.STARTUP || connection == null) {
                results.add(frame);
                return;
            }
            FrameBodyTransformer transformer = connection.getTransformer();
            if (transformer == null) {
                results.add(frame);
                return;
            }
            try {
                results.add(frame.with(transformer.transformOutbound(frame.body)));
                frame.header.flags.addAll(transformer.getOutboundHeaderFlags());
            }
            finally {
                frame.release();
            }
        }
    }

    @ChannelHandler.Sharable
    public static class InboundBodyTransformer
    extends MessageToMessageDecoder<Frame> {
        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void decode(ChannelHandlerContext ctx, Frame frame, List<Object> results) throws IOException {
            Connection connection = (Connection)ctx.channel().attr(Connection.attributeKey).get();
            if (!frame.header.flags.contains((Object)Header.Flag.COMPRESSED) && !frame.header.flags.contains((Object)Header.Flag.CHECKSUMMED) || connection == null) {
                results.add(frame);
                return;
            }
            FrameBodyTransformer transformer = connection.getTransformer();
            if (transformer == null) {
                results.add(frame);
                return;
            }
            try {
                results.add(frame.with(transformer.transformInbound(frame.body, frame.header.flags)));
            }
            finally {
                frame.release();
            }
        }
    }

    @ChannelHandler.Sharable
    public static class Encoder
    extends MessageToMessageEncoder<Frame> {
        public void encode(ChannelHandlerContext ctx, Frame frame, List<Object> results) throws IOException {
            ByteBuf header = CBUtil.allocator.buffer(9);
            Message.Type type = frame.header.type;
            header.writeByte(type.direction.addToVersion(frame.header.version.asInt()));
            header.writeByte(Header.Flag.serialize(frame.header.flags));
            if (frame.header.version.isGreaterOrEqualTo(ProtocolVersion.V3)) {
                header.writeShort(frame.header.streamId);
            } else {
                header.writeByte(frame.header.streamId);
            }
            header.writeByte(type.opcode);
            header.writeInt(frame.body.readableBytes());
            int messageSize = header.readableBytes() + frame.body.readableBytes();
            ClientRequestSizeMetrics.totalBytesWritten.inc((long)messageSize);
            ClientRequestSizeMetrics.bytesTransmittedPerFrame.update(messageSize);
            results.add(header);
            results.add(frame.body);
        }
    }

    public static class Decoder
    extends ByteToMessageDecoder {
        private static final int MAX_FRAME_LENGTH = DatabaseDescriptor.getNativeTransportMaxFrameSize();
        private boolean discardingTooLongFrame;
        private long tooLongFrameLength;
        private long bytesToDiscard;
        private int tooLongStreamId;
        private final Connection.Factory factory;

        public Decoder(Connection.Factory factory) {
            this.factory = factory;
        }

        @VisibleForTesting
        Frame decodeFrame(ByteBuf buffer) throws Exception {
            Message.Type type;
            if (this.discardingTooLongFrame) {
                this.bytesToDiscard = Frame.discard(buffer, this.bytesToDiscard);
                if (this.bytesToDiscard <= 0L) {
                    this.fail();
                }
                return null;
            }
            int readableBytes = buffer.readableBytes();
            if (readableBytes == 0) {
                return null;
            }
            int idx = buffer.readerIndex();
            byte firstByte = buffer.getByte(idx++);
            Message.Direction direction = Message.Direction.extractFromVersion(firstByte);
            int versionNum = firstByte & 0x7F;
            ProtocolVersion version = ProtocolVersion.decode(versionNum, DatabaseDescriptor.getNativeTransportAllowOlderProtocols());
            if (readableBytes < 9) {
                return null;
            }
            byte flags = buffer.getByte(idx++);
            EnumSet<Header.Flag> decodedFlags = Header.Flag.deserialize(flags);
            if (version.isBeta() && !decodedFlags.contains((Object)Header.Flag.USE_BETA)) {
                throw new ProtocolException(String.format("Beta version of the protocol used (%s), but USE_BETA flag is unset", version), version);
            }
            short streamId = buffer.getShort(idx);
            idx += 2;
            try {
                type = Message.Type.fromOpcode(buffer.getByte(idx++), direction);
            }
            catch (ProtocolException e) {
                throw ErrorMessage.wrap(e, streamId);
            }
            long bodyLength = buffer.getUnsignedInt(idx);
            idx += 4;
            long frameLength = bodyLength + 9L;
            if (frameLength > (long)MAX_FRAME_LENGTH) {
                this.discardingTooLongFrame = true;
                this.tooLongStreamId = streamId;
                this.tooLongFrameLength = frameLength;
                this.bytesToDiscard = Frame.discard(buffer, frameLength);
                if (this.bytesToDiscard <= 0L) {
                    this.fail();
                }
                return null;
            }
            if ((long)buffer.readableBytes() < frameLength) {
                return null;
            }
            ClientRequestSizeMetrics.totalBytesRead.inc(frameLength);
            ClientRequestSizeMetrics.bytesReceivedPerFrame.update(frameLength);
            ByteBuf body = buffer.slice(idx, (int)bodyLength);
            body.retain();
            idx = (int)((long)idx + bodyLength);
            buffer.readerIndex(idx);
            return new Frame(new Header(version, decodedFlags, streamId, type, bodyLength), body);
        }

        protected void decode(ChannelHandlerContext ctx, ByteBuf buffer, List<Object> results) throws Exception {
            Frame frame = this.decodeFrame(buffer);
            if (frame == null) {
                return;
            }
            Attribute attrConn = ctx.channel().attr(Connection.attributeKey);
            Connection connection = (Connection)attrConn.get();
            if (connection == null) {
                connection = this.factory.newConnection(ctx.channel(), frame.header.version);
                attrConn.set((Object)connection);
            } else if (connection.getVersion() != frame.header.version) {
                throw ErrorMessage.wrap(new ProtocolException(String.format("Invalid message version. Got %s but previous messages on this connection had version %s", frame.header.version, connection.getVersion())), frame.header.streamId);
            }
            results.add(frame);
        }

        private void fail() {
            long tooLongFrameLength = this.tooLongFrameLength;
            this.tooLongFrameLength = 0L;
            this.discardingTooLongFrame = false;
            String msg = String.format("Request is too big: length %d exceeds maximum allowed length %d.", tooLongFrameLength, MAX_FRAME_LENGTH);
            throw ErrorMessage.wrap(new InvalidRequestException(msg), this.tooLongStreamId);
        }
    }

    public static class Header {
        public static final int LENGTH = 9;
        public static final int BODY_LENGTH_SIZE = 4;
        public final ProtocolVersion version;
        public final EnumSet<Flag> flags;
        public final int streamId;
        public final Message.Type type;
        public final long bodySizeInBytes;

        private Header(ProtocolVersion version, EnumSet<Flag> flags, int streamId, Message.Type type, long bodySizeInBytes) {
            this.version = version;
            this.flags = flags;
            this.streamId = streamId;
            this.type = type;
            this.bodySizeInBytes = bodySizeInBytes;
        }

        public static enum Flag {
            COMPRESSED,
            TRACING,
            CUSTOM_PAYLOAD,
            WARNING,
            USE_BETA,
            CHECKSUMMED;

            private static final Flag[] ALL_VALUES;

            public static EnumSet<Flag> deserialize(int flags) {
                EnumSet<Flag> set = EnumSet.noneOf(Flag.class);
                for (int n = 0; n < ALL_VALUES.length; ++n) {
                    if ((flags & 1 << n) == 0) continue;
                    set.add(ALL_VALUES[n]);
                }
                return set;
            }

            public static int serialize(EnumSet<Flag> flags) {
                int i = 0;
                for (Flag flag : flags) {
                    i |= 1 << flag.ordinal();
                }
                return i;
            }

            static {
                ALL_VALUES = Flag.values();
            }
        }
    }
}

