/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.transport.frame.checksum;

import com.google.common.collect.ImmutableTable;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.io.IOException;
import java.util.EnumSet;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.transport.CBUtil;
import org.apache.cassandra.transport.Frame;
import org.apache.cassandra.transport.ProtocolException;
import org.apache.cassandra.transport.frame.FrameBodyTransformer;
import org.apache.cassandra.transport.frame.compress.Compressor;
import org.apache.cassandra.transport.frame.compress.LZ4Compressor;
import org.apache.cassandra.transport.frame.compress.SnappyCompressor;
import org.apache.cassandra.utils.ChecksumType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChecksummingTransformer
implements FrameBodyTransformer {
    private static final Logger logger = LoggerFactory.getLogger(ChecksummingTransformer.class);
    private static final EnumSet<Frame.Header.Flag> CHECKSUMS_ONLY = EnumSet.of(Frame.Header.Flag.CHECKSUMMED);
    private static final EnumSet<Frame.Header.Flag> CHECKSUMS_AND_COMPRESSION = EnumSet.of(Frame.Header.Flag.CHECKSUMMED, Frame.Header.Flag.COMPRESSED);
    private static final int CHUNK_HEADER_OVERHEAD = 16;
    private static final ChecksummingTransformer CRC32_NO_COMPRESSION = new ChecksummingTransformer(ChecksumType.CRC32, null);
    private static final ChecksummingTransformer ADLER32_NO_COMPRESSION = new ChecksummingTransformer(ChecksumType.ADLER32, null);
    private static final ImmutableTable<ChecksumType, Compressor, ChecksummingTransformer> transformers;
    private final int blockSize;
    private final Compressor compressor;
    private final ChecksumType checksum;

    public static ChecksummingTransformer getTransformer(ChecksumType checksumType, Compressor compressor) {
        ChecksummingTransformer transformer;
        ChecksummingTransformer checksummingTransformer = compressor == null ? (checksumType == ChecksumType.CRC32 ? CRC32_NO_COMPRESSION : ADLER32_NO_COMPRESSION) : (transformer = (ChecksummingTransformer)transformers.get((Object)checksumType, (Object)compressor));
        if (transformer == null) {
            logger.warn("Invalid compression/checksum options supplied. %s / %s", (Object)checksumType, (Object)compressor.getClass().getName());
            throw new RuntimeException("Invalid compression / checksum options supplied");
        }
        return transformer;
    }

    ChecksummingTransformer(ChecksumType checksumType, Compressor compressor) {
        this(checksumType, DatabaseDescriptor.getNativeTransportFrameBlockSize(), compressor);
    }

    ChecksummingTransformer(ChecksumType checksumType, int blockSize, Compressor compressor) {
        this.checksum = checksumType;
        this.blockSize = blockSize;
        this.compressor = compressor;
    }

    @Override
    public EnumSet<Frame.Header.Flag> getOutboundHeaderFlags() {
        return null == this.compressor ? CHECKSUMS_ONLY : CHECKSUMS_AND_COMPRESSION;
    }

    @Override
    public ByteBuf transformOutbound(ByteBuf inputBuf) {
        int readableBytes;
        int maxTotalCompressedLength = this.maxCompressedLength(inputBuf.readableBytes());
        int expectedChunks = (int)Math.ceil((double)maxTotalCompressedLength / (double)this.blockSize);
        int expectedMaxSerializedLength = 2 + expectedChunks * 16 + maxTotalCompressedLength;
        byte[] retBuf = new byte[expectedMaxSerializedLength];
        ByteBuf ret = Unpooled.wrappedBuffer((byte[])retBuf);
        ret.writerIndex(0);
        ret.readerIndex(0);
        ret.writeShort(0);
        byte[] inBuf = new byte[this.blockSize];
        byte[] outBuf = new byte[this.maxCompressedLength(this.blockSize)];
        byte[] chunkLengths = new byte[8];
        int numCompressedChunks = 0;
        while ((readableBytes = inputBuf.readableBytes()) > 0) {
            int lengthToRead = Math.min(this.blockSize, readableBytes);
            inputBuf.readBytes(inBuf, 0, lengthToRead);
            int uncompressedChunkChecksum = (int)this.checksum.of(inBuf, 0, lengthToRead);
            int compressedSize = this.maybeCompress(inBuf, lengthToRead, outBuf);
            if (compressedSize < lengthToRead) {
                ret.writeInt(compressedSize);
                ret.writeInt(lengthToRead);
                this.putInt(compressedSize, chunkLengths, 0);
            } else {
                ret.writeInt(-lengthToRead);
                this.putInt(-lengthToRead, chunkLengths, 0);
            }
            this.putInt(lengthToRead, chunkLengths, 4);
            int lengthsChecksum = (int)this.checksum.of(chunkLengths, 0, chunkLengths.length);
            ret.writeInt(lengthsChecksum);
            int toWrite = Math.min(compressedSize, lengthToRead);
            if (ret.writableBytes() < 16 + toWrite) {
                byte[] resizedRetBuf = new byte[(retBuf.length + (16 + toWrite)) * 3 / 2];
                System.arraycopy(retBuf, 0, resizedRetBuf, 0, retBuf.length);
                retBuf = resizedRetBuf;
                ByteBuf resizedRetByteBuf = Unpooled.wrappedBuffer((byte[])retBuf);
                resizedRetByteBuf.writerIndex(ret.writerIndex());
                ret = resizedRetByteBuf;
            }
            if (compressedSize < lengthToRead) {
                ret.writeBytes(outBuf, 0, toWrite);
            } else {
                ret.writeBytes(inBuf, 0, toWrite);
            }
            ret.writeInt(uncompressedChunkChecksum);
            ++numCompressedChunks;
        }
        ret.setShort(0, (int)((short)numCompressedChunks));
        return ret;
    }

    @Override
    public ByteBuf transformInbound(ByteBuf inputBuf, EnumSet<Frame.Header.Flag> flags) {
        int numChunks = CBUtil.readUnsignedShort(inputBuf);
        int currentPosition = 0;
        byte[] buf = null;
        byte[] retBuf = new byte[inputBuf.readableBytes()];
        byte[] chunkLengths = new byte[8];
        for (int i = 0; i < numChunks; ++i) {
            int compressedLength = inputBuf.readInt();
            int decompressedLength = compressedLength >= 0 ? inputBuf.readInt() : Math.abs(compressedLength);
            this.putInt(compressedLength, chunkLengths, 0);
            this.putInt(decompressedLength, chunkLengths, 4);
            int lengthsChecksum = inputBuf.readInt();
            int calculatedLengthsChecksum = (int)this.checksum.of(chunkLengths, 0, chunkLengths.length);
            if (lengthsChecksum != calculatedLengthsChecksum) {
                throw new ProtocolException(String.format("Checksum invalid on chunk bytes lengths. Deserialized compressed length: %d decompressed length: %d. %d != %d", compressedLength, decompressedLength, lengthsChecksum, calculatedLengthsChecksum));
            }
            if (currentPosition + decompressedLength > retBuf.length) {
                byte[] resizedBuf = new byte[retBuf.length + decompressedLength * 3 / 2];
                System.arraycopy(retBuf, 0, resizedBuf, 0, retBuf.length);
                retBuf = resizedBuf;
            }
            int toRead = Math.abs(compressedLength);
            if (buf == null || buf.length < toRead) {
                buf = new byte[toRead];
            }
            inputBuf.readBytes(buf, 0, toRead);
            byte[] decompressedChunk = this.maybeDecompress(buf, compressedLength, decompressedLength, flags);
            System.arraycopy(decompressedChunk, 0, retBuf, currentPosition, decompressedLength);
            currentPosition += decompressedLength;
            int expectedDecompressedChecksum = inputBuf.readInt();
            int calculatedDecompressedChecksum = (int)this.checksum.of(decompressedChunk, 0, decompressedLength);
            if (expectedDecompressedChecksum == calculatedDecompressedChecksum) continue;
            throw new ProtocolException("Decompressed checksum for chunk does not match expected checksum");
        }
        ByteBuf ret = Unpooled.wrappedBuffer((byte[])retBuf, (int)0, (int)currentPosition);
        ret.writerIndex(currentPosition);
        return ret;
    }

    private int maxCompressedLength(int uncompressedLength) {
        return null == this.compressor ? uncompressedLength : this.compressor.maxCompressedLength(uncompressedLength);
    }

    private int maybeCompress(byte[] input, int length, byte[] output) {
        if (null == this.compressor) {
            System.arraycopy(input, 0, output, 0, length);
            return length;
        }
        try {
            return this.compressor.compress(input, 0, length, output, 0);
        }
        catch (IOException e) {
            logger.info("IO error during compression of frame body chunk", (Throwable)e);
            throw new ProtocolException("Error compressing frame body chunk");
        }
    }

    private byte[] maybeDecompress(byte[] input, int length, int expectedLength, EnumSet<Frame.Header.Flag> flags) {
        if (null == this.compressor || !flags.contains((Object)Frame.Header.Flag.COMPRESSED) || length < 0) {
            return input;
        }
        try {
            return this.compressor.decompress(input, 0, length, expectedLength);
        }
        catch (IOException e) {
            logger.info("IO error during decompression of frame body chunk", (Throwable)e);
            throw new ProtocolException("Error decompressing frame body chunk");
        }
    }

    private void putInt(int val, byte[] dest, int offset) {
        dest[offset] = (byte)(val >>> 24);
        dest[offset + 1] = (byte)(val >>> 16);
        dest[offset + 2] = (byte)(val >>> 8);
        dest[offset + 3] = (byte)val;
    }

    static {
        ImmutableTable.Builder builder = ImmutableTable.builder();
        builder.put((Object)ChecksumType.CRC32, (Object)LZ4Compressor.INSTANCE, (Object)new ChecksummingTransformer(ChecksumType.CRC32, LZ4Compressor.INSTANCE));
        builder.put((Object)ChecksumType.CRC32, (Object)SnappyCompressor.INSTANCE, (Object)new ChecksummingTransformer(ChecksumType.CRC32, SnappyCompressor.INSTANCE));
        builder.put((Object)ChecksumType.ADLER32, (Object)LZ4Compressor.INSTANCE, (Object)new ChecksummingTransformer(ChecksumType.ADLER32, LZ4Compressor.INSTANCE));
        builder.put((Object)ChecksumType.ADLER32, (Object)SnappyCompressor.INSTANCE, (Object)new ChecksummingTransformer(ChecksumType.ADLER32, SnappyCompressor.INSTANCE));
        transformers = builder.build();
    }
}

