/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.repair;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.hash.Funnel;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Random;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.rows.UnfilteredRowIterator;
import org.apache.cassandra.db.rows.UnfilteredRowIterators;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.repair.RepairJobDesc;
import org.apache.cassandra.repair.messages.ValidationResponse;
import org.apache.cassandra.service.ActiveRepairService;
import org.apache.cassandra.streaming.PreviewKind;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.MerkleTree;
import org.apache.cassandra.utils.MerkleTrees;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Validator
implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(Validator.class);
    public final RepairJobDesc desc;
    public final InetAddressAndPort initiator;
    public final int nowInSec;
    private final boolean evenTreeDistribution;
    public final boolean isIncremental;
    private long validated;
    private MerkleTrees trees;
    private MerkleTree.TreeRange range;
    private MerkleTrees.TreeRangeIterator ranges;
    private DecoratedKey lastKey;
    private final PreviewKind previewKind;

    public Validator(RepairJobDesc desc, InetAddressAndPort initiator, int nowInSec, PreviewKind previewKind) {
        this(desc, initiator, nowInSec, false, false, previewKind);
    }

    public Validator(RepairJobDesc desc, InetAddressAndPort initiator, int nowInSec, boolean isIncremental, PreviewKind previewKind) {
        this(desc, initiator, nowInSec, false, isIncremental, previewKind);
    }

    public Validator(RepairJobDesc desc, InetAddressAndPort initiator, int nowInSec, boolean evenTreeDistribution, boolean isIncremental, PreviewKind previewKind) {
        this.desc = desc;
        this.initiator = initiator;
        this.nowInSec = nowInSec;
        this.isIncremental = isIncremental;
        this.previewKind = previewKind;
        this.validated = 0L;
        this.range = null;
        this.ranges = null;
        this.evenTreeDistribution = evenTreeDistribution;
    }

    public void prepare(ColumnFamilyStore cfs, MerkleTrees tree) {
        this.trees = tree;
        if (!tree.partitioner().preservesOrder() || this.evenTreeDistribution) {
            tree.init();
        } else {
            ArrayList<DecoratedKey> keys = new ArrayList<DecoratedKey>();
            Random random = new Random();
            for (Range<Token> range : tree.ranges()) {
                DecoratedKey dk;
                for (DecoratedKey sample : cfs.keySamples(range)) {
                    assert (range.contains(sample.getToken())) : "Token " + sample.getToken() + " is not within range " + this.desc.ranges;
                    keys.add(sample);
                }
                if (keys.isEmpty()) {
                    tree.init(range);
                    continue;
                }
                int numKeys = keys.size();
                while (tree.split((dk = (DecoratedKey)keys.get(random.nextInt(numKeys))).getToken())) {
                }
                keys.clear();
            }
        }
        logger.debug("Prepared AEService trees of size {} for {}", (Object)this.trees.size(), (Object)this.desc);
        this.ranges = tree.rangeIterator();
    }

    public void add(UnfilteredRowIterator partition) {
        assert (Range.isInRanges(partition.partitionKey().getToken(), this.desc.ranges)) : partition.partitionKey().getToken() + " is not contained in " + this.desc.ranges;
        assert (this.lastKey == null || this.lastKey.compareTo(partition.partitionKey()) < 0) : "partition " + partition.partitionKey() + " received out of order wrt " + this.lastKey;
        this.lastKey = partition.partitionKey();
        if (this.range == null) {
            this.range = (MerkleTree.TreeRange)this.ranges.next();
        }
        if (!this.findCorrectRange(this.lastKey.getToken())) {
            this.ranges = this.trees.rangeIterator();
            this.findCorrectRange(this.lastKey.getToken());
        }
        assert (this.range.contains(this.lastKey.getToken())) : "Token not in MerkleTree: " + this.lastKey.getToken();
        MerkleTree.RowHash rowHash = this.rowHash(partition);
        if (rowHash != null) {
            this.range.addHash(rowHash);
        }
    }

    public boolean findCorrectRange(Token t) {
        while (!this.range.contains(t) && this.ranges.hasNext()) {
            this.range = (MerkleTree.TreeRange)this.ranges.next();
        }
        return this.range.contains(t);
    }

    private MerkleTree.RowHash rowHash(UnfilteredRowIterator partition) {
        ++this.validated;
        CountingHasher hasher = new CountingHasher();
        UnfilteredRowIterators.digest(partition, hasher, 12);
        return hasher.count > 0L ? new MerkleTree.RowHash(partition.partitionKey().getToken(), hasher.hash().asBytes(), hasher.count) : null;
    }

    public void complete() {
        assert (this.ranges != null) : "Validator was not prepared()";
        if (logger.isDebugEnabled()) {
            logger.debug("Validated {} partitions for {}.  Partitions per leaf are:", (Object)this.validated, (Object)this.desc.sessionId);
            this.trees.logRowCountPerLeaf(logger);
            logger.debug("Validated {} partitions for {}.  Partition sizes are:", (Object)this.validated, (Object)this.desc.sessionId);
            this.trees.logRowSizePerLeaf(logger);
        }
        StageManager.getStage(Stage.ANTI_ENTROPY).execute(this);
    }

    public void fail() {
        logger.error("Failed creating a merkle tree for {}, {} (see log for details)", (Object)this.desc, (Object)this.initiator);
        this.respond(new ValidationResponse(this.desc));
    }

    @Override
    public void run() {
        if (this.initiatorIsRemote()) {
            logger.info("{} Sending completed merkle tree to {} for {}.{}", new Object[]{this.previewKind.logPrefix(this.desc.sessionId), this.initiator, this.desc.keyspace, this.desc.columnFamily});
            Tracing.traceRepair("Sending completed merkle tree to {} for {}.{}", this.initiator, this.desc.keyspace, this.desc.columnFamily);
        } else {
            logger.info("{} Local completed merkle tree for {} for {}.{}", new Object[]{this.previewKind.logPrefix(this.desc.sessionId), this.initiator, this.desc.keyspace, this.desc.columnFamily});
            Tracing.traceRepair("Local completed merkle tree for {} for {}.{}", this.initiator, this.desc.keyspace, this.desc.columnFamily);
        }
        this.respond(new ValidationResponse(this.desc, this.trees));
    }

    private boolean initiatorIsRemote() {
        return !FBUtilities.getBroadcastAddressAndPort().equals(this.initiator);
    }

    private void respond(ValidationResponse response) {
        if (this.initiatorIsRemote()) {
            MessagingService.instance().send(Message.out(Verb.VALIDATION_RSP, response), this.initiator);
            return;
        }
        StageManager.getStage(Stage.ANTI_ENTROPY).execute(() -> {
            ValidationResponse movedResponse = response;
            try {
                movedResponse = response.tryMoveOffHeap();
            }
            catch (IOException e) {
                logger.error("Failed to move local merkle tree for {} off heap", (Object)this.desc, (Object)e);
            }
            ActiveRepairService.instance.handleMessage(Message.out(Verb.VALIDATION_RSP, movedResponse));
        });
    }

    @VisibleForTesting
    static class CountingHasher
    implements Hasher {
        @VisibleForTesting
        static final HashFunction[] hashFunctions = new HashFunction[2];
        private long count;
        private final int bits;
        private final Hasher[] underlying = new Hasher[2];

        CountingHasher() {
            int bits = 0;
            for (int i = 0; i < this.underlying.length; ++i) {
                this.underlying[i] = hashFunctions[i].newHasher();
                bits += hashFunctions[i].bits();
            }
            this.bits = bits;
        }

        public Hasher putByte(byte b) {
            ++this.count;
            for (Hasher h : this.underlying) {
                h.putByte(b);
            }
            return this;
        }

        public Hasher putBytes(byte[] bytes) {
            this.count += (long)bytes.length;
            for (Hasher h : this.underlying) {
                h.putBytes(bytes);
            }
            return this;
        }

        public Hasher putBytes(byte[] bytes, int offset, int length) {
            this.count += (long)length;
            for (Hasher h : this.underlying) {
                h.putBytes(bytes, offset, length);
            }
            return this;
        }

        public Hasher putBytes(ByteBuffer byteBuffer) {
            this.count += (long)byteBuffer.remaining();
            for (Hasher h : this.underlying) {
                h.putBytes(byteBuffer.duplicate());
            }
            return this;
        }

        public Hasher putShort(short i) {
            this.count += 2L;
            for (Hasher h : this.underlying) {
                h.putShort(i);
            }
            return this;
        }

        public Hasher putInt(int i) {
            this.count += 4L;
            for (Hasher h : this.underlying) {
                h.putInt(i);
            }
            return this;
        }

        public Hasher putLong(long l) {
            this.count += 8L;
            for (Hasher h : this.underlying) {
                h.putLong(l);
            }
            return this;
        }

        public Hasher putFloat(float v) {
            this.count += 4L;
            for (Hasher h : this.underlying) {
                h.putFloat(v);
            }
            return this;
        }

        public Hasher putDouble(double v) {
            this.count += 8L;
            for (Hasher h : this.underlying) {
                h.putDouble(v);
            }
            return this;
        }

        public Hasher putBoolean(boolean b) {
            ++this.count;
            for (Hasher h : this.underlying) {
                h.putBoolean(b);
            }
            return this;
        }

        public Hasher putChar(char c) {
            this.count += 2L;
            for (Hasher h : this.underlying) {
                h.putChar(c);
            }
            return this;
        }

        public Hasher putUnencodedChars(CharSequence charSequence) {
            throw new UnsupportedOperationException();
        }

        public Hasher putString(CharSequence charSequence, Charset charset) {
            throw new UnsupportedOperationException();
        }

        public <T> Hasher putObject(T t, Funnel<? super T> funnel) {
            throw new UnsupportedOperationException();
        }

        public HashCode hash() {
            byte[] res = new byte[this.bits / 8];
            int i = 0;
            for (Hasher hasher : this.underlying) {
                HashCode newHash = hasher.hash();
                i += newHash.writeBytesTo(res, i, newHash.bits() / 8);
            }
            return HashCode.fromBytes((byte[])res);
        }

        public long getCount() {
            return this.count;
        }

        static {
            for (int i = 0; i < hashFunctions.length; ++i) {
                CountingHasher.hashFunctions[i] = Hashing.murmur3_128((int)(i * 1000));
            }
        }
    }
}

