/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.compress;

import com.github.luben.zstd.Zstd;
import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.cassandra.io.compress.BufferType;
import org.apache.cassandra.io.compress.ICompressor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZstdCompressor
implements ICompressor {
    private static final Logger logger = LoggerFactory.getLogger(ZstdCompressor.class);
    public static final int FAST_COMPRESSION_LEVEL = Zstd.minCompressionLevel();
    public static final int BEST_COMPRESSION_LEVEL = Zstd.maxCompressionLevel();
    public static final int DEFAULT_COMPRESSION_LEVEL = 3;
    private static final boolean ENABLE_CHECKSUM_FLAG = true;
    @VisibleForTesting
    public static final String COMPRESSION_LEVEL_OPTION_NAME = "compression_level";
    private static final ConcurrentHashMap<Integer, ZstdCompressor> instances = new ConcurrentHashMap();
    private final int compressionLevel;

    public static ZstdCompressor create(Map<String, String> options) {
        int level = ZstdCompressor.getOrDefaultCompressionLevel(options);
        if (!ZstdCompressor.isValid(level)) {
            throw new IllegalArgumentException(String.format("%s=%d is invalid", COMPRESSION_LEVEL_OPTION_NAME, level));
        }
        return ZstdCompressor.getOrCreate(level);
    }

    private ZstdCompressor(int compressionLevel) {
        this.compressionLevel = compressionLevel;
        logger.trace("Creating Zstd Compressor with compression level={}", (Object)compressionLevel);
    }

    public static ZstdCompressor getOrCreate(int level) {
        return instances.computeIfAbsent(level, l -> new ZstdCompressor(level));
    }

    @Override
    public int initialCompressedBufferLength(int chunkLength) {
        return (int)Zstd.compressBound((long)chunkLength);
    }

    @Override
    public int uncompress(byte[] input, int inputOffset, int inputLength, byte[] output, int outputOffset) throws IOException {
        long dsz = Zstd.decompressByteArray((byte[])output, (int)outputOffset, (int)(output.length - outputOffset), (byte[])input, (int)inputOffset, (int)inputLength);
        if (Zstd.isError((long)dsz)) {
            throw new IOException(String.format("Decompression failed due to %s", Zstd.getErrorName((long)dsz)));
        }
        return (int)dsz;
    }

    @Override
    public void uncompress(ByteBuffer input, ByteBuffer output) throws IOException {
        try {
            Zstd.decompress((ByteBuffer)output, (ByteBuffer)input);
        }
        catch (Exception e) {
            throw new IOException("Decompression failed", e);
        }
    }

    @Override
    public void compress(ByteBuffer input, ByteBuffer output) throws IOException {
        try {
            Zstd.compress((ByteBuffer)output, (ByteBuffer)input, (int)this.compressionLevel, (boolean)true);
        }
        catch (Exception e) {
            throw new IOException("Compression failed", e);
        }
    }

    private static boolean isValid(int level) {
        return level >= FAST_COMPRESSION_LEVEL && level <= BEST_COMPRESSION_LEVEL;
    }

    private static int getOrDefaultCompressionLevel(Map<String, String> options) {
        if (options == null) {
            return 3;
        }
        String val = options.get(COMPRESSION_LEVEL_OPTION_NAME);
        if (val == null) {
            return 3;
        }
        return Integer.valueOf(val);
    }

    @Override
    public BufferType preferredBufferType() {
        return BufferType.OFF_HEAP;
    }

    @Override
    public boolean supports(BufferType bufferType) {
        return bufferType == BufferType.OFF_HEAP;
    }

    @Override
    public Set<String> supportedOptions() {
        return new HashSet<String>(Collections.singletonList(COMPRESSION_LEVEL_OPTION_NAME));
    }

    @VisibleForTesting
    public int getCompressionLevel() {
        return this.compressionLevel;
    }
}

