/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import java.io.Closeable;
import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.UUID;
import org.apache.cassandra.db.ClusteringComparator;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.DeletionTime;
import org.apache.cassandra.db.RegularAndStaticColumns;
import org.apache.cassandra.db.RowIndexEntry;
import org.apache.cassandra.db.compaction.CompactionInfo;
import org.apache.cassandra.db.compaction.CompactionInterruptedException;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.db.partitions.ImmutableBTreePartition;
import org.apache.cassandra.db.partitions.Partition;
import org.apache.cassandra.db.rows.Cell;
import org.apache.cassandra.db.rows.ColumnData;
import org.apache.cassandra.db.rows.ComplexColumnData;
import org.apache.cassandra.db.rows.EncodingStats;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.db.rows.Rows;
import org.apache.cassandra.db.rows.Unfiltered;
import org.apache.cassandra.db.rows.UnfilteredRowIterator;
import org.apache.cassandra.db.rows.UnfilteredRowIterators;
import org.apache.cassandra.db.rows.WrappingUnfilteredRowIterator;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.SSTableIdentityIterator;
import org.apache.cassandra.io.sstable.SSTableRewriter;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.sstable.format.SSTableWriter;
import org.apache.cassandra.io.sstable.metadata.StatsMetadata;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.utils.AbstractIterator;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.OutputHandler;
import org.apache.cassandra.utils.UUIDGen;
import org.apache.cassandra.utils.concurrent.Refs;
import org.apache.cassandra.utils.memory.HeapAllocator;

public class Scrubber
implements Closeable {
    private final ColumnFamilyStore cfs;
    private final SSTableReader sstable;
    private final LifecycleTransaction transaction;
    private final File destination;
    private final boolean skipCorrupted;
    private final boolean reinsertOverflowedTTLRows;
    private final boolean isCommutative;
    private final boolean isIndex;
    private final boolean checkData;
    private final long expectedBloomFilterSize;
    private final RandomAccessReader dataFile;
    private final RandomAccessReader indexFile;
    private final ScrubInfo scrubInfo;
    private final RowIndexEntry.IndexSerializer rowIndexEntrySerializer;
    private int goodRows;
    private int badRows;
    private int emptyRows;
    private ByteBuffer currentIndexKey;
    private ByteBuffer nextIndexKey;
    long currentRowPositionFromIndex;
    long nextRowPositionFromIndex;
    private NegativeLocalDeletionInfoMetrics negativeLocalDeletionInfoMetrics = new NegativeLocalDeletionInfoMetrics();
    private final OutputHandler outputHandler;
    private static final Comparator<Partition> partitionComparator = new Comparator<Partition>(){

        @Override
        public int compare(Partition r1, Partition r2) {
            return r1.partitionKey().compareTo(r2.partitionKey());
        }
    };
    private final SortedSet<Partition> outOfOrder = new TreeSet<Partition>(partitionComparator);

    public Scrubber(ColumnFamilyStore cfs, LifecycleTransaction transaction, boolean skipCorrupted, boolean checkData) {
        this(cfs, transaction, skipCorrupted, checkData, false);
    }

    public Scrubber(ColumnFamilyStore cfs, LifecycleTransaction transaction, boolean skipCorrupted, boolean checkData, boolean reinsertOverflowedTTLRows) {
        this(cfs, transaction, skipCorrupted, new OutputHandler.LogOutput(), checkData, reinsertOverflowedTTLRows);
    }

    public Scrubber(ColumnFamilyStore cfs, LifecycleTransaction transaction, boolean skipCorrupted, OutputHandler outputHandler, boolean checkData, boolean reinsertOverflowedTTLRows) {
        this.cfs = cfs;
        this.transaction = transaction;
        this.sstable = transaction.onlyOne();
        this.outputHandler = outputHandler;
        this.skipCorrupted = skipCorrupted;
        this.reinsertOverflowedTTLRows = reinsertOverflowedTTLRows;
        this.rowIndexEntrySerializer = this.sstable.descriptor.version.getSSTableFormat().getIndexSerializer(cfs.metadata(), this.sstable.descriptor.version, this.sstable.header);
        List<SSTableReader> toScrub = Collections.singletonList(this.sstable);
        this.destination = cfs.getDirectories().getLocationForDisk(cfs.getDiskBoundaries().getCorrectDiskForSSTable(this.sstable));
        this.isCommutative = cfs.metadata().isCounter();
        boolean hasIndexFile = new File(this.sstable.descriptor.filenameFor(Component.PRIMARY_INDEX)).exists();
        this.isIndex = cfs.isIndex();
        if (!hasIndexFile) {
            outputHandler.warn("Missing component: " + this.sstable.descriptor.filenameFor(Component.PRIMARY_INDEX));
        }
        this.checkData = checkData && !this.isIndex;
        this.expectedBloomFilterSize = Math.max((long)cfs.metadata().params.minIndexInterval, hasIndexFile ? SSTableReader.getApproximateKeyCount(toScrub) : 0L);
        this.dataFile = transaction.isOffline() ? this.sstable.openDataReader() : this.sstable.openDataReader(CompactionManager.instance.getRateLimiter());
        this.indexFile = hasIndexFile ? RandomAccessReader.open(new File(this.sstable.descriptor.filenameFor(Component.PRIMARY_INDEX))) : null;
        this.scrubInfo = new ScrubInfo(this.dataFile, this.sstable);
        this.currentRowPositionFromIndex = 0L;
        this.nextRowPositionFromIndex = 0L;
        if (reinsertOverflowedTTLRows) {
            outputHandler.output("Starting scrub with reinsert overflowed TTL option");
        }
    }

    private UnfilteredRowIterator withValidation(UnfilteredRowIterator iter, String filename) {
        return this.checkData ? UnfilteredRowIterators.withValidation(iter, filename) : iter;
    }

    public void scrub() {
        ArrayList<SSTableReader> finished = new ArrayList<SSTableReader>();
        boolean completed = false;
        this.outputHandler.output(String.format("Scrubbing %s (%s)", this.sstable, FBUtilities.prettyPrintMemory(this.dataFile.length())));
        try (SSTableRewriter writer = SSTableRewriter.construct(this.cfs, this.transaction, false, this.sstable.maxDataAge);
             Refs<SSTableReader> refs = Refs.ref(Collections.singleton(this.sstable));){
            ByteBuffer byteBuffer = this.nextIndexKey = this.indexAvailable() ? ByteBufferUtil.readWithShortLength(this.indexFile) : null;
            if (this.indexAvailable()) {
                long firstRowPositionFromIndex = this.rowIndexEntrySerializer.deserializePositionAndSkip(this.indexFile);
                assert (firstRowPositionFromIndex == 0L) : firstRowPositionFromIndex;
            }
            StatsMetadata metadata = this.sstable.getSSTableMetadata();
            writer.switchWriter(CompactionManager.createWriter(this.cfs, this.destination, this.expectedBloomFilterSize, metadata.repairedAt, metadata.pendingRepair, metadata.isTransient, this.sstable, this.transaction));
            DecoratedKey prevKey = null;
            while (!this.dataFile.isEOF()) {
                if (this.scrubInfo.isStopRequested()) {
                    throw new CompactionInterruptedException(this.scrubInfo.getCompactionInfo());
                }
                long rowStart = this.dataFile.getFilePointer();
                this.outputHandler.debug("Reading row at " + rowStart);
                DecoratedKey key = null;
                try {
                    key = this.sstable.decorateKey(ByteBufferUtil.readWithShortLength(this.dataFile));
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                }
                this.updateIndexKey();
                long dataStart = this.dataFile.getFilePointer();
                long dataStartFromIndex = -1L;
                long dataSizeFromIndex = -1L;
                if (this.currentIndexKey != null) {
                    dataStartFromIndex = this.currentRowPositionFromIndex + 2L + (long)this.currentIndexKey.remaining();
                    dataSizeFromIndex = this.nextRowPositionFromIndex - dataStartFromIndex;
                }
                String keyName = key == null ? "(unreadable key)" : ByteBufferUtil.bytesToHex(key.getKey());
                this.outputHandler.debug(String.format("row %s is %s", keyName, FBUtilities.prettyPrintMemory(dataSizeFromIndex)));
                assert (this.currentIndexKey != null || !this.indexAvailable());
                try {
                    if (key == null) {
                        throw new IOError(new IOException("Unable to read row key from data file"));
                    }
                    if (this.currentIndexKey != null && !key.getKey().equals(this.currentIndexKey)) {
                        throw new IOError(new IOException(String.format("Key from data file (%s) does not match key from index file (%s)", "_too big_", ByteBufferUtil.bytesToHex(this.currentIndexKey))));
                    }
                    if (this.indexFile != null && dataSizeFromIndex > this.dataFile.length()) {
                        throw new IOError(new IOException("Impossible row size (greater than file length): " + dataSizeFromIndex));
                    }
                    if (this.indexFile != null && dataStart != dataStartFromIndex) {
                        this.outputHandler.warn(String.format("Data file row position %d differs from index file row position %d", dataStart, dataStartFromIndex));
                    }
                    if (!this.tryAppend(prevKey, key, writer)) continue;
                    prevKey = key;
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                    this.outputHandler.warn("Error reading row (stacktrace follows):", th);
                    if (!(this.currentIndexKey == null || key != null && key.getKey().equals(this.currentIndexKey) && dataStart == dataStartFromIndex)) {
                        this.outputHandler.output(String.format("Retrying from row index; data is %s bytes starting at %s", dataSizeFromIndex, dataStartFromIndex));
                        key = this.sstable.decorateKey(this.currentIndexKey);
                        try {
                            this.dataFile.seek(dataStartFromIndex);
                            if (!this.tryAppend(prevKey, key, writer)) continue;
                            prevKey = key;
                        }
                        catch (Throwable th2) {
                            this.throwIfFatal(th2);
                            this.throwIfCannotContinue(key, th2);
                            this.outputHandler.warn("Retry failed too. Skipping to next row (retry's stacktrace follows)", th2);
                            ++this.badRows;
                            this.seekToNextRow();
                        }
                        continue;
                    }
                    this.throwIfCannotContinue(key, th);
                    this.outputHandler.warn("Row starting at position " + dataStart + " is unreadable; skipping to next");
                    ++this.badRows;
                    if (this.currentIndexKey == null) continue;
                    this.seekToNextRow();
                }
            }
            if (!this.outOfOrder.isEmpty()) {
                SSTableReader newInOrderSstable;
                long repairedAt = this.badRows > 0 ? 0L : metadata.repairedAt;
                try (SSTableWriter inOrderWriter = CompactionManager.createWriter(this.cfs, this.destination, this.expectedBloomFilterSize, repairedAt, metadata.pendingRepair, metadata.isTransient, this.sstable, this.transaction);){
                    for (Partition partition : this.outOfOrder) {
                        inOrderWriter.append(partition.unfilteredIterator());
                    }
                    newInOrderSstable = inOrderWriter.finish(-1L, this.sstable.maxDataAge, true);
                }
                this.transaction.update(newInOrderSstable, false);
                finished.add(newInOrderSstable);
                this.outputHandler.warn(String.format("%d out of order rows found while scrubbing %s; Those have been written (in order) to a new sstable (%s)", this.outOfOrder.size(), this.sstable, newInOrderSstable));
            }
            finished.addAll((Collection<SSTableReader>)writer.setRepairedAt(this.badRows > 0 ? 0L : this.sstable.getSSTableMetadata().repairedAt).finish());
            completed = true;
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
        finally {
            if (this.transaction.isOffline()) {
                finished.forEach(sstable -> sstable.selfRef().release());
            }
        }
        if (completed) {
            this.outputHandler.output("Scrub of " + this.sstable + " complete: " + this.goodRows + " rows in new sstable and " + this.emptyRows + " empty (tombstoned) rows dropped");
            if (this.negativeLocalDeletionInfoMetrics.fixedRows > 0) {
                this.outputHandler.output("Fixed " + this.negativeLocalDeletionInfoMetrics.fixedRows + " rows with overflowed local deletion time.");
            }
            if (this.badRows > 0) {
                this.outputHandler.warn("Unable to recover " + this.badRows + " rows that were skipped.  You can attempt manual recovery from the pre-scrub snapshot.  You can also run nodetool repair to transfer the data from a healthy replica, if any");
            }
        } else if (this.badRows > 0) {
            this.outputHandler.warn("No valid rows found while scrubbing " + this.sstable + "; it is marked for deletion now. If you want to attempt manual recovery, you can find a copy in the pre-scrub snapshot");
        } else {
            this.outputHandler.output("Scrub of " + this.sstable + " complete; looks like all " + this.emptyRows + " rows were tombstoned");
        }
    }

    private boolean tryAppend(DecoratedKey prevKey, DecoratedKey key, SSTableRewriter writer) {
        OrderCheckerIterator sstableIterator = new OrderCheckerIterator(this.getIterator(key), this.cfs.metadata().comparator);
        try (UnfilteredRowIterator iterator = this.withValidation(sstableIterator, this.dataFile.getPath());){
            if (prevKey != null && prevKey.compareTo(key) > 0) {
                this.saveOutOfOrderRow(prevKey, key, iterator);
                boolean bl = false;
                return bl;
            }
            if (writer.tryAppend(iterator) == null) {
                ++this.emptyRows;
            } else {
                ++this.goodRows;
            }
        }
        if (sstableIterator.hasRowsOutOfOrder()) {
            this.outputHandler.warn(String.format("Out of order rows found in partition: %s", key));
            this.outOfOrder.add(sstableIterator.getRowsOutOfOrder());
        }
        return true;
    }

    private UnfilteredRowIterator getIterator(DecoratedKey key) {
        RowMergingSSTableIterator rowMergingIterator = new RowMergingSSTableIterator(SSTableIdentityIterator.create(this.sstable, this.dataFile, key));
        return this.reinsertOverflowedTTLRows ? new FixNegativeLocalDeletionTimeIterator(rowMergingIterator, this.outputHandler, this.negativeLocalDeletionInfoMetrics) : rowMergingIterator;
    }

    private void updateIndexKey() {
        this.currentIndexKey = this.nextIndexKey;
        this.currentRowPositionFromIndex = this.nextRowPositionFromIndex;
        try {
            this.nextIndexKey = !this.indexAvailable() ? null : ByteBufferUtil.readWithShortLength(this.indexFile);
            this.nextRowPositionFromIndex = !this.indexAvailable() ? this.dataFile.length() : this.rowIndexEntrySerializer.deserializePositionAndSkip(this.indexFile);
        }
        catch (Throwable th) {
            JVMStabilityInspector.inspectThrowable(th);
            this.outputHandler.warn("Error reading index file", th);
            this.nextIndexKey = null;
            this.nextRowPositionFromIndex = this.dataFile.length();
        }
    }

    private boolean indexAvailable() {
        return this.indexFile != null && !this.indexFile.isEOF();
    }

    private void seekToNextRow() {
        while (this.nextRowPositionFromIndex < this.dataFile.length()) {
            try {
                this.dataFile.seek(this.nextRowPositionFromIndex);
                return;
            }
            catch (Throwable th) {
                this.throwIfFatal(th);
                this.outputHandler.warn(String.format("Failed to seek to next row position %d", this.nextRowPositionFromIndex), th);
                ++this.badRows;
                this.updateIndexKey();
            }
        }
    }

    private void saveOutOfOrderRow(DecoratedKey prevKey, DecoratedKey key, UnfilteredRowIterator iterator) {
        this.outputHandler.warn(String.format("Out of order row detected (%s found after %s)", key, prevKey));
        this.outOfOrder.add(ImmutableBTreePartition.create(iterator));
    }

    private void throwIfFatal(Throwable th) {
        if (th instanceof Error && !(th instanceof AssertionError) && !(th instanceof IOError)) {
            throw (Error)th;
        }
    }

    private void throwIfCannotContinue(DecoratedKey key, Throwable th) {
        if (this.isIndex) {
            this.outputHandler.warn(String.format("An error occurred while scrubbing the row with key '%s' for an index table. Scrubbing will abort for this table and the index will be rebuilt.", key));
            throw new IOError(th);
        }
        if (this.isCommutative && !this.skipCorrupted) {
            this.outputHandler.warn(String.format("An error occurred while scrubbing the row with key '%s'.  Skipping corrupt rows in counter tables will result in undercounts for the affected counters (see CASSANDRA-2759 for more details), so by default the scrub will stop at this point.  If you would like to skip the row anyway and continue scrubbing, re-run the scrub with the --skip-corrupted option.", key));
            throw new IOError(th);
        }
    }

    @Override
    public void close() {
        FileUtils.closeQuietly(this.dataFile);
        FileUtils.closeQuietly(this.indexFile);
    }

    public CompactionInfo.Holder getScrubInfo() {
        return this.scrubInfo;
    }

    @VisibleForTesting
    public ScrubResult scrubWithResult() {
        this.scrub();
        return new ScrubResult(this);
    }

    private static final class FixNegativeLocalDeletionTimeIterator
    extends AbstractIterator<Unfiltered>
    implements UnfilteredRowIterator {
        private final UnfilteredRowIterator iterator;
        private final OutputHandler outputHandler;
        private final NegativeLocalDeletionInfoMetrics negativeLocalExpirationTimeMetrics;

        public FixNegativeLocalDeletionTimeIterator(UnfilteredRowIterator iterator, OutputHandler outputHandler, NegativeLocalDeletionInfoMetrics negativeLocalDeletionInfoMetrics) {
            this.iterator = iterator;
            this.outputHandler = outputHandler;
            this.negativeLocalExpirationTimeMetrics = negativeLocalDeletionInfoMetrics;
        }

        @Override
        public TableMetadata metadata() {
            return this.iterator.metadata();
        }

        @Override
        public boolean isReverseOrder() {
            return this.iterator.isReverseOrder();
        }

        @Override
        public RegularAndStaticColumns columns() {
            return this.iterator.columns();
        }

        @Override
        public DecoratedKey partitionKey() {
            return this.iterator.partitionKey();
        }

        @Override
        public Row staticRow() {
            return this.iterator.staticRow();
        }

        @Override
        public boolean isEmpty() {
            return this.iterator.isEmpty();
        }

        @Override
        public void close() {
            this.iterator.close();
        }

        @Override
        public DeletionTime partitionLevelDeletion() {
            return this.iterator.partitionLevelDeletion();
        }

        @Override
        public EncodingStats stats() {
            return this.iterator.stats();
        }

        @Override
        protected Unfiltered computeNext() {
            if (!this.iterator.hasNext()) {
                return (Unfiltered)this.endOfData();
            }
            Unfiltered next = (Unfiltered)this.iterator.next();
            if (!next.isRow()) {
                return next;
            }
            if (this.hasNegativeLocalExpirationTime((Row)next)) {
                this.outputHandler.debug(String.format("Found row with negative local expiration time: %s", next.toString(this.metadata(), false)));
                ++this.negativeLocalExpirationTimeMetrics.fixedRows;
                return this.fixNegativeLocalExpirationTime((Row)next);
            }
            return next;
        }

        private boolean hasNegativeLocalExpirationTime(Row next) {
            Row row = next;
            if (row.primaryKeyLivenessInfo().isExpiring() && row.primaryKeyLivenessInfo().localExpirationTime() < 0) {
                return true;
            }
            for (ColumnData cd : row) {
                if (cd.column().isSimple()) {
                    Cell cell = (Cell)cd;
                    if (!cell.isExpiring() || cell.localDeletionTime() >= 0) continue;
                    return true;
                }
                ComplexColumnData complexData = (ComplexColumnData)cd;
                for (Cell cell : complexData) {
                    if (!cell.isExpiring() || cell.localDeletionTime() >= 0) continue;
                    return true;
                }
            }
            return false;
        }

        private Unfiltered fixNegativeLocalExpirationTime(Row row) {
            Row.Builder builder = HeapAllocator.instance.cloningBTreeRowBuilder();
            builder.newRow(row.clustering());
            builder.addPrimaryKeyLivenessInfo(row.primaryKeyLivenessInfo().isExpiring() && row.primaryKeyLivenessInfo().localExpirationTime() < 0 ? row.primaryKeyLivenessInfo().withUpdatedTimestampAndLocalDeletionTime(row.primaryKeyLivenessInfo().timestamp() + 1L, 0x7FFFFFFE) : row.primaryKeyLivenessInfo());
            builder.addRowDeletion(row.deletion());
            for (ColumnData cd : row) {
                if (cd.column().isSimple()) {
                    Cell cell = (Cell)cd;
                    builder.addCell(cell.isExpiring() && cell.localDeletionTime() < 0 ? cell.withUpdatedTimestampAndLocalDeletionTime(cell.timestamp() + 1L, 0x7FFFFFFE) : cell);
                    continue;
                }
                ComplexColumnData complexData = (ComplexColumnData)cd;
                builder.addComplexDeletion(complexData.column(), complexData.complexDeletion());
                for (Cell cell : complexData) {
                    builder.addCell(cell.isExpiring() && cell.localDeletionTime() < 0 ? cell.withUpdatedTimestampAndLocalDeletionTime(cell.timestamp() + 1L, 0x7FFFFFFE) : cell);
                }
            }
            return builder.build();
        }
    }

    private static final class OrderCheckerIterator
    extends AbstractIterator<Unfiltered>
    implements UnfilteredRowIterator {
        private final UnfilteredRowIterator iterator;
        private final ClusteringComparator comparator;
        private Unfiltered previous;
        private Partition rowsOutOfOrder;

        public OrderCheckerIterator(UnfilteredRowIterator iterator, ClusteringComparator comparator) {
            this.iterator = iterator;
            this.comparator = comparator;
        }

        @Override
        public TableMetadata metadata() {
            return this.iterator.metadata();
        }

        @Override
        public boolean isReverseOrder() {
            return this.iterator.isReverseOrder();
        }

        @Override
        public RegularAndStaticColumns columns() {
            return this.iterator.columns();
        }

        @Override
        public DecoratedKey partitionKey() {
            return this.iterator.partitionKey();
        }

        @Override
        public Row staticRow() {
            return this.iterator.staticRow();
        }

        @Override
        public boolean isEmpty() {
            return this.iterator.isEmpty();
        }

        @Override
        public void close() {
            this.iterator.close();
        }

        @Override
        public DeletionTime partitionLevelDeletion() {
            return this.iterator.partitionLevelDeletion();
        }

        @Override
        public EncodingStats stats() {
            return this.iterator.stats();
        }

        public boolean hasRowsOutOfOrder() {
            return this.rowsOutOfOrder != null;
        }

        public Partition getRowsOutOfOrder() {
            return this.rowsOutOfOrder;
        }

        @Override
        protected Unfiltered computeNext() {
            if (!this.iterator.hasNext()) {
                return (Unfiltered)this.endOfData();
            }
            Unfiltered next = (Unfiltered)this.iterator.next();
            if (this.previous != null && this.comparator.compare(next, this.previous) < 0) {
                this.rowsOutOfOrder = ImmutableBTreePartition.create(UnfilteredRowIterators.concat(next, this.iterator), false);
                return (Unfiltered)this.endOfData();
            }
            this.previous = next;
            return next;
        }
    }

    private static class RowMergingSSTableIterator
    extends WrappingUnfilteredRowIterator {
        Unfiltered nextToOffer = null;

        RowMergingSSTableIterator(UnfilteredRowIterator source) {
            super(source);
        }

        @Override
        public boolean hasNext() {
            return this.nextToOffer != null || this.wrapped.hasNext();
        }

        @Override
        public Unfiltered next() {
            Unfiltered next;
            Unfiltered unfiltered = next = this.nextToOffer != null ? this.nextToOffer : (Unfiltered)this.wrapped.next();
            if (next.isRow()) {
                while (this.wrapped.hasNext()) {
                    Unfiltered peek = (Unfiltered)this.wrapped.next();
                    if (!peek.isRow() || !next.clustering().equals(peek.clustering())) {
                        this.nextToOffer = peek;
                        return next;
                    }
                    next = Rows.merge((Row)next, (Row)peek);
                }
            }
            this.nextToOffer = null;
            return next;
        }
    }

    public class NegativeLocalDeletionInfoMetrics {
        public volatile int fixedRows = 0;
    }

    public static final class ScrubResult {
        public final int goodRows;
        public final int badRows;
        public final int emptyRows;

        public ScrubResult(Scrubber scrubber) {
            this.goodRows = scrubber.goodRows;
            this.badRows = scrubber.badRows;
            this.emptyRows = scrubber.emptyRows;
        }
    }

    private static class ScrubInfo
    extends CompactionInfo.Holder {
        private final RandomAccessReader dataFile;
        private final SSTableReader sstable;
        private final UUID scrubCompactionId;

        public ScrubInfo(RandomAccessReader dataFile, SSTableReader sstable) {
            this.dataFile = dataFile;
            this.sstable = sstable;
            this.scrubCompactionId = UUIDGen.getTimeUUID();
        }

        @Override
        public CompactionInfo getCompactionInfo() {
            try {
                return new CompactionInfo(this.sstable.metadata(), OperationType.SCRUB, this.dataFile.getFilePointer(), this.dataFile.length(), this.scrubCompactionId, (Collection<SSTableReader>)ImmutableSet.of((Object)this.sstable));
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
    }
}

