/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.thrift;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ColumnDefinition;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.cql3.Operator;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.KeyspaceNotDefinedException;
import org.apache.cassandra.db.LegacyLayout;
import org.apache.cassandra.db.PartitionPosition;
import org.apache.cassandra.db.UnknownColumnException;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.index.Index;
import org.apache.cassandra.index.SecondaryIndexManager;
import org.apache.cassandra.serializers.MarshalException;
import org.apache.cassandra.thrift.Column;
import org.apache.cassandra.thrift.ColumnOrSuperColumn;
import org.apache.cassandra.thrift.ColumnParent;
import org.apache.cassandra.thrift.ColumnPath;
import org.apache.cassandra.thrift.Deletion;
import org.apache.cassandra.thrift.IndexClause;
import org.apache.cassandra.thrift.IndexExpression;
import org.apache.cassandra.thrift.KeyRange;
import org.apache.cassandra.thrift.Mutation;
import org.apache.cassandra.thrift.SlicePredicate;
import org.apache.cassandra.thrift.SliceRange;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ThriftValidation {
    private static final Logger logger = LoggerFactory.getLogger(ThriftValidation.class);

    public static void validateKey(CFMetaData metadata, ByteBuffer key) throws InvalidRequestException {
        if (key == null || key.remaining() == 0) {
            throw new InvalidRequestException("Key may not be empty");
        }
        if (key.remaining() > 65535) {
            throw new InvalidRequestException("Key length of " + key.remaining() + " is longer than maximum of " + 65535);
        }
        try {
            metadata.getKeyValidator().validate(key);
        }
        catch (MarshalException e) {
            throw new InvalidRequestException(e.getMessage());
        }
    }

    public static void validateKeyspace(String keyspaceName) throws KeyspaceNotDefinedException {
        if (!Schema.instance.getKeyspaces().contains(keyspaceName)) {
            throw new KeyspaceNotDefinedException("Keyspace " + keyspaceName + " does not exist");
        }
    }

    public static CFMetaData validateColumnFamily(String keyspaceName, String cfName, boolean isCommutativeOp) throws InvalidRequestException {
        CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspaceName, cfName);
        if (isCommutativeOp) {
            if (!metadata.isCounter()) {
                throw new InvalidRequestException("invalid operation for non commutative table " + cfName);
            }
        } else if (metadata.isCounter()) {
            throw new InvalidRequestException("invalid operation for commutative table " + cfName);
        }
        return metadata;
    }

    public static CFMetaData validateColumnFamily(String keyspaceName, String cfName) throws InvalidRequestException {
        ThriftValidation.validateKeyspace(keyspaceName);
        if (cfName.isEmpty()) {
            throw new InvalidRequestException("non-empty table is required");
        }
        CFMetaData metadata = Schema.instance.getCFMetaData(keyspaceName, cfName);
        if (metadata == null) {
            throw new InvalidRequestException("unconfigured table " + cfName);
        }
        return metadata;
    }

    public static void validateColumnPath(CFMetaData metadata, ColumnPath column_path) throws InvalidRequestException {
        if (!metadata.isSuper()) {
            if (column_path.super_column != null) {
                throw new InvalidRequestException("supercolumn parameter is invalid for standard CF " + metadata.cfName);
            }
            if (column_path.column == null) {
                throw new InvalidRequestException("column parameter is not optional for standard CF " + metadata.cfName);
            }
        } else if (column_path.super_column == null) {
            throw new InvalidRequestException("supercolumn parameter is not optional for super CF " + metadata.cfName);
        }
        if (column_path.column != null) {
            ThriftValidation.validateColumnNames(metadata, column_path.super_column, Arrays.asList(column_path.column));
        }
        if (column_path.super_column != null) {
            ThriftValidation.validateColumnNames(metadata, (ByteBuffer)null, Arrays.asList(column_path.super_column));
        }
    }

    public static void validateColumnParent(CFMetaData metadata, ColumnParent column_parent) throws InvalidRequestException {
        if (!metadata.isSuper() && column_parent.super_column != null) {
            throw new InvalidRequestException("table alone is required for standard CF " + metadata.cfName);
        }
        if (column_parent.super_column != null) {
            ThriftValidation.validateColumnNames(metadata, (ByteBuffer)null, Arrays.asList(column_parent.super_column));
        }
    }

    static void validateColumnPathOrParent(CFMetaData metadata, ColumnPath column_path_or_parent) throws InvalidRequestException {
        if (metadata.isSuper()) {
            if (column_path_or_parent.super_column == null && column_path_or_parent.column != null) {
                throw new InvalidRequestException("A column cannot be specified without specifying a super column for removal on super CF " + metadata.cfName);
            }
        } else if (column_path_or_parent.super_column != null) {
            throw new InvalidRequestException("supercolumn may not be specified for standard CF " + metadata.cfName);
        }
        if (column_path_or_parent.column != null) {
            ThriftValidation.validateColumnNames(metadata, column_path_or_parent.super_column, Arrays.asList(column_path_or_parent.column));
        }
        if (column_path_or_parent.super_column != null) {
            ThriftValidation.validateColumnNames(metadata, (ByteBuffer)null, Arrays.asList(column_path_or_parent.super_column));
        }
    }

    private static AbstractType<?> getThriftColumnNameComparator(CFMetaData metadata, ByteBuffer superColumnName) {
        if (!metadata.isSuper()) {
            return LegacyLayout.makeLegacyComparator(metadata);
        }
        if (superColumnName == null) {
            return metadata.comparator.subtype(0);
        }
        return metadata.thriftColumnNameType();
    }

    private static void validateColumnNames(CFMetaData metadata, ByteBuffer superColumnName, Iterable<ByteBuffer> column_names) throws InvalidRequestException {
        int maxNameLength = 65535;
        if (superColumnName != null) {
            if (superColumnName.remaining() > maxNameLength) {
                throw new InvalidRequestException("supercolumn name length must not be greater than " + maxNameLength);
            }
            if (superColumnName.remaining() == 0) {
                throw new InvalidRequestException("supercolumn name must not be empty");
            }
            if (!metadata.isSuper()) {
                throw new InvalidRequestException("supercolumn specified to table " + metadata.cfName + " containing normal columns");
            }
        }
        AbstractType<?> comparator = ThriftValidation.getThriftColumnNameComparator(metadata, superColumnName);
        boolean isCQL3Table = !metadata.isThriftCompatible();
        for (ByteBuffer name : column_names) {
            if (name.remaining() > maxNameLength) {
                throw new InvalidRequestException("column name length must not be greater than " + maxNameLength);
            }
            if (name.remaining() == 0) {
                throw new InvalidRequestException("column name must not be empty");
            }
            try {
                comparator.validate(name);
            }
            catch (MarshalException e) {
                throw new InvalidRequestException(e.getMessage());
            }
            if (!isCQL3Table) continue;
            try {
                LegacyLayout.LegacyCellName cname = LegacyLayout.decodeCellName(metadata, name);
                assert (cname.clustering.size() == metadata.comparator.size());
                for (int i = 0; i < cname.clustering.size(); ++i) {
                    if (cname.clustering.get(i) != null) continue;
                    throw new InvalidRequestException(String.format("Not enough components (found %d but %d expected) for column name since %s is a CQL3 table", i, metadata.comparator.size() + 1, metadata.cfName));
                }
                if (cname.column == null || cname.collectionElement == null || cname.column.type.isCollection()) continue;
                throw new InvalidRequestException(String.format("Invalid collection component, %s is not a collection", cname.column.name));
            }
            catch (IllegalArgumentException | UnknownColumnException e) {
                throw new InvalidRequestException(String.format("Error validating cell name for CQL3 table %s: %s", metadata.cfName, e.getMessage()));
            }
        }
    }

    public static void validateColumnNames(CFMetaData metadata, ColumnParent column_parent, Iterable<ByteBuffer> column_names) throws InvalidRequestException {
        ThriftValidation.validateColumnNames(metadata, column_parent.super_column, column_names);
    }

    public static void validateRange(CFMetaData metadata, ColumnParent column_parent, SliceRange range) throws InvalidRequestException {
        Comparator<ByteBuffer> orderedComparator;
        if (range.count < 0) {
            throw new InvalidRequestException("get_slice requires non-negative count");
        }
        int maxNameLength = 65535;
        if (range.start.remaining() > maxNameLength) {
            throw new InvalidRequestException("range start length cannot be larger than " + maxNameLength);
        }
        if (range.finish.remaining() > maxNameLength) {
            throw new InvalidRequestException("range finish length cannot be larger than " + maxNameLength);
        }
        AbstractType<?> comparator = ThriftValidation.getThriftColumnNameComparator(metadata, column_parent.super_column);
        try {
            comparator.validate(range.start);
            comparator.validate(range.finish);
        }
        catch (MarshalException e) {
            throw new InvalidRequestException(e.getMessage());
        }
        Comparator<ByteBuffer> comparator2 = orderedComparator = range.isReversed() ? comparator.reverseComparator : comparator;
        if (range.start.remaining() > 0 && range.finish.remaining() > 0 && orderedComparator.compare(range.start, range.finish) > 0) {
            throw new InvalidRequestException("range finish must come after start in the order of traversal");
        }
    }

    public static void validateColumnOrSuperColumn(CFMetaData metadata, ColumnOrSuperColumn cosc) throws InvalidRequestException {
        boolean isCommutative = metadata.isCounter();
        int nulls = 0;
        if (cosc.column == null) {
            ++nulls;
        }
        if (cosc.super_column == null) {
            ++nulls;
        }
        if (cosc.counter_column == null) {
            ++nulls;
        }
        if (cosc.counter_super_column == null) {
            ++nulls;
        }
        if (nulls != 3) {
            throw new InvalidRequestException("ColumnOrSuperColumn must have one (and only one) of column, super_column, counter and counter_super_column");
        }
        if (cosc.column != null) {
            if (isCommutative) {
                throw new InvalidRequestException("invalid operation for commutative table " + metadata.cfName);
            }
            ThriftValidation.validateTtl(cosc.column);
            ThriftValidation.validateColumnPath(metadata, new ColumnPath(metadata.cfName).setSuper_column((ByteBuffer)null).setColumn(cosc.column.name));
            ThriftValidation.validateColumnData(metadata, null, cosc.column);
        }
        if (cosc.super_column != null) {
            if (isCommutative) {
                throw new InvalidRequestException("invalid operation for commutative table " + metadata.cfName);
            }
            for (Column c : cosc.super_column.columns) {
                ThriftValidation.validateColumnPath(metadata, new ColumnPath(metadata.cfName).setSuper_column(cosc.super_column.name).setColumn(c.name));
                ThriftValidation.validateColumnData(metadata, cosc.super_column.name, c);
            }
        }
        if (cosc.counter_column != null) {
            if (!isCommutative) {
                throw new InvalidRequestException("invalid operation for non commutative table " + metadata.cfName);
            }
            ThriftValidation.validateColumnPath(metadata, new ColumnPath(metadata.cfName).setSuper_column((ByteBuffer)null).setColumn(cosc.counter_column.name));
        }
        if (cosc.counter_super_column != null) {
            if (!isCommutative) {
                throw new InvalidRequestException("invalid operation for non commutative table " + metadata.cfName);
            }
            for (Column c : cosc.counter_super_column.columns) {
                ThriftValidation.validateColumnPath(metadata, new ColumnPath(metadata.cfName).setSuper_column(cosc.counter_super_column.name).setColumn(c.name));
            }
        }
    }

    private static void validateTtl(Column column) throws InvalidRequestException {
        if (column.isSetTtl()) {
            if (column.ttl < 0) {
                throw new InvalidRequestException("ttl must be greater or equal to 0");
            }
            if (column.ttl > 630720000) {
                throw new InvalidRequestException(String.format("ttl is too large. requested (%d) maximum (%d)", column.ttl, 630720000));
            }
        } else assert (column.ttl == 0);
    }

    public static void validateMutation(CFMetaData metadata, Mutation mut) throws InvalidRequestException {
        ColumnOrSuperColumn cosc = mut.column_or_supercolumn;
        Deletion del = mut.deletion;
        int nulls = 0;
        if (cosc == null) {
            ++nulls;
        }
        if (del == null) {
            ++nulls;
        }
        if (nulls != 1) {
            throw new InvalidRequestException("mutation must have one and only one of column_or_supercolumn or deletion");
        }
        if (cosc != null) {
            ThriftValidation.validateColumnOrSuperColumn(metadata, cosc);
        } else {
            ThriftValidation.validateDeletion(metadata, del);
        }
    }

    public static void validateDeletion(CFMetaData metadata, Deletion del) throws InvalidRequestException {
        if (del.super_column != null) {
            ThriftValidation.validateColumnNames(metadata, (ByteBuffer)null, Arrays.asList(del.super_column));
        }
        if (del.predicate != null) {
            ThriftValidation.validateSlicePredicate(metadata, del.super_column, del.predicate);
        }
        if (!metadata.isSuper() && del.super_column != null) {
            String msg = String.format("Deletion of super columns is not possible on a standard table (KeySpace=%s Table=%s Deletion=%s)", metadata.ksName, metadata.cfName, del);
            throw new InvalidRequestException(msg);
        }
        if (metadata.isCounter()) {
            del.timestamp = FBUtilities.timestampMicros();
        } else if (!del.isSetTimestamp()) {
            throw new InvalidRequestException("Deletion timestamp is not optional for non commutative table " + metadata.cfName);
        }
    }

    public static void validateSlicePredicate(CFMetaData metadata, ByteBuffer scName, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.column_names == null && predicate.slice_range == null) {
            throw new InvalidRequestException("A SlicePredicate must be given a list of Columns, a SliceRange, or both");
        }
        if (predicate.slice_range != null) {
            ThriftValidation.validateRange(metadata, new ColumnParent(metadata.cfName).setSuper_column(scName), predicate.slice_range);
        }
        if (predicate.column_names != null) {
            ThriftValidation.validateColumnNames(metadata, scName, (Iterable<ByteBuffer>)predicate.column_names);
        }
    }

    public static void validateColumnData(CFMetaData metadata, ByteBuffer scName, Column column) throws InvalidRequestException {
        ThriftValidation.validateTtl(column);
        if (!column.isSetValue()) {
            throw new InvalidRequestException("Column value is required");
        }
        if (!column.isSetTimestamp()) {
            throw new InvalidRequestException("Column timestamp is required");
        }
        try {
            LegacyLayout.LegacyCellName cn = LegacyLayout.decodeCellName(metadata, scName, column.name);
            cn.column.validateCellValue(column.value);
        }
        catch (UnknownColumnException e) {
            throw new InvalidRequestException(e.getMessage());
        }
        catch (MarshalException me) {
            if (logger.isTraceEnabled()) {
                logger.trace("rejecting invalid value {}", (Object)ByteBufferUtil.bytesToHex(ThriftValidation.summarize(column.value)));
            }
            throw new InvalidRequestException(String.format("(%s) [%s][%s][%s] failed validation", me.getMessage(), metadata.ksName, metadata.cfName, ThriftValidation.getThriftColumnNameComparator(metadata, scName).getString(column.name)));
        }
    }

    private static ByteBuffer summarize(ByteBuffer buffer) {
        int MAX = Short.MAX_VALUE;
        if (buffer.remaining() <= MAX) {
            return buffer;
        }
        return (ByteBuffer)buffer.slice().limit(buffer.position() + MAX);
    }

    public static void validatePredicate(CFMetaData metadata, ColumnParent column_parent, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.column_names == null && predicate.slice_range == null) {
            throw new InvalidRequestException("predicate column_names and slice_range may not both be null");
        }
        if (predicate.column_names != null && predicate.slice_range != null) {
            throw new InvalidRequestException("predicate column_names and slice_range may not both be present");
        }
        if (predicate.getSlice_range() != null) {
            ThriftValidation.validateRange(metadata, column_parent, predicate.slice_range);
        } else {
            ThriftValidation.validateColumnNames(metadata, column_parent, (Iterable<ByteBuffer>)predicate.column_names);
        }
    }

    public static void validateKeyRange(CFMetaData metadata, ByteBuffer superColumn, KeyRange range) throws InvalidRequestException {
        if (range.start_key == null == (range.start_token == null) || range.end_key == null == (range.end_token == null)) {
            throw new InvalidRequestException("exactly one each of {start key, start token} and {end key, end token} must be specified");
        }
        if (range.start_token != null && range.end_key != null) {
            throw new InvalidRequestException("start token + end key is not a supported key range");
        }
        IPartitioner p = metadata.partitioner;
        if (range.start_key != null && range.end_key != null) {
            Token endToken;
            Token startToken = p.getToken(range.start_key);
            if (startToken.compareTo(endToken = p.getToken(range.end_key)) > 0 && !endToken.isMinimum()) {
                if (p.preservesOrder()) {
                    throw new InvalidRequestException("start key must sort before (or equal to) finish key in your partitioner!");
                }
                throw new InvalidRequestException("start key's token sorts after end key's token.  this is not allowed; you probably should not specify end key at all except with an ordered partitioner");
            }
        } else if (range.start_key != null && range.end_token != null) {
            Token.KeyBound stop = p.getTokenFactory().fromString(range.end_token).maxKeyBound();
            if (PartitionPosition.ForKey.get(range.start_key, p).compareTo(stop) > 0 && !stop.isMinimum()) {
                throw new InvalidRequestException("Start key's token sorts after end token");
            }
        }
        ThriftValidation.validateFilterClauses(metadata, range.row_filter);
        if (!ThriftValidation.isEmpty(range.row_filter) && superColumn != null) {
            throw new InvalidRequestException("super columns are not supported for indexing");
        }
        if (range.count <= 0) {
            throw new InvalidRequestException("maxRows must be positive");
        }
    }

    private static boolean isEmpty(List<IndexExpression> clause) {
        return clause == null || clause.isEmpty();
    }

    public static void validateIndexClauses(CFMetaData metadata, IndexClause index_clause) throws InvalidRequestException {
        if (index_clause.expressions.isEmpty()) {
            throw new InvalidRequestException("index clause list may not be empty");
        }
        if (!ThriftValidation.validateFilterClauses(metadata, index_clause.expressions)) {
            throw new InvalidRequestException("No indexed columns present in index clause with operator EQ");
        }
    }

    public static boolean validateFilterClauses(CFMetaData metadata, List<IndexExpression> index_clause) throws InvalidRequestException {
        if (ThriftValidation.isEmpty(index_clause)) {
            return false;
        }
        SecondaryIndexManager idxManager = Keyspace.open((String)metadata.ksName).getColumnFamilyStore((String)metadata.cfName).indexManager;
        AbstractType<?> nameValidator = ThriftValidation.getThriftColumnNameComparator(metadata, null);
        boolean isIndexed = false;
        for (IndexExpression expression : index_clause) {
            try {
                nameValidator.validate(expression.column_name);
            }
            catch (MarshalException me) {
                throw new InvalidRequestException(String.format("[%s]=[%s] failed name validation (%s)", ByteBufferUtil.bytesToHex(expression.column_name), ByteBufferUtil.bytesToHex(expression.value), me.getMessage()));
            }
            if (expression.value.remaining() > 65535) {
                throw new InvalidRequestException("Index expression values may not be larger than 64K");
            }
            ColumnDefinition def = metadata.getColumnDefinition(expression.column_name);
            if (def == null) {
                if (!metadata.isCompactTable()) {
                    throw new InvalidRequestException(String.format("Unknown column %s", nameValidator.getString(expression.column_name)));
                }
                def = metadata.compactValueColumn();
            }
            try {
                def.type.validate(expression.value);
            }
            catch (MarshalException me) {
                throw new InvalidRequestException(String.format("[%s]=[%s] failed value validation (%s)", ByteBufferUtil.bytesToHex(expression.column_name), ByteBufferUtil.bytesToHex(expression.value), me.getMessage()));
            }
            for (Index index : idxManager.listIndexes()) {
                isIndexed |= index.supportsExpression(def, Operator.valueOf(expression.op.name()));
            }
        }
        return isIndexed;
    }

    public static void validateKeyspaceNotYetExisting(String newKsName) throws InvalidRequestException {
        for (String ksName : Schema.instance.getKeyspaces()) {
            if (!ksName.equalsIgnoreCase(newKsName)) continue;
            throw new InvalidRequestException(String.format("Keyspace names must be case-insensitively unique (\"%s\" conflicts with \"%s\")", newKsName, ksName));
        }
    }

    public static void validateKeyspaceNotSystem(String modifiedKeyspace) throws InvalidRequestException {
        if (Schema.isSystemKeyspace(modifiedKeyspace)) {
            throw new InvalidRequestException(String.format("%s keyspace is not user-modifiable", modifiedKeyspace));
        }
    }
}

