/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;
import java.util.zip.CRC32;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.ParameterizedClass;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.AbstractCommitLogSegmentManager;
import org.apache.cassandra.db.commitlog.AbstractCommitLogService;
import org.apache.cassandra.db.commitlog.BatchCommitLogService;
import org.apache.cassandra.db.commitlog.CommitLogArchiver;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogMBean;
import org.apache.cassandra.db.commitlog.CommitLogPosition;
import org.apache.cassandra.db.commitlog.CommitLogReplayer;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.db.commitlog.CommitLogSegmentManagerCDC;
import org.apache.cassandra.db.commitlog.CommitLogSegmentManagerStandard;
import org.apache.cassandra.db.commitlog.PeriodicCommitLogService;
import org.apache.cassandra.exceptions.WriteTimeoutException;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.util.DataOutputBufferFixed;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.metrics.CommitLogMetrics;
import org.apache.cassandra.schema.CompressionParams;
import org.apache.cassandra.security.EncryptionContext;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLog
implements CommitLogMBean {
    private static final Logger logger = LoggerFactory.getLogger(CommitLog.class);
    public static final CommitLog instance = CommitLog.construct();
    final long MAX_MUTATION_SIZE = DatabaseDescriptor.getMaxMutationSize();
    public final AbstractCommitLogSegmentManager segmentManager;
    public final CommitLogArchiver archiver;
    final CommitLogMetrics metrics;
    final AbstractCommitLogService executor;
    volatile Configuration configuration = new Configuration(DatabaseDescriptor.getCommitLogCompression(), DatabaseDescriptor.getEncryptionContext());

    private static CommitLog construct() {
        CommitLog log = new CommitLog(CommitLogArchiver.construct());
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(log, new ObjectName("org.apache.cassandra.db:type=Commitlog"));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        return log.start();
    }

    @VisibleForTesting
    CommitLog(CommitLogArchiver archiver) {
        DatabaseDescriptor.createAllDirectories();
        this.archiver = archiver;
        this.metrics = new CommitLogMetrics();
        this.executor = DatabaseDescriptor.getCommitLogSync() == Config.CommitLogSync.batch ? new BatchCommitLogService(this) : new PeriodicCommitLogService(this);
        this.segmentManager = DatabaseDescriptor.isCDCEnabled() ? new CommitLogSegmentManagerCDC(this, DatabaseDescriptor.getCommitLogLocation()) : new CommitLogSegmentManagerStandard(this, DatabaseDescriptor.getCommitLogLocation());
        this.metrics.attach(this.executor, this.segmentManager);
    }

    CommitLog start() {
        this.executor.start();
        this.segmentManager.start();
        return this;
    }

    public int recoverSegmentsOnDisk() throws IOException {
        if (this.segmentManager.createReserveSegments) {
            return 0;
        }
        FilenameFilter unmanagedFilesFilter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return CommitLogDescriptor.isValid(name) && CommitLogSegment.shouldReplay(name);
            }
        };
        for (File file : new File(this.segmentManager.storageDirectory).listFiles(unmanagedFilesFilter)) {
            this.archiver.maybeArchive(file.getPath(), file.getName());
            this.archiver.maybeWaitForArchiving(file.getName());
        }
        assert (this.archiver.archivePending.isEmpty()) : "Not all commit log archive tasks were completed before restore";
        this.archiver.maybeRestoreArchive();
        Object[] files = new File(this.segmentManager.storageDirectory).listFiles(unmanagedFilesFilter);
        int replayed = 0;
        if (files.length == 0) {
            logger.info("No commitlog files found; skipping replay");
        } else {
            Arrays.sort(files, new CommitLogSegment.CommitLogSegmentFileComparator());
            logger.info("Replaying {}", (Object)StringUtils.join((Object[])files, (String)", "));
            replayed = this.recoverFiles((File[])files);
            logger.info("Log replay complete, {} replayed mutations", (Object)replayed);
            for (Object f : files) {
                this.segmentManager.handleReplayedSegment((File)f);
            }
        }
        this.segmentManager.enableReserveSegmentCreation();
        return replayed;
    }

    public int recoverFiles(File ... clogs) throws IOException {
        CommitLogReplayer replayer = CommitLogReplayer.construct(this);
        replayer.replayFiles(clogs);
        return replayer.blockForWrites();
    }

    public void recoverPath(String path) throws IOException {
        CommitLogReplayer replayer = CommitLogReplayer.construct(this);
        replayer.replayPath(new File(path), false);
        replayer.blockForWrites();
    }

    @Override
    public void recover(String path) throws IOException {
        this.recoverPath(path);
    }

    public CommitLogPosition getCurrentPosition() {
        return this.segmentManager.getCurrentPosition();
    }

    public void forceRecycleAllSegments(Iterable<UUID> droppedCfs) {
        this.segmentManager.forceRecycleAll(droppedCfs);
    }

    public void forceRecycleAllSegments() {
        this.segmentManager.forceRecycleAll(Collections.emptyList());
    }

    public void sync(boolean syncAllSegments) throws IOException {
        this.segmentManager.sync(syncAllSegments);
    }

    public void requestExtraSync() {
        this.executor.requestExtraSync();
    }

    public CommitLogPosition add(Mutation mutation) throws WriteTimeoutException {
        assert (mutation != null);
        int size = (int)Mutation.serializer.serializedSize(mutation, 10);
        int totalSize = size + 12;
        if ((long)totalSize > this.MAX_MUTATION_SIZE) {
            throw new IllegalArgumentException(String.format("Mutation of %s is too large for the maximum size of %s", FBUtilities.prettyPrintMemory(totalSize), FBUtilities.prettyPrintMemory(this.MAX_MUTATION_SIZE)));
        }
        CommitLogSegment.Allocation alloc = this.segmentManager.allocate(mutation, totalSize);
        CRC32 checksum = new CRC32();
        ByteBuffer buffer = alloc.getBuffer();
        try (DataOutputBufferFixed dos = new DataOutputBufferFixed(buffer);){
            dos.writeInt(size);
            FBUtilities.updateChecksumInt(checksum, size);
            buffer.putInt((int)checksum.getValue());
            Mutation.serializer.serialize(mutation, (DataOutputPlus)dos, 10);
            FBUtilities.updateChecksum(checksum, buffer, buffer.position() - size, size);
            buffer.putInt((int)checksum.getValue());
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, alloc.getSegment().getPath());
        }
        finally {
            alloc.markWritten();
        }
        this.executor.finishWriteFor(alloc);
        return alloc.getCommitLogPosition();
    }

    public void discardCompletedSegments(UUID cfId, CommitLogPosition lowerBound, CommitLogPosition upperBound) {
        logger.trace("discard completed log segments for {}-{}, table {}", new Object[]{lowerBound, upperBound, cfId});
        Iterator<CommitLogSegment> iter = this.segmentManager.getActiveSegments().iterator();
        while (iter.hasNext()) {
            CommitLogSegment segment = iter.next();
            segment.markClean(cfId, lowerBound, upperBound);
            if (segment.isUnused()) {
                logger.trace("Commit log segment {} is unused", (Object)segment);
                this.segmentManager.recycleSegment(segment);
            } else if (logger.isTraceEnabled()) {
                logger.trace("Not safe to delete{} commit log segment {}; dirty is {}", new Object[]{iter.hasNext() ? "" : " active", segment, segment.dirtyString()});
            }
            if (!segment.contains(upperBound)) continue;
            break;
        }
    }

    @Override
    public String getArchiveCommand() {
        return this.archiver.archiveCommand;
    }

    @Override
    public String getRestoreCommand() {
        return this.archiver.restoreCommand;
    }

    @Override
    public String getRestoreDirectories() {
        return this.archiver.restoreDirectories;
    }

    @Override
    public long getRestorePointInTime() {
        return this.archiver.restorePointInTime;
    }

    @Override
    public String getRestorePrecision() {
        return this.archiver.precision.toString();
    }

    @Override
    public List<String> getActiveSegmentNames() {
        ArrayList<String> segmentNames = new ArrayList<String>();
        for (CommitLogSegment seg : this.segmentManager.getActiveSegments()) {
            segmentNames.add(seg.getName());
        }
        return segmentNames;
    }

    @Override
    public List<String> getArchivingSegmentNames() {
        return new ArrayList<String>(this.archiver.archivePending.keySet());
    }

    @Override
    public long getActiveContentSize() {
        long size = 0L;
        for (CommitLogSegment seg : this.segmentManager.getActiveSegments()) {
            size += seg.contentSize();
        }
        return size;
    }

    @Override
    public long getActiveOnDiskSize() {
        return this.segmentManager.onDiskSize();
    }

    @Override
    public Map<String, Double> getActiveSegmentCompressionRatios() {
        TreeMap<String, Double> segmentRatios = new TreeMap<String, Double>();
        for (CommitLogSegment seg : this.segmentManager.getActiveSegments()) {
            segmentRatios.put(seg.getName(), 1.0 * (double)seg.onDiskSize() / (double)seg.contentSize());
        }
        return segmentRatios;
    }

    public void shutdownBlocking() throws InterruptedException {
        this.executor.shutdown();
        this.executor.awaitTermination();
        this.segmentManager.shutdown();
        this.segmentManager.awaitTermination();
    }

    public int resetUnsafe(boolean deleteSegments) throws IOException {
        this.stopUnsafe(deleteSegments);
        this.resetConfiguration();
        return this.restartUnsafe();
    }

    public void resetConfiguration() {
        this.configuration = new Configuration(DatabaseDescriptor.getCommitLogCompression(), DatabaseDescriptor.getEncryptionContext());
    }

    public void stopUnsafe(boolean deleteSegments) {
        this.executor.shutdown();
        try {
            this.executor.awaitTermination();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        this.segmentManager.stopUnsafe(deleteSegments);
        CommitLogSegment.resetReplayLimit();
        if (DatabaseDescriptor.isCDCEnabled() && deleteSegments) {
            for (File f : new File(DatabaseDescriptor.getCDCLogLocation()).listFiles()) {
                FileUtils.deleteWithConfirm(f);
            }
        }
    }

    public int restartUnsafe() throws IOException {
        this.segmentManager.start();
        this.executor.restartUnsafe();
        try {
            return this.recoverSegmentsOnDisk();
        }
        catch (FSWriteError e) {
            return 0;
        }
    }

    @VisibleForTesting
    public static boolean handleCommitError(String message, Throwable t) {
        JVMStabilityInspector.inspectCommitLogThrowable(t);
        switch (DatabaseDescriptor.getCommitFailurePolicy()) {
            case die: 
            case stop: {
                StorageService.instance.stopTransports();
            }
            case stop_commit: {
                logger.error(String.format("%s. Commit disk failure policy is %s; terminating thread", new Object[]{message, DatabaseDescriptor.getCommitFailurePolicy()}), t);
                return false;
            }
            case ignore: {
                logger.error(message, t);
                return true;
            }
        }
        throw new AssertionError((Object)DatabaseDescriptor.getCommitFailurePolicy());
    }

    public static final class Configuration {
        private final ParameterizedClass compressorClass;
        private final ICompressor compressor;
        private EncryptionContext encryptionContext;

        public Configuration(ParameterizedClass compressorClass, EncryptionContext encryptionContext) {
            this.compressorClass = compressorClass;
            this.compressor = compressorClass != null ? CompressionParams.createCompressor(compressorClass) : null;
            this.encryptionContext = encryptionContext;
        }

        public boolean useCompression() {
            return this.compressor != null;
        }

        public boolean useEncryption() {
            return this.encryptionContext.isEnabled();
        }

        public ICompressor getCompressor() {
            return this.compressor;
        }

        public ParameterizedClass getCompressorClass() {
            return this.compressorClass;
        }

        public String getCompressorName() {
            return this.useCompression() ? this.compressor.getClass().getSimpleName() : "none";
        }

        public EncryptionContext getEncryptionContext() {
            return this.encryptionContext;
        }
    }
}

