/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Memtable;
import org.apache.cassandra.db.PartitionPosition;
import org.apache.cassandra.db.lifecycle.SSTableIntervalTree;
import org.apache.cassandra.db.lifecycle.SSTableSet;
import org.apache.cassandra.db.partitions.Partition;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.utils.AlwaysPresentFilter;
import org.apache.cassandra.utils.OverlapIterator;
import org.apache.cassandra.utils.concurrent.Refs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionController
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(CompactionController.class);
    static final boolean NEVER_PURGE_TOMBSTONES = Boolean.getBoolean("cassandra.never_purge_tombstones");
    public final ColumnFamilyStore cfs;
    private final boolean compactingRepaired;
    private Refs<SSTableReader> overlappingSSTables;
    private OverlapIterator<PartitionPosition, SSTableReader> overlapIterator;
    private final Iterable<SSTableReader> compacting;
    public final int gcBefore;

    protected CompactionController(ColumnFamilyStore cfs, int maxValue) {
        this(cfs, null, maxValue);
    }

    public CompactionController(ColumnFamilyStore cfs, Set<SSTableReader> compacting, int gcBefore) {
        assert (cfs != null);
        this.cfs = cfs;
        this.gcBefore = gcBefore;
        this.compacting = compacting;
        this.compactingRepaired = compacting != null && compacting.stream().allMatch(SSTableReader::isRepaired);
        this.refreshOverlaps();
        if (NEVER_PURGE_TOMBSTONES) {
            logger.warn("You are running with -Dcassandra.never_purge_tombstones=true, this is dangerous!");
        }
    }

    public void maybeRefreshOverlaps() {
        if (NEVER_PURGE_TOMBSTONES) {
            logger.debug("not refreshing overlaps - running with -Dcassandra.never_purge_tombstones=true");
            return;
        }
        for (SSTableReader reader : this.overlappingSSTables) {
            if (!reader.isMarkedCompacted()) continue;
            this.refreshOverlaps();
            return;
        }
    }

    private void refreshOverlaps() {
        if (NEVER_PURGE_TOMBSTONES) {
            return;
        }
        if (this.overlappingSSTables != null) {
            this.overlappingSSTables.release();
        }
        this.overlappingSSTables = this.compacting == null ? Refs.tryRef(Collections.emptyList()) : this.cfs.getAndReferenceOverlappingSSTables(SSTableSet.LIVE, this.compacting);
        this.overlapIterator = new OverlapIterator(SSTableIntervalTree.buildIntervals(this.overlappingSSTables));
    }

    public Set<SSTableReader> getFullyExpiredSSTables() {
        return CompactionController.getFullyExpiredSSTables(this.cfs, this.compacting, this.overlappingSSTables, this.gcBefore);
    }

    public static Set<SSTableReader> getFullyExpiredSSTables(ColumnFamilyStore cfStore, Iterable<SSTableReader> compacting, Iterable<SSTableReader> overlapping, int gcBefore) {
        logger.trace("Checking droppable sstables in {}", (Object)cfStore);
        if (compacting == null || NEVER_PURGE_TOMBSTONES) {
            return Collections.emptySet();
        }
        if (cfStore.getCompactionStrategyManager().onlyPurgeRepairedTombstones() && !Iterables.all(compacting, SSTableReader::isRepaired)) {
            return Collections.emptySet();
        }
        ArrayList<SSTableReader> candidates = new ArrayList<SSTableReader>();
        long minTimestamp = Long.MAX_VALUE;
        for (SSTableReader sstable : overlapping) {
            if (sstable.getSSTableMetadata().maxLocalDeletionTime < gcBefore) continue;
            minTimestamp = Math.min(minTimestamp, sstable.getMinTimestamp());
        }
        for (SSTableReader candidate : compacting) {
            if (candidate.getSSTableMetadata().maxLocalDeletionTime < gcBefore) {
                candidates.add(candidate);
                continue;
            }
            minTimestamp = Math.min(minTimestamp, candidate.getMinTimestamp());
        }
        for (Memtable memtable : cfStore.getTracker().getView().getAllMemtables()) {
            minTimestamp = Math.min(minTimestamp, memtable.getMinTimestamp());
        }
        Iterator iterator = candidates.iterator();
        while (iterator.hasNext()) {
            SSTableReader candidate;
            candidate = (SSTableReader)iterator.next();
            if (candidate.getMaxTimestamp() >= minTimestamp) {
                iterator.remove();
                continue;
            }
            logger.trace("Dropping expired SSTable {} (maxLocalDeletionTime={}, gcBefore={})", new Object[]{candidate, candidate.getSSTableMetadata().maxLocalDeletionTime, gcBefore});
        }
        return new HashSet<SSTableReader>(candidates);
    }

    public String getKeyspace() {
        return this.cfs.keyspace.getName();
    }

    public String getColumnFamily() {
        return this.cfs.name;
    }

    public long maxPurgeableTimestamp(DecoratedKey key) {
        if (!this.compactingRepaired() || NEVER_PURGE_TOMBSTONES) {
            return Long.MIN_VALUE;
        }
        long min = Long.MAX_VALUE;
        this.overlapIterator.update(key);
        for (SSTableReader sstable : this.overlapIterator.overlaps()) {
            if ((!(sstable.getBloomFilter() instanceof AlwaysPresentFilter) || sstable.getPosition(key, SSTableReader.Operator.EQ, false) == null) && !sstable.getBloomFilter().isPresent(key)) continue;
            min = Math.min(min, sstable.getMinTimestamp());
        }
        for (Memtable memtable : this.cfs.getTracker().getView().getAllMemtables()) {
            Partition partition = memtable.getPartition(key);
            if (partition == null) continue;
            min = Math.min(min, partition.stats().minTimestamp);
        }
        return min;
    }

    @Override
    public void close() {
        if (this.overlappingSSTables != null) {
            this.overlappingSSTables.release();
        }
    }

    public boolean compactingRepaired() {
        return !this.cfs.getCompactionStrategyManager().onlyPurgeRepairedTombstones() || this.compactingRepaired;
    }
}

