/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.collect.AbstractIterator;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.primitives.Doubles;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionLogger;
import org.apache.cassandra.db.compaction.LeveledCompactionTask;
import org.apache.cassandra.db.compaction.LeveledManifest;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategyOptions;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.db.rows.UnfilteredRowIterator;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.ISSTableScanner;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.node.JsonNodeFactory;
import org.codehaus.jackson.node.ObjectNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeveledCompactionStrategy
extends AbstractCompactionStrategy {
    private static final Logger logger = LoggerFactory.getLogger(LeveledCompactionStrategy.class);
    private static final String SSTABLE_SIZE_OPTION = "sstable_size_in_mb";
    private static final boolean tolerateSstableSize = Boolean.getBoolean("cassandra.tolerate_sstable_size");
    private static final String LEVEL_FANOUT_SIZE_OPTION = "fanout_size";
    public static final int DEFAULT_LEVEL_FANOUT_SIZE = 10;
    @VisibleForTesting
    final LeveledManifest manifest;
    private final int maxSSTableSizeInMB;
    private final int levelFanoutSize;

    public LeveledCompactionStrategy(ColumnFamilyStore cfs, Map<String, String> options) {
        super(cfs, options);
        int configuredMaxSSTableSize = 160;
        int configuredLevelFanoutSize = 10;
        SizeTieredCompactionStrategyOptions localOptions = new SizeTieredCompactionStrategyOptions(options);
        if (options != null) {
            if (options.containsKey(SSTABLE_SIZE_OPTION)) {
                configuredMaxSSTableSize = Integer.parseInt(options.get(SSTABLE_SIZE_OPTION));
                if (!tolerateSstableSize) {
                    if (configuredMaxSSTableSize >= 1000) {
                        logger.warn("Max sstable size of {}MB is configured for {}.{}; having a unit of compaction this large is probably a bad idea", new Object[]{configuredMaxSSTableSize, cfs.name, cfs.getColumnFamilyName()});
                    }
                    if (configuredMaxSSTableSize < 50) {
                        logger.warn("Max sstable size of {}MB is configured for {}.{}.  Testing done for CASSANDRA-5727 indicates that performance improves up to 160MB", new Object[]{configuredMaxSSTableSize, cfs.name, cfs.getColumnFamilyName()});
                    }
                }
            }
            if (options.containsKey(LEVEL_FANOUT_SIZE_OPTION)) {
                configuredLevelFanoutSize = Integer.parseInt(options.get(LEVEL_FANOUT_SIZE_OPTION));
            }
        }
        this.maxSSTableSizeInMB = configuredMaxSSTableSize;
        this.levelFanoutSize = configuredLevelFanoutSize;
        this.manifest = new LeveledManifest(cfs, this.maxSSTableSizeInMB, this.levelFanoutSize, localOptions);
        logger.trace("Created {}", (Object)this.manifest);
    }

    public int getLevelSize(int i) {
        return this.manifest.getLevelSize(i);
    }

    public int[] getAllLevelSize() {
        return this.manifest.getAllLevelSize();
    }

    @Override
    public void startup() {
        this.manifest.calculateLastCompactedKeys();
        super.startup();
    }

    @Override
    public AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        OperationType op;
        LeveledManifest.CompactionCandidate candidate;
        LifecycleTransaction txn;
        do {
            if ((candidate = this.manifest.getCompactionCandidates()) == null) {
                SSTableReader sstable = this.findDroppableSSTable(gcBefore);
                if (sstable == null) {
                    logger.trace("No compaction necessary for {}", (Object)this);
                    return null;
                }
                candidate = new LeveledManifest.CompactionCandidate(Collections.singleton(sstable), sstable.getSSTableLevel(), this.getMaxSSTableBytes());
                op = OperationType.TOMBSTONE_COMPACTION;
                continue;
            }
            op = OperationType.COMPACTION;
        } while ((txn = this.cfs.getTracker().tryModify(candidate.sstables, OperationType.COMPACTION)) == null);
        LeveledCompactionTask newTask = new LeveledCompactionTask(this.cfs, txn, candidate.level, gcBefore, candidate.maxSSTableBytes, false);
        newTask.setCompactionType(op);
        return newTask;
    }

    @Override
    public synchronized Collection<AbstractCompactionTask> getMaximalTask(int gcBefore, boolean splitOutput) {
        Iterable<SSTableReader> sstables = this.manifest.getAllSSTables();
        Iterable<SSTableReader> filteredSSTables = LeveledCompactionStrategy.filterSuspectSSTables(sstables);
        if (Iterables.isEmpty(sstables)) {
            return null;
        }
        LifecycleTransaction txn = this.cfs.getTracker().tryModify(filteredSSTables, OperationType.COMPACTION);
        if (txn == null) {
            return null;
        }
        return Arrays.asList(new LeveledCompactionTask(this.cfs, txn, 0, gcBefore, this.getMaxSSTableBytes(), true));
    }

    @Override
    public AbstractCompactionTask getUserDefinedTask(Collection<SSTableReader> sstables, int gcBefore) {
        if (sstables.isEmpty()) {
            return null;
        }
        LifecycleTransaction transaction = this.cfs.getTracker().tryModify(sstables, OperationType.COMPACTION);
        if (transaction == null) {
            logger.trace("Unable to mark {} for compaction; probably a background compaction got to it first.  You can disable background compactions temporarily if this is a problem", sstables);
            return null;
        }
        int level = sstables.size() > 1 ? 0 : sstables.iterator().next().getSSTableLevel();
        return new LeveledCompactionTask(this.cfs, transaction, level, gcBefore, level == 0 ? Long.MAX_VALUE : this.getMaxSSTableBytes(), false);
    }

    @Override
    public AbstractCompactionTask getCompactionTask(LifecycleTransaction txn, int gcBefore, long maxSSTableBytes) {
        assert (txn.originals().size() > 0);
        int level = -1;
        for (SSTableReader sstable : txn.originals()) {
            if (level == -1) {
                level = sstable.getSSTableLevel();
            }
            if (level == sstable.getSSTableLevel()) continue;
            level = 0;
        }
        return new LeveledCompactionTask(this.cfs, txn, level, gcBefore, maxSSTableBytes, false);
    }

    @Override
    public Collection<Collection<SSTableReader>> groupSSTablesForAntiCompaction(Collection<SSTableReader> ssTablesToGroup) {
        int groupSize = 2;
        HashMap<Integer, ArrayList<SSTableReader>> sstablesByLevel = new HashMap<Integer, ArrayList<SSTableReader>>();
        for (SSTableReader sstable : ssTablesToGroup) {
            Integer level = sstable.getSSTableLevel();
            ArrayList<SSTableReader> sstablesForLevel = (ArrayList<SSTableReader>)sstablesByLevel.get(level);
            if (sstablesForLevel == null) {
                sstablesForLevel = new ArrayList<SSTableReader>();
                sstablesByLevel.put(level, sstablesForLevel);
            }
            sstablesForLevel.add(sstable);
        }
        ArrayList<Collection<SSTableReader>> groupedSSTables = new ArrayList<Collection<SSTableReader>>();
        for (Collection levelOfSSTables : sstablesByLevel.values()) {
            ArrayList<SSTableReader> currGroup = new ArrayList<SSTableReader>();
            for (SSTableReader sstable : levelOfSSTables) {
                currGroup.add(sstable);
                if (currGroup.size() != groupSize) continue;
                groupedSSTables.add(currGroup);
                currGroup = new ArrayList();
            }
            if (currGroup.size() == 0) continue;
            groupedSSTables.add(currGroup);
        }
        return groupedSSTables;
    }

    @Override
    public int getEstimatedRemainingTasks() {
        int n = this.manifest.getEstimatedTasks();
        this.cfs.getCompactionStrategyManager().compactionLogger.pending(this, n);
        return n;
    }

    @Override
    public long getMaxSSTableBytes() {
        return (long)this.maxSSTableSizeInMB * 1024L * 1024L;
    }

    public int getLevelFanoutSize() {
        return this.levelFanoutSize;
    }

    @Override
    public AbstractCompactionStrategy.ScannerList getScanners(Collection<SSTableReader> sstables, Collection<Range<Token>> ranges) {
        Set<SSTableReader>[] sstablesPerLevel = this.manifest.getSStablesPerLevelSnapshot();
        ArrayListMultimap byLevel = ArrayListMultimap.create();
        for (SSTableReader sSTableReader : sstables) {
            int level = sSTableReader.getSSTableLevel();
            if (level >= sstablesPerLevel.length || !sstablesPerLevel[level].contains(sSTableReader)) {
                logger.warn("Live sstable {} from level {} is not on corresponding level in the leveled manifest. This is not a problem per se, but may indicate an orphaned sstable due to a failed compaction not cleaned up properly.", (Object)sSTableReader.getFilename(), (Object)level);
                level = -1;
            }
            byLevel.get((Object)level).add(sSTableReader);
        }
        ArrayList<ISSTableScanner> scanners = new ArrayList<ISSTableScanner>(sstables.size());
        try {
            for (Integer level : byLevel.keySet()) {
                if (level <= 0) {
                    for (SSTableReader sstable : byLevel.get((Object)level)) {
                        scanners.add(sstable.getScanner(ranges, null));
                    }
                    continue;
                }
                Collection<SSTableReader> intersecting = LeveledScanner.intersecting(byLevel.get((Object)level), ranges);
                if (intersecting.isEmpty()) continue;
                LeveledScanner scanner = new LeveledScanner(intersecting, ranges);
                scanners.add(scanner);
            }
        }
        catch (Throwable throwable) {
            try {
                new AbstractCompactionStrategy.ScannerList(scanners).close();
            }
            catch (Throwable t2) {
                throwable.addSuppressed(t2);
            }
            throw throwable;
        }
        return new AbstractCompactionStrategy.ScannerList(scanners);
    }

    @Override
    public void replaceSSTables(Collection<SSTableReader> removed, Collection<SSTableReader> added) {
        this.manifest.replace(removed, added);
    }

    @Override
    public void addSSTable(SSTableReader added) {
        this.manifest.add(added);
    }

    @Override
    public void removeSSTable(SSTableReader sstable) {
        this.manifest.remove(sstable);
    }

    public String toString() {
        return String.format("LCS@%d(%s)", this.hashCode(), this.cfs.name);
    }

    private SSTableReader findDroppableSSTable(final int gcBefore) {
        for (int i = this.manifest.getLevelCount(); i >= 0; --i) {
            SSTableReader sstable;
            SortedSet<SSTableReader> sstables = this.manifest.getLevelSorted(i, new Comparator<SSTableReader>(){

                @Override
                public int compare(SSTableReader o1, SSTableReader o2) {
                    double r1 = o1.getEstimatedDroppableTombstoneRatio(gcBefore);
                    double r2 = o2.getEstimatedDroppableTombstoneRatio(gcBefore);
                    return -1 * Doubles.compare((double)r1, (double)r2);
                }
            });
            if (sstables.isEmpty()) continue;
            Set<SSTableReader> compacting = this.cfs.getTracker().getCompacting();
            Iterator iterator = sstables.iterator();
            while (iterator.hasNext() && !((sstable = (SSTableReader)iterator.next()).getEstimatedDroppableTombstoneRatio(gcBefore) <= (double)this.tombstoneThreshold)) {
                if (compacting.contains(sstable) || sstable.isMarkedSuspect() || !this.worthDroppingTombstones(sstable, gcBefore)) continue;
                return sstable;
            }
        }
        return null;
    }

    @Override
    public CompactionLogger.Strategy strategyLogger() {
        return new CompactionLogger.Strategy(){

            @Override
            public JsonNode sstable(SSTableReader sstable) {
                ObjectNode node = JsonNodeFactory.instance.objectNode();
                node.put("level", sstable.getSSTableLevel());
                node.put("min_token", sstable.first.getToken().toString());
                node.put("max_token", sstable.last.getToken().toString());
                return node;
            }

            @Override
            public JsonNode options() {
                return null;
            }
        };
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        Map<String, String> uncheckedOptions = AbstractCompactionStrategy.validateOptions(options);
        String size = options.containsKey(SSTABLE_SIZE_OPTION) ? options.get(SSTABLE_SIZE_OPTION) : "1";
        try {
            int ssSize = Integer.parseInt(size);
            if (ssSize < 1) {
                throw new ConfigurationException(String.format("%s must be larger than 0, but was %s", SSTABLE_SIZE_OPTION, ssSize));
            }
        }
        catch (NumberFormatException ex) {
            throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", size, SSTABLE_SIZE_OPTION), ex);
        }
        uncheckedOptions.remove(SSTABLE_SIZE_OPTION);
        String levelFanoutSize = options.containsKey(LEVEL_FANOUT_SIZE_OPTION) ? options.get(LEVEL_FANOUT_SIZE_OPTION) : String.valueOf(10);
        try {
            int fanoutSize = Integer.parseInt(levelFanoutSize);
            if (fanoutSize < 1) {
                throw new ConfigurationException(String.format("%s must be larger than 0, but was %s", LEVEL_FANOUT_SIZE_OPTION, fanoutSize));
            }
        }
        catch (NumberFormatException ex) {
            throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", size, LEVEL_FANOUT_SIZE_OPTION), ex);
        }
        uncheckedOptions.remove(LEVEL_FANOUT_SIZE_OPTION);
        uncheckedOptions = SizeTieredCompactionStrategyOptions.validateOptions(options, uncheckedOptions);
        return uncheckedOptions;
    }

    private static class LeveledScanner
    extends AbstractIterator<UnfilteredRowIterator>
    implements ISSTableScanner {
        private final Collection<Range<Token>> ranges;
        private final List<SSTableReader> sstables;
        private final Iterator<SSTableReader> sstableIterator;
        private final long totalLength;
        private final long compressedLength;
        private ISSTableScanner currentScanner;
        private long positionOffset;
        private long totalBytesScanned = 0L;

        public LeveledScanner(Collection<SSTableReader> sstables, Collection<Range<Token>> ranges) {
            this.ranges = ranges;
            this.sstables = new ArrayList<SSTableReader>(sstables.size());
            long length = 0L;
            long cLength = 0L;
            for (SSTableReader sstable : sstables) {
                this.sstables.add(sstable);
                long estimatedKeys = sstable.estimatedKeys();
                double estKeysInRangeRatio = 1.0;
                if (estimatedKeys > 0L && ranges != null) {
                    estKeysInRangeRatio = (double)sstable.estimatedKeysForRanges(ranges) / (double)estimatedKeys;
                }
                length = (long)((double)length + (double)sstable.uncompressedLength() * estKeysInRangeRatio);
                cLength = (long)((double)cLength + (double)sstable.onDiskLength() * estKeysInRangeRatio);
            }
            this.totalLength = length;
            this.compressedLength = cLength;
            Collections.sort(this.sstables, SSTableReader.sstableComparator);
            this.sstableIterator = this.sstables.iterator();
            assert (this.sstableIterator.hasNext());
            SSTableReader currentSSTable = this.sstableIterator.next();
            this.currentScanner = currentSSTable.getScanner(ranges, null);
        }

        public static Collection<SSTableReader> intersecting(Collection<SSTableReader> sstables, Collection<Range<Token>> ranges) {
            if (ranges == null) {
                return Lists.newArrayList(sstables);
            }
            HashSet<SSTableReader> filtered = new HashSet<SSTableReader>();
            for (Range<Token> range : ranges) {
                for (SSTableReader sstable : sstables) {
                    Range<Token> sstableRange = new Range<Token>(sstable.first.getToken(), sstable.last.getToken());
                    if (range != null && !sstableRange.intersects(range)) continue;
                    filtered.add(sstable);
                }
            }
            return filtered;
        }

        @Override
        public boolean isForThrift() {
            return false;
        }

        @Override
        public CFMetaData metadata() {
            return this.sstables.get((int)0).metadata;
        }

        protected UnfilteredRowIterator computeNext() {
            if (this.currentScanner == null) {
                return (UnfilteredRowIterator)this.endOfData();
            }
            while (!this.currentScanner.hasNext()) {
                this.positionOffset += this.currentScanner.getLengthInBytes();
                this.totalBytesScanned += this.currentScanner.getBytesScanned();
                this.currentScanner.close();
                if (!this.sstableIterator.hasNext()) {
                    this.currentScanner = null;
                    return (UnfilteredRowIterator)this.endOfData();
                }
                SSTableReader currentSSTable = this.sstableIterator.next();
                this.currentScanner = currentSSTable.getScanner(this.ranges, null);
            }
            return (UnfilteredRowIterator)this.currentScanner.next();
        }

        @Override
        public void close() {
            if (this.currentScanner != null) {
                this.currentScanner.close();
            }
        }

        @Override
        public long getLengthInBytes() {
            return this.totalLength;
        }

        @Override
        public long getCurrentPosition() {
            return this.positionOffset + (this.currentScanner == null ? 0L : this.currentScanner.getCurrentPosition());
        }

        @Override
        public long getCompressedLengthInBytes() {
            return this.compressedLength;
        }

        @Override
        public long getBytesScanned() {
            return this.currentScanner == null ? this.totalBytesScanned : this.totalBytesScanned + this.currentScanner.getBytesScanned();
        }

        @Override
        public String getBackingFiles() {
            return Joiner.on((String)", ").join(this.sstables);
        }
    }
}

