/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.index.internal;

import com.google.common.collect.ImmutableSet;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.function.BiFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ColumnDefinition;
import org.apache.cassandra.cql3.ColumnIdentifier;
import org.apache.cassandra.cql3.Operator;
import org.apache.cassandra.cql3.statements.IndexTarget;
import org.apache.cassandra.cql3.statements.RequestValidations;
import org.apache.cassandra.db.CBuilder;
import org.apache.cassandra.db.Clustering;
import org.apache.cassandra.db.ClusteringComparator;
import org.apache.cassandra.db.ClusteringPrefix;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.CompactTables;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.DeletionTime;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.LivenessInfo;
import org.apache.cassandra.db.PartitionColumns;
import org.apache.cassandra.db.RangeTombstone;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.filter.RowFilter;
import org.apache.cassandra.db.lifecycle.SSTableSet;
import org.apache.cassandra.db.lifecycle.View;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.CollectionType;
import org.apache.cassandra.db.marshal.EmptyType;
import org.apache.cassandra.db.partitions.PartitionIterator;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.db.rows.BTreeRow;
import org.apache.cassandra.db.rows.Cell;
import org.apache.cassandra.db.rows.CellPath;
import org.apache.cassandra.db.rows.ComplexColumnData;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.dht.LocalPartitioner;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.index.Index;
import org.apache.cassandra.index.IndexRegistry;
import org.apache.cassandra.index.SecondaryIndexBuilder;
import org.apache.cassandra.index.internal.CassandraIndexFunctions;
import org.apache.cassandra.index.internal.IndexEntry;
import org.apache.cassandra.index.internal.composites.CompositesSearcher;
import org.apache.cassandra.index.internal.keys.KeysSearcher;
import org.apache.cassandra.index.transactions.IndexTransaction;
import org.apache.cassandra.index.transactions.UpdateTransaction;
import org.apache.cassandra.io.sstable.ReducingKeyIterator;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.schema.IndexMetadata;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.concurrent.OpOrder;
import org.apache.cassandra.utils.concurrent.Refs;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class CassandraIndex
implements Index {
    private static final Logger logger = LoggerFactory.getLogger(CassandraIndex.class);
    public static final Pattern TARGET_REGEX = Pattern.compile("^(keys|entries|values|full)\\((.+)\\)$");
    public final ColumnFamilyStore baseCfs;
    protected IndexMetadata metadata;
    protected ColumnFamilyStore indexCfs;
    protected ColumnDefinition indexedColumn;
    protected CassandraIndexFunctions functions;

    protected CassandraIndex(ColumnFamilyStore baseCfs, IndexMetadata indexDef) {
        this.baseCfs = baseCfs;
        this.setMetadata(indexDef);
    }

    protected boolean supportsOperator(ColumnDefinition indexedColumn, Operator operator) {
        return operator == Operator.EQ;
    }

    protected abstract CBuilder buildIndexClusteringPrefix(ByteBuffer var1, ClusteringPrefix var2, CellPath var3);

    public abstract IndexEntry decodeEntry(DecoratedKey var1, Row var2);

    public abstract boolean isStale(Row var1, ByteBuffer var2, int var3);

    protected abstract ByteBuffer getIndexedValue(ByteBuffer var1, Clustering var2, CellPath var3, ByteBuffer var4);

    public ColumnDefinition getIndexedColumn() {
        return this.indexedColumn;
    }

    public ClusteringComparator getIndexComparator() {
        return this.indexCfs.metadata.comparator;
    }

    public ColumnFamilyStore getIndexCfs() {
        return this.indexCfs;
    }

    @Override
    public void register(IndexRegistry registry) {
        registry.registerIndex(this);
    }

    @Override
    public Callable<?> getInitializationTask() {
        return this.isBuilt() || this.baseCfs.isEmpty() ? null : this.getBuildIndexTask();
    }

    @Override
    public IndexMetadata getIndexMetadata() {
        return this.metadata;
    }

    @Override
    public Optional<ColumnFamilyStore> getBackingTable() {
        return this.indexCfs == null ? Optional.empty() : Optional.of(this.indexCfs);
    }

    public Callable<Void> getBlockingFlushTask() {
        return () -> {
            this.indexCfs.forceBlockingFlush();
            return null;
        };
    }

    @Override
    public Callable<?> getInvalidateTask() {
        return () -> {
            this.invalidate();
            return null;
        };
    }

    @Override
    public Callable<?> getMetadataReloadTask(IndexMetadata indexDef) {
        return () -> {
            this.indexCfs.metadata.reloadIndexMetadataProperties(this.baseCfs.metadata);
            this.indexCfs.reload();
            return null;
        };
    }

    @Override
    public void validate(ReadCommand command) throws InvalidRequestException {
        Optional<RowFilter.Expression> target = this.getTargetExpression(command.rowFilter().getExpressions());
        if (target.isPresent()) {
            ByteBuffer indexValue = target.get().getIndexValue();
            RequestValidations.checkFalse(indexValue.remaining() > 65535, "Index expression values may not be larger than 64K");
        }
    }

    private void setMetadata(IndexMetadata indexDef) {
        this.metadata = indexDef;
        Pair<ColumnDefinition, IndexTarget.Type> target = CassandraIndex.parseTarget(this.baseCfs.metadata, indexDef);
        this.functions = CassandraIndex.getFunctions(indexDef, target);
        CFMetaData cfm = CassandraIndex.indexCfsMetadata(this.baseCfs.metadata, indexDef);
        this.indexCfs = ColumnFamilyStore.createColumnFamilyStore(this.baseCfs.keyspace, cfm.cfName, cfm, this.baseCfs.getTracker().loadsstables);
        this.indexedColumn = (ColumnDefinition)target.left;
    }

    @Override
    public Callable<?> getTruncateTask(long truncatedAt) {
        return () -> {
            this.indexCfs.discardSSTables(truncatedAt);
            return null;
        };
    }

    @Override
    public boolean shouldBuildBlocking() {
        return true;
    }

    @Override
    public boolean dependsOn(ColumnDefinition column) {
        return this.indexedColumn.name.equals(column.name);
    }

    @Override
    public boolean supportsExpression(ColumnDefinition column, Operator operator) {
        return this.indexedColumn.name.equals(column.name) && this.supportsOperator(this.indexedColumn, operator);
    }

    private boolean supportsExpression(RowFilter.Expression expression) {
        return this.supportsExpression(expression.column(), expression.operator());
    }

    @Override
    public AbstractType<?> customExpressionValueType() {
        return null;
    }

    @Override
    public long getEstimatedResultRows() {
        long totalRows = 0L;
        long totalPartitions = 0L;
        for (SSTableReader sstable : this.indexCfs.getSSTables(SSTableSet.CANONICAL)) {
            if (sstable.descriptor.version.storeRows()) {
                totalPartitions += sstable.getEstimatedPartitionSize().count();
                totalRows += sstable.getTotalRows();
                continue;
            }
            long colCount = sstable.getEstimatedColumnCount().count();
            totalPartitions += colCount;
            totalRows += sstable.getEstimatedColumnCount().mean() * colCount;
        }
        return totalPartitions > 0L ? (long)((int)(totalRows / totalPartitions)) : 0L;
    }

    @Override
    public BiFunction<PartitionIterator, ReadCommand, PartitionIterator> postProcessorFor(ReadCommand command) {
        return (partitionIterator, readCommand) -> partitionIterator;
    }

    @Override
    public RowFilter getPostIndexQueryFilter(RowFilter filter) {
        return this.getTargetExpression(filter.getExpressions()).map(filter::without).orElse(filter);
    }

    private Optional<RowFilter.Expression> getTargetExpression(List<RowFilter.Expression> expressions) {
        return expressions.stream().filter(this::supportsExpression).findFirst();
    }

    @Override
    public Index.Searcher searcherFor(ReadCommand command) {
        Optional<RowFilter.Expression> target = this.getTargetExpression(command.rowFilter().getExpressions());
        if (target.isPresent()) {
            target.get().validateForIndexing();
            switch (this.getIndexMetadata().kind) {
                case COMPOSITES: {
                    return new CompositesSearcher(command, target.get(), this);
                }
                case KEYS: {
                    return new KeysSearcher(command, target.get(), this);
                }
            }
            throw new IllegalStateException(String.format("Unsupported index type %s for index %s on %s", new Object[]{this.metadata.kind, this.metadata.name, this.indexedColumn.name.toString()}));
        }
        return null;
    }

    @Override
    public void validate(PartitionUpdate update) throws InvalidRequestException {
        switch (this.indexedColumn.kind) {
            case PARTITION_KEY: {
                this.validatePartitionKey(update.partitionKey());
                break;
            }
            case CLUSTERING: {
                this.validateClusterings(update);
                break;
            }
            case REGULAR: {
                if (!update.columns().regulars.contains(this.indexedColumn)) break;
                this.validateRows(update);
                break;
            }
            case STATIC: {
                if (!update.columns().statics.contains(this.indexedColumn)) break;
                this.validateRows(Collections.singleton(update.staticRow()));
            }
        }
    }

    @Override
    public Index.Indexer indexerFor(final DecoratedKey key, PartitionColumns columns, final int nowInSec, final OpOrder.Group opGroup, IndexTransaction.Type transactionType) {
        if (!this.isPrimaryKeyIndex() && !columns.contains(this.indexedColumn)) {
            return null;
        }
        return new Index.Indexer(){

            @Override
            public void begin() {
            }

            @Override
            public void partitionDelete(DeletionTime deletionTime) {
            }

            @Override
            public void rangeTombstone(RangeTombstone tombstone) {
            }

            @Override
            public void insertRow(Row row) {
                if (row.isStatic() && !CassandraIndex.this.indexedColumn.isStatic() && !CassandraIndex.this.indexedColumn.isPartitionKey()) {
                    return;
                }
                if (CassandraIndex.this.isPrimaryKeyIndex()) {
                    this.indexPrimaryKey(row.clustering(), this.getPrimaryKeyIndexLiveness(row), row.deletion());
                } else if (CassandraIndex.this.indexedColumn.isComplex()) {
                    this.indexCells(row.clustering(), row.getComplexColumnData(CassandraIndex.this.indexedColumn));
                } else {
                    this.indexCell(row.clustering(), row.getCell(CassandraIndex.this.indexedColumn));
                }
            }

            @Override
            public void removeRow(Row row) {
                if (CassandraIndex.this.isPrimaryKeyIndex()) {
                    return;
                }
                if (CassandraIndex.this.indexedColumn.isComplex()) {
                    this.removeCells(row.clustering(), row.getComplexColumnData(CassandraIndex.this.indexedColumn));
                } else {
                    this.removeCell(row.clustering(), row.getCell(CassandraIndex.this.indexedColumn));
                }
            }

            @Override
            public void updateRow(Row oldRow, Row newRow) {
                assert (oldRow.isStatic() == newRow.isStatic());
                if (newRow.isStatic() != CassandraIndex.this.indexedColumn.isStatic()) {
                    return;
                }
                if (CassandraIndex.this.isPrimaryKeyIndex()) {
                    this.indexPrimaryKey(newRow.clustering(), newRow.primaryKeyLivenessInfo(), newRow.deletion());
                }
                if (CassandraIndex.this.indexedColumn.isComplex()) {
                    this.indexCells(newRow.clustering(), newRow.getComplexColumnData(CassandraIndex.this.indexedColumn));
                    this.removeCells(oldRow.clustering(), oldRow.getComplexColumnData(CassandraIndex.this.indexedColumn));
                } else {
                    this.indexCell(newRow.clustering(), newRow.getCell(CassandraIndex.this.indexedColumn));
                    this.removeCell(oldRow.clustering(), oldRow.getCell(CassandraIndex.this.indexedColumn));
                }
            }

            @Override
            public void finish() {
            }

            private void indexCells(Clustering clustering, Iterable<Cell> cells) {
                if (cells == null) {
                    return;
                }
                for (Cell cell : cells) {
                    this.indexCell(clustering, cell);
                }
            }

            private void indexCell(Clustering clustering, Cell cell) {
                if (cell == null || !cell.isLive(nowInSec)) {
                    return;
                }
                CassandraIndex.this.insert(key.getKey(), clustering, cell, LivenessInfo.create(cell.timestamp(), cell.ttl(), cell.localDeletionTime()), opGroup);
            }

            private void removeCells(Clustering clustering, Iterable<Cell> cells) {
                if (cells == null) {
                    return;
                }
                for (Cell cell : cells) {
                    this.removeCell(clustering, cell);
                }
            }

            private void removeCell(Clustering clustering, Cell cell) {
                if (cell == null || !cell.isLive(nowInSec)) {
                    return;
                }
                CassandraIndex.this.delete(key.getKey(), clustering, cell, opGroup, nowInSec);
            }

            private void indexPrimaryKey(Clustering clustering, LivenessInfo liveness, Row.Deletion deletion) {
                if (liveness.timestamp() != Long.MIN_VALUE) {
                    CassandraIndex.this.insert(key.getKey(), clustering, null, liveness, opGroup);
                }
                if (!deletion.isLive()) {
                    CassandraIndex.this.delete(key.getKey(), clustering, deletion.time(), opGroup);
                }
            }

            private LivenessInfo getPrimaryKeyIndexLiveness(Row row) {
                long timestamp = row.primaryKeyLivenessInfo().timestamp();
                int ttl = row.primaryKeyLivenessInfo().ttl();
                for (Cell cell : row.cells()) {
                    long cellTimestamp = cell.timestamp();
                    if (!cell.isLive(nowInSec) || cellTimestamp <= timestamp) continue;
                    timestamp = cellTimestamp;
                    ttl = cell.ttl();
                }
                return LivenessInfo.create(CassandraIndex.this.baseCfs.metadata, timestamp, ttl, nowInSec);
            }
        };
    }

    public void deleteStaleEntry(DecoratedKey indexKey, Clustering indexClustering, DeletionTime deletion, OpOrder.Group opGroup) {
        this.doDelete(indexKey, indexClustering, deletion, opGroup);
        logger.trace("Removed index entry for stale value {}", (Object)indexKey);
    }

    private void insert(ByteBuffer rowKey, Clustering clustering, Cell cell, LivenessInfo info, OpOrder.Group opGroup) {
        DecoratedKey valueKey = this.getIndexKeyFor(this.getIndexedValue(rowKey, clustering, cell));
        BTreeRow row = BTreeRow.noCellLiveRow(this.buildIndexClustering(rowKey, clustering, cell), info);
        PartitionUpdate upd = this.partitionUpdate(valueKey, row);
        this.indexCfs.apply(upd, UpdateTransaction.NO_OP, opGroup, null);
        logger.trace("Inserted entry into index for value {}", (Object)valueKey);
    }

    private void delete(ByteBuffer rowKey, Clustering clustering, Cell cell, OpOrder.Group opGroup, int nowInSec) {
        DecoratedKey valueKey = this.getIndexKeyFor(this.getIndexedValue(rowKey, clustering, cell));
        this.doDelete(valueKey, this.buildIndexClustering(rowKey, clustering, cell), new DeletionTime(cell.timestamp(), nowInSec), opGroup);
    }

    private void delete(ByteBuffer rowKey, Clustering clustering, DeletionTime deletion, OpOrder.Group opGroup) {
        DecoratedKey valueKey = this.getIndexKeyFor(this.getIndexedValue(rowKey, clustering, null));
        this.doDelete(valueKey, this.buildIndexClustering(rowKey, clustering, null), deletion, opGroup);
    }

    private void doDelete(DecoratedKey indexKey, Clustering indexClustering, DeletionTime deletion, OpOrder.Group opGroup) {
        BTreeRow row = BTreeRow.emptyDeletedRow(indexClustering, Row.Deletion.regular(deletion));
        PartitionUpdate upd = this.partitionUpdate(indexKey, row);
        this.indexCfs.apply(upd, UpdateTransaction.NO_OP, opGroup, null);
        logger.trace("Removed index entry for value {}", (Object)indexKey);
    }

    private void validatePartitionKey(DecoratedKey partitionKey) throws InvalidRequestException {
        assert (this.indexedColumn.isPartitionKey());
        this.validateIndexedValue(this.getIndexedValue(partitionKey.getKey(), null, null));
    }

    private void validateClusterings(PartitionUpdate update) throws InvalidRequestException {
        assert (this.indexedColumn.isClusteringColumn());
        for (Row row : update) {
            this.validateIndexedValue(this.getIndexedValue(null, row.clustering(), null));
        }
    }

    private void validateRows(Iterable<Row> rows) {
        assert (!this.indexedColumn.isPrimaryKeyColumn());
        for (Row row : rows) {
            if (this.indexedColumn.isComplex()) {
                ComplexColumnData data = row.getComplexColumnData(this.indexedColumn);
                if (data == null) continue;
                for (Cell cell : data) {
                    this.validateIndexedValue(this.getIndexedValue(null, null, cell.path(), cell.value()));
                }
                continue;
            }
            this.validateIndexedValue(this.getIndexedValue(null, null, row.getCell(this.indexedColumn)));
        }
    }

    private void validateIndexedValue(ByteBuffer value) {
        if (value != null && value.remaining() >= 65535) {
            throw new InvalidRequestException(String.format("Cannot index value of size %d for index %s on %s.%s(%s) (maximum allowed size=%d)", value.remaining(), this.metadata.name, this.baseCfs.metadata.ksName, this.baseCfs.metadata.cfName, this.indexedColumn.name.toString(), 65535));
        }
    }

    private ByteBuffer getIndexedValue(ByteBuffer rowKey, Clustering clustering, Cell cell) {
        return this.getIndexedValue(rowKey, clustering, cell == null ? null : cell.path(), cell == null ? null : cell.value());
    }

    private Clustering buildIndexClustering(ByteBuffer rowKey, Clustering clustering, Cell cell) {
        return this.buildIndexClusteringPrefix(rowKey, clustering, cell == null ? null : cell.path()).build();
    }

    private DecoratedKey getIndexKeyFor(ByteBuffer value) {
        return this.indexCfs.decorateKey(value);
    }

    private PartitionUpdate partitionUpdate(DecoratedKey valueKey, Row row) {
        return PartitionUpdate.singleRowUpdate(this.indexCfs.metadata, valueKey, row);
    }

    private void invalidate() {
        Set<ColumnFamilyStore> cfss = Collections.singleton(this.indexCfs);
        CompactionManager.instance.interruptCompactionForCFs(cfss, true);
        CompactionManager.instance.waitForCessation(cfss);
        Keyspace.writeOrder.awaitNewBarrier();
        this.indexCfs.forceBlockingFlush();
        this.indexCfs.readOrdering.awaitNewBarrier();
        this.indexCfs.invalidate();
    }

    private boolean isBuilt() {
        return SystemKeyspace.isIndexBuilt(this.baseCfs.keyspace.getName(), this.metadata.name);
    }

    private boolean isPrimaryKeyIndex() {
        return this.indexedColumn.isPrimaryKeyColumn();
    }

    private Callable<?> getBuildIndexTask() {
        return () -> {
            this.buildBlocking();
            return null;
        };
    }

    private void buildBlocking() {
        this.baseCfs.forceBlockingFlush();
        try (ColumnFamilyStore.RefViewFragment viewFragment = this.baseCfs.selectAndReference(View.selectFunction(SSTableSet.CANONICAL));
             Refs<SSTableReader> sstables = viewFragment.refs;){
            if (sstables.isEmpty()) {
                logger.info("No SSTable data for {}.{} to build index {} from, marking empty index as built", new Object[]{this.baseCfs.metadata.ksName, this.baseCfs.metadata.cfName, this.metadata.name});
                this.baseCfs.indexManager.markIndexBuilt(this.metadata.name);
                return;
            }
            logger.info("Submitting index build of {} for data in {}", (Object)this.metadata.name, (Object)CassandraIndex.getSSTableNames(sstables));
            SecondaryIndexBuilder builder = new SecondaryIndexBuilder(this.baseCfs, Collections.singleton(this), new ReducingKeyIterator(sstables));
            Future<?> future = CompactionManager.instance.submitIndexBuild(builder);
            FBUtilities.waitOnFuture(future);
            this.indexCfs.forceBlockingFlush();
            this.baseCfs.indexManager.markIndexBuilt(this.metadata.name);
        }
        logger.info("Index build of {} complete", (Object)this.metadata.name);
    }

    private static String getSSTableNames(Collection<SSTableReader> sstables) {
        return StreamSupport.stream(sstables.spliterator(), false).map(SSTable::toString).collect(Collectors.joining(", "));
    }

    public static final CFMetaData indexCfsMetadata(CFMetaData baseCfsMetadata, IndexMetadata indexMetadata) {
        Pair<ColumnDefinition, IndexTarget.Type> target = CassandraIndex.parseTarget(baseCfsMetadata, indexMetadata);
        CassandraIndexFunctions utils = CassandraIndex.getFunctions(indexMetadata, target);
        ColumnDefinition indexedColumn = (ColumnDefinition)target.left;
        AbstractType<?> indexedValueType = utils.getIndexedValueType(indexedColumn);
        CFMetaData.Builder builder = indexMetadata.isKeys() ? CFMetaData.Builder.create(baseCfsMetadata.ksName, baseCfsMetadata.indexColumnFamilyName(indexMetadata), true, false, false) : CFMetaData.Builder.create(baseCfsMetadata.ksName, baseCfsMetadata.indexColumnFamilyName(indexMetadata));
        builder = builder.withId(baseCfsMetadata.cfId).withPartitioner(new LocalPartitioner(indexedValueType)).addPartitionKey(indexedColumn.name, indexedColumn.type).addClusteringColumn("partition_key", baseCfsMetadata.partitioner.partitionOrdering());
        if (indexMetadata.isKeys()) {
            CompactTables.DefaultNames names = CompactTables.defaultNameGenerator((Set<String>)ImmutableSet.of((Object)indexedColumn.name.toString(), (Object)"partition_key"));
            builder = builder.addRegularColumn(names.defaultCompactValueName(), (AbstractType)EmptyType.instance);
        } else {
            builder = utils.addIndexClusteringColumns(builder, baseCfsMetadata, indexedColumn);
        }
        return builder.build().reloadIndexMetadataProperties(baseCfsMetadata);
    }

    public static CassandraIndex newIndex(ColumnFamilyStore baseCfs, IndexMetadata indexMetadata) {
        return CassandraIndex.getFunctions(indexMetadata, CassandraIndex.parseTarget(baseCfs.metadata, indexMetadata)).newIndexInstance(baseCfs, indexMetadata);
    }

    public static Pair<ColumnDefinition, IndexTarget.Type> parseTarget(CFMetaData cfm, IndexMetadata indexDef) {
        String target = indexDef.options.get("target");
        assert (target != null) : String.format("No target definition found for index %s", indexDef.name);
        Pair<ColumnDefinition, IndexTarget.Type> result = CassandraIndex.parseTarget(cfm, target);
        if (result == null) {
            throw new ConfigurationException(String.format("Unable to parse targets for index %s (%s)", indexDef.name, target));
        }
        return result;
    }

    public static Pair<ColumnDefinition, IndexTarget.Type> parseTarget(CFMetaData cfm, String target) {
        ColumnDefinition cd;
        String columnName;
        IndexTarget.Type targetType;
        Matcher matcher = TARGET_REGEX.matcher(target);
        if (matcher.matches()) {
            targetType = IndexTarget.Type.fromString(matcher.group(1));
            columnName = matcher.group(2);
        } else {
            columnName = target;
            targetType = IndexTarget.Type.VALUES;
        }
        if (columnName.startsWith("\"")) {
            columnName = StringUtils.substring((String)StringUtils.substring((String)columnName, (int)1), (int)0, (int)-1);
            columnName = columnName.replaceAll("\"\"", "\"");
        }
        if ((cd = cfm.getColumnDefinition(new ColumnIdentifier(columnName, true))) != null) {
            return Pair.create(cd, targetType);
        }
        for (ColumnDefinition column : cfm.allColumns()) {
            if (!column.name.toString().equals(columnName)) continue;
            return Pair.create(column, targetType);
        }
        return null;
    }

    static CassandraIndexFunctions getFunctions(IndexMetadata indexDef, Pair<ColumnDefinition, IndexTarget.Type> target) {
        if (indexDef.isKeys()) {
            return CassandraIndexFunctions.KEYS_INDEX_FUNCTIONS;
        }
        ColumnDefinition indexedColumn = (ColumnDefinition)target.left;
        if (indexedColumn.type.isCollection() && indexedColumn.type.isMultiCell()) {
            switch (((CollectionType)indexedColumn.type).kind) {
                case LIST: {
                    return CassandraIndexFunctions.COLLECTION_VALUE_INDEX_FUNCTIONS;
                }
                case SET: {
                    return CassandraIndexFunctions.COLLECTION_KEY_INDEX_FUNCTIONS;
                }
                case MAP: {
                    switch ((IndexTarget.Type)((Object)target.right)) {
                        case KEYS: {
                            return CassandraIndexFunctions.COLLECTION_KEY_INDEX_FUNCTIONS;
                        }
                        case KEYS_AND_VALUES: {
                            return CassandraIndexFunctions.COLLECTION_ENTRY_INDEX_FUNCTIONS;
                        }
                        case VALUES: {
                            return CassandraIndexFunctions.COLLECTION_VALUE_INDEX_FUNCTIONS;
                        }
                    }
                    throw new AssertionError();
                }
            }
        }
        switch (indexedColumn.kind) {
            case CLUSTERING: {
                return CassandraIndexFunctions.CLUSTERING_COLUMN_INDEX_FUNCTIONS;
            }
            case REGULAR: {
                return CassandraIndexFunctions.REGULAR_COLUMN_INDEX_FUNCTIONS;
            }
            case PARTITION_KEY: {
                return CassandraIndexFunctions.PARTITION_KEY_INDEX_FUNCTIONS;
            }
        }
        throw new AssertionError();
    }
}

