/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.batchlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.util.concurrent.RateLimiter;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.batchlog.Batch;
import org.apache.cassandra.batchlog.BatchlogManagerMBean;
import org.apache.cassandra.concurrent.DebuggableScheduledThreadPoolExecutor;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.RowUpdateBuilder;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.WriteType;
import org.apache.cassandra.db.marshal.BytesType;
import org.apache.cassandra.db.marshal.UUIDType;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.WriteFailureException;
import org.apache.cassandra.exceptions.WriteTimeoutException;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.hints.Hint;
import org.apache.cassandra.hints.HintsService;
import org.apache.cassandra.io.util.DataInputBuffer;
import org.apache.cassandra.io.util.DataOutputBuffer;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.net.MessageIn;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.service.WriteResponseHandler;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.UUIDGen;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BatchlogManager
implements BatchlogManagerMBean {
    public static final String MBEAN_NAME = "org.apache.cassandra.db:type=BatchlogManager";
    private static final long REPLAY_INTERVAL = 10000L;
    static final int DEFAULT_PAGE_SIZE = 128;
    private static final Logger logger = LoggerFactory.getLogger(BatchlogManager.class);
    public static final BatchlogManager instance = new BatchlogManager();
    private volatile long totalBatchesReplayed = 0L;
    private volatile UUID lastReplayedUuid = UUIDGen.minTimeUUID(0L);
    private final ScheduledExecutorService batchlogTasks = new DebuggableScheduledThreadPoolExecutor("BatchlogTasks");

    public void start() {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(this, new ObjectName(MBEAN_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        this.batchlogTasks.scheduleWithFixedDelay(this::replayFailedBatches, StorageService.RING_DELAY, 10000L, TimeUnit.MILLISECONDS);
    }

    public void shutdown() throws InterruptedException {
        this.batchlogTasks.shutdown();
        this.batchlogTasks.awaitTermination(60L, TimeUnit.SECONDS);
    }

    public static void remove(UUID id) {
        new Mutation(PartitionUpdate.fullPartitionDelete(SystemKeyspace.Batches, UUIDType.instance.decompose(id), FBUtilities.timestampMicros(), FBUtilities.nowInSeconds())).apply();
    }

    public static void store(Batch batch) {
        BatchlogManager.store(batch, true);
    }

    public static void store(Batch batch, boolean durableWrites) {
        RowUpdateBuilder builder = new RowUpdateBuilder(SystemKeyspace.Batches, batch.creationTime, batch.id).clustering(new Object[0]).add("version", (Object)10);
        for (ByteBuffer byteBuffer : batch.encodedMutations) {
            builder.addListEntry("mutations", byteBuffer);
        }
        for (Mutation mutation : batch.decodedMutations) {
            try {
                DataOutputBuffer buffer = new DataOutputBuffer();
                Throwable throwable = null;
                try {
                    Mutation.serializer.serialize(mutation, (DataOutputPlus)buffer, 10);
                    builder.addListEntry("mutations", buffer.buffer());
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (buffer == null) continue;
                    if (throwable != null) {
                        try {
                            buffer.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    buffer.close();
                }
            }
            catch (IOException e) {
                throw new AssertionError((Object)e);
            }
        }
        builder.build().apply(durableWrites);
    }

    @Override
    @VisibleForTesting
    public int countAllBatches() {
        String query = String.format("SELECT count(*) FROM %s.%s", "system", "batches");
        UntypedResultSet results = QueryProcessor.executeInternal(query, new Object[0]);
        if (results == null || results.isEmpty()) {
            return 0;
        }
        return (int)results.one().getLong("count");
    }

    @Override
    public long getTotalBatchesReplayed() {
        return this.totalBatchesReplayed;
    }

    @Override
    public void forceBatchlogReplay() throws Exception {
        this.startBatchlogReplay().get();
    }

    public Future<?> startBatchlogReplay() {
        return this.batchlogTasks.submit(this::replayFailedBatches);
    }

    void performInitialReplay() throws InterruptedException, ExecutionException {
        this.batchlogTasks.submit(this::replayFailedBatches).get();
    }

    private void replayFailedBatches() {
        logger.trace("Started replayFailedBatches");
        int throttleInKB = DatabaseDescriptor.getBatchlogReplayThrottleInKB() / StorageService.instance.getTokenMetadata().getAllEndpoints().size();
        RateLimiter rateLimiter = RateLimiter.create((double)(throttleInKB == 0 ? Double.MAX_VALUE : (double)(throttleInKB * 1024)));
        UUID limitUuid = UUIDGen.maxTimeUUID(System.currentTimeMillis() - BatchlogManager.getBatchlogTimeout());
        ColumnFamilyStore store = Keyspace.open("system").getColumnFamilyStore("batches");
        int pageSize = BatchlogManager.calculatePageSize(store);
        String query = String.format("SELECT id, mutations, version FROM %s.%s WHERE token(id) > token(?) AND token(id) <= token(?)", "system", "batches");
        UntypedResultSet batches = QueryProcessor.executeInternalWithPaging(query, pageSize, this.lastReplayedUuid, limitUuid);
        this.processBatchlogEntries(batches, pageSize, rateLimiter);
        this.lastReplayedUuid = limitUuid;
        logger.trace("Finished replayFailedBatches");
    }

    static int calculatePageSize(ColumnFamilyStore store) {
        double averageRowSize = store.getMeanPartitionSize();
        if (averageRowSize <= 0.0) {
            return 128;
        }
        return (int)Math.max(1.0, Math.min(128.0, 4194304.0 / averageRowSize));
    }

    private void processBatchlogEntries(UntypedResultSet batches, int pageSize, RateLimiter rateLimiter) {
        int positionInPage = 0;
        ArrayList<ReplayingBatch> unfinishedBatches = new ArrayList<ReplayingBatch>(pageSize);
        HashSet<InetAddress> hintedNodes = new HashSet<InetAddress>();
        HashSet<UUID> replayedBatches = new HashSet<UUID>();
        for (UntypedResultSet.Row row : batches) {
            UUID id = row.getUUID("id");
            int version = row.getInt("version");
            try {
                ReplayingBatch batch = new ReplayingBatch(id, version, row.getList("mutations", BytesType.instance));
                if (batch.replay(rateLimiter, hintedNodes) > 0) {
                    unfinishedBatches.add(batch);
                } else {
                    BatchlogManager.remove(id);
                    ++this.totalBatchesReplayed;
                }
            }
            catch (IOException e) {
                logger.warn("Skipped batch replay of {} due to {}", (Object)id, (Object)e);
                BatchlogManager.remove(id);
            }
            if (++positionInPage != pageSize) continue;
            this.finishAndClearBatches(unfinishedBatches, hintedNodes, replayedBatches);
            positionInPage = 0;
        }
        this.finishAndClearBatches(unfinishedBatches, hintedNodes, replayedBatches);
        HintsService.instance.flushAndFsyncBlockingly(Iterables.transform(hintedNodes, StorageService.instance::getHostIdForEndpoint));
        replayedBatches.forEach(BatchlogManager::remove);
    }

    private void finishAndClearBatches(ArrayList<ReplayingBatch> batches, Set<InetAddress> hintedNodes, Set<UUID> replayedBatches) {
        for (ReplayingBatch batch : batches) {
            batch.finish(hintedNodes);
            replayedBatches.add(batch.id);
        }
        this.totalBatchesReplayed += (long)batches.size();
        batches.clear();
    }

    public static long getBatchlogTimeout() {
        return DatabaseDescriptor.getWriteRpcTimeout() * 2L;
    }

    public static class EndpointFilter {
        private final String localRack;
        private final Multimap<String, InetAddress> endpoints;

        public EndpointFilter(String localRack, Multimap<String, InetAddress> endpoints) {
            this.localRack = localRack;
            this.endpoints = endpoints;
        }

        public Collection<InetAddress> filter() {
            Collection racks;
            if (this.endpoints.values().size() == 1) {
                return this.endpoints.values();
            }
            ArrayListMultimap validated = ArrayListMultimap.create();
            for (Map.Entry entry : this.endpoints.entries()) {
                if (!this.isValid((InetAddress)entry.getValue())) continue;
                validated.put(entry.getKey(), entry.getValue());
            }
            if (validated.size() <= 2) {
                return validated.values();
            }
            if (validated.size() - validated.get((Object)this.localRack).size() >= 2) {
                validated.removeAll((Object)this.localRack);
            }
            if (validated.keySet().size() == 1) {
                Collection otherRack = (Collection)Iterables.getOnlyElement(validated.asMap().values());
                return Lists.newArrayList((Iterable)Iterables.limit((Iterable)otherRack, (int)2));
            }
            if (validated.keySet().size() == 2) {
                racks = validated.keySet();
            } else {
                racks = Lists.newArrayList((Iterable)validated.keySet());
                Collections.shuffle((List)racks);
            }
            ArrayList<InetAddress> result = new ArrayList<InetAddress>(2);
            for (String rack : Iterables.limit((Iterable)racks, (int)2)) {
                List rackMembers = validated.get((Object)rack);
                result.add((InetAddress)rackMembers.get(this.getRandomInt(rackMembers.size())));
            }
            return result;
        }

        @VisibleForTesting
        protected boolean isValid(InetAddress input) {
            return !input.equals(FBUtilities.getBroadcastAddress()) && FailureDetector.instance.isAlive(input);
        }

        @VisibleForTesting
        protected int getRandomInt(int bound) {
            return ThreadLocalRandom.current().nextInt(bound);
        }
    }

    private static class ReplayingBatch {
        private final UUID id;
        private final long writtenAt;
        private final List<Mutation> mutations;
        private final int replayedBytes;
        private List<ReplayWriteResponseHandler<Mutation>> replayHandlers;

        ReplayingBatch(UUID id, int version, List<ByteBuffer> serializedMutations) throws IOException {
            this.id = id;
            this.writtenAt = UUIDGen.unixTimestamp(id);
            this.mutations = new ArrayList<Mutation>(serializedMutations.size());
            this.replayedBytes = this.addMutations(version, serializedMutations);
        }

        public int replay(RateLimiter rateLimiter, Set<InetAddress> hintedNodes) throws IOException {
            logger.trace("Replaying batch {}", (Object)this.id);
            if (this.mutations.isEmpty()) {
                return 0;
            }
            int gcgs = ReplayingBatch.gcgs(this.mutations);
            if (TimeUnit.MILLISECONDS.toSeconds(this.writtenAt) + (long)gcgs <= (long)FBUtilities.nowInSeconds()) {
                return 0;
            }
            this.replayHandlers = ReplayingBatch.sendReplays(this.mutations, this.writtenAt, hintedNodes);
            rateLimiter.acquire(this.replayedBytes);
            return this.replayHandlers.size();
        }

        public void finish(Set<InetAddress> hintedNodes) {
            for (int i = 0; i < this.replayHandlers.size(); ++i) {
                ReplayWriteResponseHandler<Mutation> handler = this.replayHandlers.get(i);
                try {
                    handler.get();
                    continue;
                }
                catch (WriteFailureException | WriteTimeoutException e) {
                    logger.trace("Failed replaying a batched mutation to a node, will write a hint");
                    logger.trace("Failure was : {}", (Object)e.getMessage());
                    this.writeHintsForUndeliveredEndpoints(i, hintedNodes);
                    return;
                }
            }
        }

        private int addMutations(int version, List<ByteBuffer> serializedMutations) throws IOException {
            int ret = 0;
            for (ByteBuffer serializedMutation : serializedMutations) {
                ret += serializedMutation.remaining();
                DataInputBuffer in = new DataInputBuffer(serializedMutation, true);
                Throwable throwable = null;
                try {
                    this.addMutation(Mutation.serializer.deserialize(in, version));
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (in == null) continue;
                    if (throwable != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    in.close();
                }
            }
            return ret;
        }

        private void addMutation(Mutation mutation) {
            for (UUID cfId : mutation.getColumnFamilyIds()) {
                if (this.writtenAt > SystemKeyspace.getTruncatedAt(cfId)) continue;
                mutation = mutation.without(cfId);
            }
            if (!mutation.isEmpty()) {
                this.mutations.add(mutation);
            }
        }

        private void writeHintsForUndeliveredEndpoints(int startFrom, Set<InetAddress> hintedNodes) {
            int gcgs = ReplayingBatch.gcgs(this.mutations);
            if (TimeUnit.MILLISECONDS.toSeconds(this.writtenAt) + (long)gcgs <= (long)FBUtilities.nowInSeconds()) {
                return;
            }
            for (int i = startFrom; i < this.replayHandlers.size(); ++i) {
                ReplayWriteResponseHandler<Mutation> handler = this.replayHandlers.get(i);
                Mutation undeliveredMutation = this.mutations.get(i);
                if (handler == null) continue;
                hintedNodes.addAll(((ReplayWriteResponseHandler)handler).undelivered);
                HintsService.instance.write(Iterables.transform((Iterable)((ReplayWriteResponseHandler)handler).undelivered, StorageService.instance::getHostIdForEndpoint), Hint.create(undeliveredMutation, this.writtenAt));
            }
        }

        private static List<ReplayWriteResponseHandler<Mutation>> sendReplays(List<Mutation> mutations, long writtenAt, Set<InetAddress> hintedNodes) {
            ArrayList<ReplayWriteResponseHandler<Mutation>> handlers = new ArrayList<ReplayWriteResponseHandler<Mutation>>(mutations.size());
            for (Mutation mutation : mutations) {
                ReplayWriteResponseHandler<Mutation> handler = ReplayingBatch.sendSingleReplayMutation(mutation, writtenAt, hintedNodes);
                if (handler == null) continue;
                handlers.add(handler);
            }
            return handlers;
        }

        private static ReplayWriteResponseHandler<Mutation> sendSingleReplayMutation(Mutation mutation, long writtenAt, Set<InetAddress> hintedNodes) {
            HashSet<InetAddress> liveEndpoints = new HashSet<InetAddress>();
            String ks = mutation.getKeyspaceName();
            Token tk = mutation.key().getToken();
            for (InetAddress endpoint : Iterables.concat(StorageService.instance.getNaturalEndpoints(ks, tk), StorageService.instance.getTokenMetadata().pendingEndpointsFor(tk, ks))) {
                if (endpoint.equals(FBUtilities.getBroadcastAddress())) {
                    mutation.apply();
                    continue;
                }
                if (FailureDetector.instance.isAlive(endpoint)) {
                    liveEndpoints.add(endpoint);
                    continue;
                }
                hintedNodes.add(endpoint);
                HintsService.instance.write(StorageService.instance.getHostIdForEndpoint(endpoint), Hint.create(mutation, writtenAt));
            }
            if (liveEndpoints.isEmpty()) {
                return null;
            }
            ReplayWriteResponseHandler<Mutation> handler = new ReplayWriteResponseHandler<Mutation>(liveEndpoints);
            MessageOut<Mutation> message = mutation.createMessage();
            for (InetAddress endpoint : liveEndpoints) {
                MessagingService.instance().sendRR(message, endpoint, handler, false);
            }
            return handler;
        }

        private static int gcgs(Collection<Mutation> mutations) {
            int gcgs = Integer.MAX_VALUE;
            for (Mutation mutation : mutations) {
                gcgs = Math.min(gcgs, mutation.smallestGCGS());
            }
            return gcgs;
        }

        private static class ReplayWriteResponseHandler<T>
        extends WriteResponseHandler<T> {
            private final Set<InetAddress> undelivered = Collections.newSetFromMap(new ConcurrentHashMap());

            ReplayWriteResponseHandler(Collection<InetAddress> writeEndpoints) {
                super(writeEndpoints, Collections.emptySet(), null, null, null, WriteType.UNLOGGED_BATCH);
                this.undelivered.addAll(writeEndpoints);
            }

            @Override
            protected int totalBlockFor() {
                return this.naturalEndpoints.size();
            }

            @Override
            public void response(MessageIn<T> m) {
                boolean removed = this.undelivered.remove(m == null ? FBUtilities.getBroadcastAddress() : m.from);
                assert (removed);
                super.response(m);
            }
        }
    }
}

