/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.batchlog;

import java.io.IOException;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.batchlog.Batch;
import org.apache.cassandra.batchlog.BatchlogManager;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.RowUpdateBuilder;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.WriteType;
import org.apache.cassandra.db.marshal.UUIDType;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.exceptions.WriteFailureException;
import org.apache.cassandra.exceptions.WriteTimeoutException;
import org.apache.cassandra.io.util.DataInputBuffer;
import org.apache.cassandra.io.util.DataOutputBuffer;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.WriteResponseHandler;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.UUIDGen;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LegacyBatchlogMigrator {
    private static final Logger logger = LoggerFactory.getLogger(LegacyBatchlogMigrator.class);

    private LegacyBatchlogMigrator() {
    }

    public static void migrate() {
        ColumnFamilyStore store = Keyspace.open("system").getColumnFamilyStore("batchlog");
        if (store.isEmpty()) {
            return;
        }
        logger.info("Migrating legacy batchlog to new storage");
        int convertedBatches = 0;
        String query = String.format("SELECT id, data, written_at, version FROM %s.%s", "system", "batchlog");
        int pageSize = BatchlogManager.calculatePageSize(store);
        UntypedResultSet rows = QueryProcessor.executeInternalWithPaging(query, pageSize, new Object[0]);
        for (UntypedResultSet.Row row : rows) {
            if (!LegacyBatchlogMigrator.apply(row, convertedBatches)) continue;
            ++convertedBatches;
        }
        if (convertedBatches > 0) {
            Keyspace.openAndGetStore(SystemKeyspace.LegacyBatchlog).truncateBlocking();
        }
    }

    public static boolean isLegacyBatchlogMutation(Mutation mutation) {
        return mutation.getKeyspaceName().equals("system") && mutation.getPartitionUpdate(SystemKeyspace.LegacyBatchlog.cfId) != null;
    }

    public static void handleLegacyMutation(Mutation mutation) {
        PartitionUpdate update = mutation.getPartitionUpdate(SystemKeyspace.LegacyBatchlog.cfId);
        logger.debug("Applying legacy batchlog mutation {}", (Object)update);
        update.forEach(row -> LegacyBatchlogMigrator.apply(UntypedResultSet.Row.fromInternalRow(update.metadata(), update.partitionKey(), row), -1L));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static boolean apply(UntypedResultSet.Row row, long counter) {
        int version;
        UUID id = row.getUUID("id");
        long timestamp = id.version() == 1 ? UUIDGen.unixTimestamp(id) : row.getLong("written_at");
        int n = version = row.has("version") ? row.getInt("version") : 6;
        if (id.version() != 1) {
            id = UUIDGen.getTimeUUID(timestamp, counter);
        }
        logger.debug("Converting mutation at {}", (Object)timestamp);
        try (DataInputBuffer in = new DataInputBuffer(row.getBytes("data"), false);){
            int numMutations = in.readInt();
            ArrayList<Mutation> mutations = new ArrayList<Mutation>(numMutations);
            for (int i = 0; i < numMutations; ++i) {
                mutations.add(Mutation.serializer.deserialize(in, version));
            }
            BatchlogManager.store(Batch.createLocal(id, TimeUnit.MILLISECONDS.toMicros(timestamp), mutations));
            boolean bl = true;
            return bl;
        }
        catch (Throwable t) {
            logger.error("Failed to convert mutation {} at timestamp {}", new Object[]{id, timestamp, t});
            return false;
        }
    }

    public static void syncWriteToBatchlog(WriteResponseHandler<?> handler, Batch batch, Collection<InetAddress> endpoints) throws WriteTimeoutException, WriteFailureException {
        for (InetAddress target : endpoints) {
            logger.debug("Sending legacy batchlog store request {} to {} for {} mutations", new Object[]{batch.id, target, batch.size()});
            int targetVersion = MessagingService.instance().getVersion(target);
            MessagingService.instance().sendRR(LegacyBatchlogMigrator.getStoreMutation(batch, targetVersion).createMessage(MessagingService.Verb.MUTATION), target, handler, false);
        }
    }

    public static void asyncRemoveFromBatchlog(Collection<InetAddress> endpoints, UUID uuid) {
        WriteResponseHandler handler = new WriteResponseHandler(endpoints, Collections.emptyList(), ConsistencyLevel.ANY, Keyspace.open("system"), null, WriteType.SIMPLE);
        Mutation mutation = LegacyBatchlogMigrator.getRemoveMutation(uuid);
        for (InetAddress target : endpoints) {
            logger.debug("Sending legacy batchlog remove request {} to {}", (Object)uuid, (Object)target);
            MessagingService.instance().sendRR(mutation.createMessage(MessagingService.Verb.MUTATION), target, handler, false);
        }
    }

    static void store(Batch batch, int version) {
        LegacyBatchlogMigrator.getStoreMutation(batch, version).apply();
    }

    static Mutation getStoreMutation(Batch batch, int version) {
        return new RowUpdateBuilder(SystemKeyspace.LegacyBatchlog, batch.creationTime, batch.id).clustering(new Object[0]).add("written_at", (Object)new Date(batch.creationTime / 1000L)).add("data", (Object)LegacyBatchlogMigrator.getSerializedMutations(version, batch.decodedMutations)).add("version", (Object)version).build();
    }

    private static Mutation getRemoveMutation(UUID uuid) {
        return new Mutation(PartitionUpdate.fullPartitionDelete(SystemKeyspace.LegacyBatchlog, UUIDType.instance.decompose(uuid), FBUtilities.timestampMicros(), FBUtilities.nowInSeconds()));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static ByteBuffer getSerializedMutations(int version, Collection<Mutation> mutations) {
        try (DataOutputBuffer buf = new DataOutputBuffer();){
            buf.writeInt(mutations.size());
            for (Mutation mutation : mutations) {
                Mutation.serializer.serialize(mutation, (DataOutputPlus)buf, version);
            }
            ByteBuffer byteBuffer = buf.buffer();
            return byteBuffer;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}

